;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/picasso-object.cl,v $
;;; $Revision: 1.19 $
;;; $Date: 90/07/31 16:26:52 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;; A picasso-object (abstract class) is any callable entity (dialog, frame, ..)
;;; that can have static and dynamic local variables, value and reference 
;;; parameters, and (in general) reentrancy by virtue of duplicating the 
;;; window.

(defclass picasso-object (variable-holder collection-widget)
  ((parent :initform (root-window))
   (lexical-parent :initform nil)
   (gc-spec :initform '(gc-res "default"))

   ;; Parameters (all five types)

   (ref-params 
    :initarg :ref-params  
    :initform nil 
    :type list 
    :accessor ref-params)
   (value-params 
    :initarg :value-params  
    :initform nil 
    :type list 
    :accessor value-params)
   (value-result-params
    :initarg :value-result-params 
    :initform nil 
    :type list 
    :accessor value-result-params)
   (value-update-params
    :initarg :value-update-params 
    :initform nil 
    :type list 
    :accessor value-update-params)
   (value-result-update-params 
    :initarg :value-result-update-params  
    :initform nil 
    :type list 
    :accessor value-result-update-params)

   ;; Local variables (dynamic and static)

   (dynamic-variables 
    :initarg :dynamic-variables  
    :initform nil 
    :type list 
    :accessor dynamic-variables)
   (static-variables 
    :initarg :static-variables  
    :initform nil 
    :type list 
    :accessor static-variables)

   ;; PO code (setup, init, and exit)

   (load-code 
    :initarg :load-code  
    :initform nil 
    :type t 
    :accessor load-code)
   (setup-code 
    :initarg :setup-code  
    :initform nil 
    :type t 
    :accessor setup-code)
   (init-code 
    :initarg :init-code  
    :initform nil 
    :type t 
    :accessor init-code)
   (exit-code 
    :initarg :exit-code  
    :initform nil 
    :type t 
    :accessor exit-code)

   ;; slots for common lexical children

   (panels
    :initarg :panels
    :initform nil
    :type list
    :accessor panels)
   (dialogs
    :initarg :dialogs
    :initform nil
    :type list
    :accessor dialogs)
   (frames
    :initarg :frames
    :initform nil
    :type list
    :accessor frames)
   (forms
    :initarg :forms
    :initform nil
    :type list
    :accessor forms)
   (active-lexical-children 
    :initform nil
    :type list
    :accessor active-lexical-children)

   ;; slots for managing PO initializtion and cloning

   (po-name 
    :initform nil
    :type t
    :accessor po-name)
   (in-use 
    :initform nil 
    :type atom 
    :accessor in-use)
   (next-copy 
    :initform nil 
    :type picasso-object 
    :accessor next-copy)
   (initialized 
    :initform nil 
    :type atom 
    :accessor initialized)
   (default-button
    :initform nil
    :type t
    :accessor default-button)

   ;; caches for active bindings (for ref params) and copies (for update params)

   (bindings 
    :initform nil 
    :type list 
    :accessor bindings)
   (active-copy-out 
    :initform nil 
    :type list 
    :accessor active-copy-out)))

(defmethod (setf default-button) (new-v (self picasso-object))
  (if (button-p (default-button self))
      (setf (default (default-button self)) nil))
  (setf (slot-value self 'default-button) new-v))

(defhandler ignore ((self picasso-object)
                    &rest args
                    &default :key-press)
  (declare (ignore args))
  t)

(defhandler activate-default-button ((self top-level-po) 
				&rest args
				&default (:key-press :detail #\Return))
  (let ((b (default-button self)))
       (when (button-p b)
	     (apply #'button-select b args)
	     (apply #'button-deselect b args))))

(defhandler activate-default-button ((self picasso-object)
				&rest args
				&default (:key-press :detail #\Return))
  (let ((b (default-button self)))
       (if (button-p b)
          (progn
	     (apply #'button-select b args)
	     (apply #'button-deselect b args))
          (let ((p (parent self)))
               (if (top-level-po-p p)
		   (apply #'top-level-po-activate-default-button p args)
		   (apply #'picasso-object-activate-default-button p args))))))

(defmethod new-instance :before ((self picasso-object)
				 &rest keys)
  (declare (ignore keys))
  (push-env self))

(defmethod new-instance ((self picasso-object)
			 &rest keys
			 &key
			 (ref-params nil)
			 (value-params nil)
			 (value-update-params nil)
			 (value-result-params nil)
			 (value-result-update-params nil)
			 (dynamic-variables nil)
			 (static-variables nil)
			 (constants nil)
			 (panels nil)
			 (dialogs nil)
			 (frames nil)
			 (forms nil)
			 &allow-other-keys
			 &aux (variable-list nil))

  (setf (dynamic-variables self)
	(delete nil 
		(mapcar #'(lambda (d) 
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			dynamic-variables)))

  (setf (static-variables self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			static-variables)))

  (setf (value-params self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			value-params)))

  (setf (ref-params self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			ref-params)))

  (setf (value-update-params self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			value-update-params)))

  (setf (value-result-params self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			value-result-params)))

  (setf (value-result-update-params self)
	(delete nil
		(mapcar #'(lambda (d)
			    (setq d (parse-var d))
			    (if (assoc (car d) variable-list)
			        (warn "Variable ~S multiply defined" (car d))
				(car (push d variable-list))))
			value-result-update-params)))

  (dolist (ch (append frames forms dialogs panels))
	  (let ((c (car ch))  ;; The new named constant for the child PO
		(expr (cadr ch))) ;; The expression or PO-name to generate it
	       (cond ((picasso-name-p expr)
		      (push `(,c ,expr) constants))
		     ((and (consp expr) (eq (car expr) 'quote)
			   (picasso-name-p (cadr expr)))
		      (push c constants))
		     ((picasso-object-p expr)
		      (push `(,c ,expr) constants))
		     ((and (consp expr) (eq (car expr) 'quote)
			   (picasso-object-p (cadr expr)))
		      (push c constants))
		     ((and (consp expr) (eq (car expr) 'quote))
		      (let ((tname (list "tmp" (symbol-name (gensym)))))
			   (register tname (cadr expr))
			   (push `(,c ,tname) constants)))
		     (t
		      (let ((tname (list "tmp" (symbol-name (gensym)))))
			   (register tname expr)
			   (push `(,c ,tname) constants))))))

  (setf (frames self) (mapcar #'car frames))
  (setf (forms self) (mapcar #'car forms))
  (setf (panels self) (mapcar #'car panels))
  (setf (dialogs self) (mapcar #'car dialogs))

  (push `(po ,self) constants)

  (apply #'call-next-method 
	 `(,self :constants ,constants :variables ,variable-list ,@keys)))

(defmethod new-instance :after ((self picasso-object)
				&key (ignore nil)
				&allow-other-keys)
  (declare (ignore ignore))
  (let ((po (po-name self)))
       (setf (name self) (case (name-type po)
			       (:singular po)
			       (:unqualified (concatenate 'string
							  (car po)
							  "."
							  (or (cdr po) "")))
			       (:qualified (concatenate 'string
							(car po)
							":"
							(cadr po)
							"."
							(or (cddr po)
							    ""))))))
  (dolist (v (static-variables self))
          (if (consp v)
	      (setf (value (find-var (car v) self)) (eval (cadr v)))
	      (setf (value (find-var v self)) nil)))
  (execute 'load-code  self)
  (pop-env))

(defun parse-var (var)
  (cond ((and var (symbolp var)) (cons var nil))
	((and (consp var) (car var) (symbolp (car var))) var)
	(t (warn "~S not a valid variable specifier" var))))

(defmethod form ((self picasso-object))
  (car (forms self)))

(defmethod get-form ((self picasso-object))
  (let ((res #!((lookup (form self) self))))
       (if (picasso-object-p res) res (find-picasso-object-named res))))

(defmethod (setf form) (val (self picasso-object))
  (cond ((form-p val) (setf (forms self) (list val)))
	((and (listp val) (form-p (car val))) (setf (forms self) val))
	(t (warn "Bad setf form"))))

(defclass top-level-po (picasso-object)
  ((status :initform :concealed)
   (lexical-parent :initform (root-window))
   (autowarp
    :initarg :autowarp
    :initform t
    :type atom
    :accessor autowarp)
   (autoraise
    :initarg :autoraise
    :initform t
    :type atom
    :accessor autoraise)
   (icon-name
    :initarg :icon-name
    :initform ""
    :type string
    :accessor icon-name)
   (icon
    :initarg :icon
    :initform nil
    :type t
    :accessor icon)
   (event-mask
    :initform '(:exposure :structure-notify :key-press))))

(defmethod (setf title) (value (self top-level-po))
  (unless (equalp value (title self))
	  (call-next-method)
	  (setf (icon-name self) value)))

(defmethod new-instance ((self top-level-po) 
			 &key
			 (title "")
			 &allow-other-keys)
  (setf (icon-name self) title)
  (call-next-method))

(defclass callable-po (picasso-object)
  ((form-args 
    :initarg :form-args 
    :initform nil
    :type list
    :accessor form-args)))

(defmethod new-instance ((self callable-po) 
			 &rest keys
			 &key
			 (gm (gm self))
			 (buttons nil)
			 (menu-bar nil)
			 (form-children nil)
			 (form-visit-order nil)
			 (form-selectable nil)
			 (constants nil)
			 (forms nil)
                         (form-background nil)
                         (form-gray nil)
			 &allow-other-keys
			 &aux bl)

  (setf (gm self) 'packed-gm)

  (if forms
      (progn
       (when (> (length forms) 1)
	     (warn "Only one form can be supplied to a panel, using first")
	     (setq forms (list (car forms))))
       (when form-children
	     (warn "Children cannot be specified when a form is selected!")
	     (warn "Ignoring children.")
	     (setq form-children nil)))
      (let ((name (cons "temp" (cons (string (gensym)) "form"))))
	   (eval
	    `(defform ,name ()
		      (gm ,gm)
                     ,@(if form-background 
                           (list (list 'background form-background)))
                     ,@(if form-gray
                           (list (list 'background "gray75")))
		      (constants ,@constants)
                      (visit-order ,@form-visit-order)
                      (selectable ,@form-selectable)
		      (children ,@form-children)))
	   (setq forms (list (list 'form name)))))

  (apply #'call-next-method self :forms forms keys)

  (when buttons
	(setf (geom-spec (get-form self)) '(:fill :pad 5))
	(setq bl (eval (make-button-strip buttons self)))
	(expose bl))

  (when menu-bar
	(if (menu-bar-p menu-bar)
	    (progn
	     (setf (geom-spec menu-bar) '(:top :before))
	     (setf (parent menu-bar) self))
	    (setq menu-bar (make-menu-bar :parent self 
					  :structure menu-bar
					  :geom-spec '(:top :before))))
	(expose menu-bar))
  )

(defun make-button-strip (button-spec-list parent &aux bsl w h font)
  (setq bsl (length button-spec-list)
	font (get-font "-b&h*bold-r*14*"))
  (setq w
	(apply #'max
	       (mapcar
		#'(lambda (b)
			  (if (and (listp b) (= (length b) 2) (symbolp (car b)))
			      (setq b (cadr b)))
			  (setq b (car b))
			  (remove-quote b)
			  (cond ((stringp b)
				 (+ (text-width b :font font) 20))
				((image-p b)
				 (+ (width b) 20))
				((and (listp b) (every #'stringp b))
				 (+ 20
				    (apply #'max
					(mapcar 
					 #'(lambda (s)
						   (text-width s :font font))
					 b))))

				(t 1)))
		button-spec-list)))
  (setq h 
	(apply #'max
	       (mapcar 
		#'(lambda (b)
			  (if (and (listp b) (= (length b) 2) (symbolp (car b)))
			      (setq b (cadr b)))
			  (setq b (car b))
			  (remove-quote b)
			  (cond
			   ((image-p b)
			    (+ (height b) 15))
			   ((and (listp b) (every #'stringp b))
			    (+ (* (length b) (font-height font)) 15))
			   (t (+ 15 (font-height font)))))
		button-spec-list)))
  `(make-collection-gadget :gm 'anchor-gm
			   :parent ',parent
			   :base-width (+ ,w 15)
			   :base-height (* (+ ,h 5) (length ',button-spec-list))
			   :geom-spec :right
			   :children
			   ',(delete nil
				     (maplist #'(lambda (b)
							(handle-button-spec
							 (car b)
							 (- bsl (length b))
							 bsl
							 w h))
					      button-spec-list))))

(defun handle-button-spec (but pos denom w h)
  (prog (label value default code retval options)
	(unless (and (consp but)
		     (or (eql (length but) 3)
			 (and (evenp (length but)) (>= (length but) 2))))
		(warn "Button-spec ~S not valid" but)
		(return-from handle-button-spec nil))
	(if (and (symbolp (car but)) (consp (cadr but)))
	    (setq label (car but) but (cadr but)))
	(setq value (car but))
	(setq code (cadr but))
	(if (evenp (length but))
	    (setq options (cddr but))
	    (setq default (caddr but)))
	(setq retval `(make-gray-button :geom-spec '(0.05 ,(/ pos denom)
						     0.9  ,(/ 1 denom)
						     :arrow (:horiz))
					:base-height ,h
					:base-width ,w
					:value ,value
					:func ',code
					,@options
					:default ,default))
	(return-from handle-button-spec
		     (if label
			 (list label retval)
			 retval))))

(defmethod invoke :before ((self top-level-po) &rest arguments)
  (declare (ignore arguments))
  (unless (parent self) (setf (parent self) (root-window)))
  (unless (lexical-parent self) (setf (lexical-parent self) (parent self))))

(defmethod invoke :before ((self picasso-object) &rest arguments)
  (declare (ignore arguments))
  (if (in-use self)
      (error "Picasso-object ~S already in use (cloning not implemented)" self)
      (setf (in-use self) t))
  (unless (attached-p self) (attach self))
  (raise self)
  (push-env self))

(defmethod invoke :after ((self picasso-object) &rest arguments)
  (declare (ignore arguments))
  (let ((par (or (lexical-parent self) (parent self))))
       (if (cares-about par self)
	   (push self (active-lexical-children par))))
  (unless (initialized self)
	  (execute 'setup-code self)
	  (setf (initialized self) t))
  (execute 'init-code self)
  (pop-env))


(defmethod invoke :after ((self top-level-po) &rest arguments)
  (declare (ignore arguments))
  (when (autowarp self)
        (warp-mouse self (round (width self) 2) (round (height self) 2)))
  (expose self :x-map nil)
  (let ((res (res self)))
       (xlib:send-event res :client-message nil :type :x-map :format 32
			:data nil :window res :event-window res)))

(defhandler map ((self top-level-po) &key window &allow-other-keys 
		 &default :map-notify) 
  (if (and (eq window (res self)) (concealed-p self) (viewable-p self))
      (expose self :x-map nil)))

(defhandler unmap ((self top-level-po) &key window &allow-other-keys 
		   &default :unmap-notify) 

  (unless (slot-value self 'position-specified)
	  (xlib:set-standard-properties (res self)
					:user-specified-position-p t
					:x (x-offset self)
					:y (y-offset self))
	  (setf (slot-value self 'position-specified) t))

  (if (and (eq window (res self)) (exposed-p self) (not (viewable-p self)))
      (conceal self :x-unmap nil)))

(defhandler map-it ((self top-level-po) &rest args
		    &default (:client-message :detail :x-map))
  (declare (ignore args))
  (xlib:map-window (res self))
  (flush-display (display self)))

(defmethod invoke ((self callable-po) &rest arguments)
  (declare (ignore arguments))
  (call-next-method)
  (migrate-vars (get-form self) self))

(defmethod invoke :after ((self callable-po) &rest arguments)
  (let ((fa (or (getf arguments :form-args) (form-args self))))
       (apply #'invoke (get-form self) fa)))

(defun migrate-vars (from to)
  "Make clone variables for form parents"
  (maphash
   #'(lambda (name variable)
	     (unless (gethash name (vtab to))
		     (setf (gethash name (vtab to))
			   (gethash name (vtab from)))))
   (vtab from)))

(defmethod invoke ((self picasso-object) &rest arguments)
  (let ((dyn (dynamic-variables self))
	(val (value-params self))
	(ref (ref-params self))
	(vu (value-update-params self))
	(vru (value-result-update-params self))
	(vr (value-result-params self)))

       (dolist (a val)
	       (let ((place (find-var (car a) self))
		     (arg-val (getf arguments (colonize (car a)) :not-provided))
		     (def-val (cadr a)))
		    (setf #!(place) (if (eq arg-val :not-provided)
					  (eval def-val)
					  arg-val))))
       (dolist (a ref)
	       (let ((arg-val (getf arguments (colonize (car a)) :not-provided))
		     (def-val (cadr a))
		     (place (find-var (car a) self)))
		    (if (eq arg-val :not-provided)
			(progn
			 (warn "Ref param ~S not provided-using default" 
			       (car a))
			 (setf #!(place) (eval def-val)))
			(progn
			 (push (bind-var (car a) self 
					 `(var value ,arg-val) :receipt t)
			       (bindings self))
			 (push (bind-slot 'value arg-val
					  `(var ,(car a) :ref ,self) :receipt t)
			       (bindings self))))))
       (dolist (a vr)
	       (let ((arg-val (getf arguments (colonize (car a)) :not-provided))
		     (def-val (cadr a))
		     (place (find-var (car a) self)))
		    (if (eq arg-val :not-provided)
			(progn
			 (warn "Value-result param ~S not provided"
			       (car a))
			 (setf #!(place) (eval def-val)))
			(progn
			 (setf #!(place) #!(arg-val))
			 (push (cons place arg-val) (active-copy-out self))))))
       (dolist (a vru)
	       (let ((arg-val (getf arguments (colonize (car a)) :not-provided))
		     (def-val (cadr a))
		     (place (find-var (car a) self)))
		    (if (eq arg-val :not-provided)
			(progn
			 (warn "Value-result-update param ~S not provided" 
			       (car a))
			 (setf #!(place) (eval def-val)))
			(progn
			 (push (bind-var (car a) self 
					 `(var value ,arg-val) :receipt t)
			       (bindings self))
			 (push (cons place arg-val) (active-copy-out self))))))
       (dolist (a vu)
	       (let ((arg-val (getf arguments (colonize (car a)) :not-provided))
		     (def-val (cadr a))
		     (place (find-var (car a) self)))
		    (if (eq arg-val :not-provided)
			(progn
			 (warn "Value-update param ~S not provided" 
			       (car a))
			 (setf #!(place) (eval def-val)))
			(progn
			 (push (bind-var (car a) self 
					 `(var value ,arg-val) :receipt t)
			       (bindings self))))))
       (dolist (v dyn)
	       (setf (value (find-var (car v) self)) (eval (cadr v))))

       (unless (initialized self)
	       (relax-constants)
	       (dolist (po (append (frames self) (forms self) 
				   (panels self) (dialogs self)))
		       (setq po (lookup po))
		       (link #!(po))
		       (if (typep #!(po) 'top-level-po)
			   (setf (lexical-parent #!(po)) self)
			   (setf (parent #!(po)) self)))
	       (enforce-constants))))

(defmethod leave ((self picasso-object) &optional (return-value nil))
  (mapc #'unbind-fast (bindings self))
  (setf (bindings self) nil)
  (unless (eql return-value :cancel)
	  (mapc #'(lambda (copy) (setf #!((cdr copy)) #!((car copy))))
		(active-copy-out self)))
  (setf (active-copy-out self) nil))

(defmethod leave :before ((self picasso-object) &optional (return-value nil))
  (declare (ignore return-value))
  (dolist (kid (active-lexical-children self))
	  (if (needs-catch-p kid)
	      (catch kid (leave kid))
	      (leave kid)))
  (execute 'exit-code self))

(defmethod leave :after ((self picasso-object) &optional (return-value nil))
  (declare (ignore return-value))
  (let ((par (or (lexical-parent self) (parent self))))
       (setf (active-lexical-children par)
	     (delete self (active-lexical-children par))))
  (setf (in-use self) nil))

(defmethod leave :after ((self top-level-po) &optional (return-value nil))
  (declare (ignore return-value))
  (conceal self))

(defmethod copy-out ((self picasso-object))
  (mapc #'(lambda (copy) (setf #!((cdr copy)) #!((car copy))))
	(active-copy-out self)))

(defmethod invoke (self &rest args)
  (declare (ignore args))
  (warn "Attempt to invoke non-picasso-object ~S" self)
  nil)

(defmethod leave (self &optional retval)
  (declare (ignore retval))
  (warn "Attempt to leave non-picasso-object ~S" self)
  nil)

(defmacro call (self &rest args)
  `(invoke ,self ,@args))

(defmethod do-conceal ((self picasso-object)
		       &key 
		       (ignore nil)
		       &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method) 
  (mapc #'conceal (active-top-level-children self)))

(defmethod do-expose ((self picasso-object)
		      &key 
		      (ignore nil)
		      &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (if (exposed-p self)
      (mapc #'expose (active-top-level-children self))))

(defmethod do-make-invisible ((self picasso-object)
			      &key 
			      (ignore nil)
			      &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (mapc #'conceal (active-top-level-children self)))

(defmethod do-make-uninvisible ((self picasso-object)
				&key 
				(ignore nil)
				&allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (if (exposed-p self)
      (mapc #'expose (active-top-level-children self))))

(defun active-top-level-children (po)
  (remove-if-not #'(lambda (x) (top-level-po-p x)) 
		 (active-lexical-children po)))

(defhandler expose ((self top-level-po) &key count &allow-other-keys
		    &default :exposure)
  (if (and (> count 0) (null (partial-repaint-p self)))
      (return-from top-level-po-expose))
  (do-expose self)
  (repaint self))

(defhandler configure ((self top-level-po) &key x y width height
		       &allow-other-keys
		       &aux oldx oldy oldw oldh repack-flag
		       &default :configure-notify)
  (setq oldx (slot-value self 'x-offset)
	oldy (slot-value self 'y-offset)
	oldw (slot-value self 'width)
	oldh (slot-value self 'height))
  (if (not (= x oldx))
      (setf (slot-value self 'x-offset) x))
  (if (not (= y oldy))
      (setf (slot-value self 'y-offset) y))
  (when (not (= width oldw))
	(setq repack-flag t)
	(setf (slot-value self 'width) width))
  (when (not (= height oldh))
	(setq repack-flag t)
	(setf (slot-value self 'height) height))
  (do-repack self))
