;;; 
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; $Author: konstan $
;;; $Source: RCS/form.cl,v $
;;; $Revision: 1.9 $
;;; $Date: 90/07/30 15:10:41 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;
;;  OVERVIEW
;;
;;  A form is a packed window that allows various operations within
;;  it.  All keyboard input is sent to the "current-field".
;;  The current-field must be on the visit order of the form-window.  
;;  Changing the current field can be accomplished in two ways:  1) clicking 
;;  on another field in the visit order of the form-window, or 2) pressing 
;;  tab.  Clicking on another field in the visit order will make that field 
;;  the current-field.  Pressing tab makes the next field in the visit order 
;;  the current-field.  If the current-field is the last field in the visit 
;;  order, the next field is defined as being the first field in the visit 
;;  order (the visit order is circular).
;;

;;;
;;;  	Define form class
;;;

(defclass form (picasso-object)
  (
   ;;  Inherited slots with overridden defaults

   (event-mask :initform '(:exposure :enter-window :leave-window :key-press))
   (parent :initform nil)
   (conform :initform :grow-shrink)
   (geom-spec :initform :fill)
   (border-width :initform 0)
   (gm :initform 'anchor-gm)

   ;;  Slots to handle current field management
   (current-field 
    :initarg :current-field  
    :initform nil 
    :type t 
    :accessor current-field)
   (current-repaint-func
    :initarg :current-repaint-func
    :initform nil
    :type t
    :accessor current-repaint-func)
   (visit-order 
    :initarg :visit-order  
    :initform nil 
    :type list 
    :accessor visit-order)
   (border-info  ;; for the current field
    :initform nil
    :type cons)))

;;;
;;; 	Make a form instance
;;;

(defun make-form (&rest keys)
  (apply #'make-instance 'form :allow-other-keys t keys))

;;;
;;;	Handle events
;;;

;;(defhandler notify-next ((self form) &rest args &aux vo next child
;;                         &default (:key-press :control #\n))
;;  (declare (ignore args))
;;  (setq child (current-field self))
;;  (format t "in form-notify-next~%")
;;  (setq vo (visit-order self))
;;  (setq next (cadr (member child vo)))
;;  (if (not next)
;;      (setq next (car vo)))
;;  (setf (current-field self) next))

;;(defhandler notify-prev ((self form) &rest args &aux vo prev child
;;                         &default (:key-press :control #\p))
;;  (declare (ignore args))
;;  (setq child (current-field self))
;;  (format t "in form-notify-prev~%")
;;  (setq vo (reverse (visit-order self)))
;;  (setq prev (cadr (member child vo)))
;;  (if (not prev)
;;      (setq prev (car vo)))
;;  (setf (current-field self) prev))
  
(defhandler null-func ((self form) &rest args
                       &default ((:key-press) (:enter-window) (:leave-window)))
  (declare (ignore self args)))

;;(defhandler clean-up ((self form) &rest args
;;                          &default (:client-message :detail :clean-up))
;;  (declare (ignore args))
;;  (funcall (current-repaint-func self) self))

(defun safe-set-focus (win)
  (if (viewable-p win)
      (xlib:set-input-focus (res (display win)) (res win) :pointer-root)
      (xlib:set-input-focus (res (display win)) :pointer-root :pointer-root)))

(defmethod (setf current-field) (val (self form) &aux oldval)
  (setq oldval (slot-value self 'current-field))
  (cond ((eq oldval val) nil)
	(t
	 (setf (slot-value self 'current-field) val)
	 (when oldval (deactivate oldval))
         (when val (activate val))
         (if (x-window-p val)
	     (safe-set-focus val)
	     (safe-set-focus self)))))

(defmethod current-field ((self callable-po))
  (current-field (get-form self)))

(defmethod (setf current-field) (val (self callable-po))
  (setf (current-field (get-form self)) val))

;;(defmethod do-attach :after ((self form))
;;  (when (exposed-p self)
;;	(let ((cf (current-field self)))
;;	     (setf (slot-value self 'current-field) nil)
;;	     (setf (current-field self) cf))))

;;(defmethod do-expose :after ((self form) &rest ignore)
;;  (declare (ignore ignore))
;;  (when (exposed-p self)
;;	(let ((cf (current-field self)))
;;	     (setf (slot-value self 'current-field) nil)
;;	     (setf (current-field self) cf))))

;;(defmethod (setf current-field) (val (self form) &aux oldval rv)
;;  (setq oldval (slot-value self 'current-field))
;;  (if (not (and (setq rv (current-repaint-func self)) (functionp rv)))
;;      (cond ((null oldval) 
;;	     (setf (slot-value self 'current-field) val)
;;	     (when val
;;		   (setf (slot-value self 'border-info) 
;;			 (cons (border-type val) (border-width val)))
;;		   (setf (border-type val) :black-frame)
;;		   (activate val)
;;		   (if (x-window-p val)
;;		       (xlib:set-input-focus (res (display val)) (res val) 
;;					     :pointer-root)
;;		       (xlib:set-input-focus (res (display self)) (res self)
;;					     :pointer-root))
;;		   (label-repaint :frame val)))
;;	    ((eq oldval val) nil)
;;	    (t
;;	     (setf (slot-value self 'current-field) val)
;;	     (let ((bi (slot-value self 'border-info)))
;;		  (setf (border-type oldval) (car bi)
;;			(border-width oldval) (cdr bi))
;;		  (deactivate oldval)
;;		  (label-repaint (label-type oldval) oldval)
;;		  (when val
;;			(setf (slot-value self 'border-info) 
;;			      (cons (border-type val) (border-width val)))
;;			(setf (border-type val) :black-frame) 
;;			(activate val)
;;			(if (x-window-p val)
;;			    (xlib:set-input-focus (res (display val)) (res val) 
;;						  :pointer-root)
;;			    (xlib:set-input-focus (res (display self)) 
;;						  (res self)
;;						  :pointer-root))
;;			(label-repaint :frame val)))))))

(defmethod activate ((self t))
  (when (viewable-p self)
        (warp-mouse self (round (width self) 2) (round (height self) 2))))

(defmethod deactivate ((self t))
  t)


;;(defun form-valid-window (win form)
;;  (if (or (eq (parent win) form) (not (eql (border-type win) :box)))
;;      win
;;      (form-valid-window (parent win) form)))

;;(defun form-intern-set-vo (win form &aux cp xp) 
;;  (defun form-notify-select (child &rest args &aux mapping)
;;	 (setq mapping (form-valid-window child form))
;;	 (if (eq mapping (current-field form))
;;	     (progn
;;	      (setq mapping (find-entry (class-event-map child) 
;;					(cons :button-press 
;;					      (cdr (descriptor args)))))
;;	      (cond ((function-p mapping)
;;		     (apply mapping (cons child args)))
;;		    ((null mapping) nil)
;;		    (t (warn "event mapped into something unexpected: ~s" 
;;			     mapping)))) 
;;	     (setf (current-field form) (form-valid-window child form)))) 
;;  (defun form-notify-next (&rest args &aux vo next child)
;;	 (declare (ignore args))
;;	 (setq child (current-field form))
;;	 (format t "in form-notify-next~%")
;;	 (setq vo (visit-order form))
;;	 (setq next (cadr (member child vo)))
;;	 (if (not next)
;;	     (setq next (car vo)))
;;	 (setf (current-field form) next)) 
;;  (defun form-notify-prev (&rest args &aux vo prev child)
;;	 (declare (ignore args))
;;	 (setq child (current-field form))
;;	 (format t "in form-notify-prev~%")
;;	 (setq vo (reverse (visit-order form)))
;;	 (setq prev (cadr (member child vo)))
;;	 (if (not prev)
;;	     (setq prev (car vo)))
;;	 (setf (current-field form) prev))
;;  (setq cp (collection-p win) xp (x-window-p win))
;;  (cond ((and cp xp) 
;;	 (pushnew :button-press (event-mask win))
;;	 (pushnew :key-press (event-mask win))
;;	 (register-callback win #'form-notify-select :button-press)
;;	 (register-callback win #'form-null-func :key-press)
;;	 (register-callback win #'form-notify-next :key-press 
;;			    :state :control :detail #\n)
;;	 (register-callback win #'form-notify-prev :key-press 
;;			    :state :control :detail #\p)
;;	 (dolist (ch (children win))
;;		 (form-intern-set-vo ch form)))
;;	(xp (pushnew :button-press (event-mask win))
;;	    (register-callback win #'form-notify-select :button-press)
;;;;	    (register-callback win #'form-null-func :key-press)
;;	    (register-callback win #'form-notify-next :key-press 
;;			       :state :control :detail #\n)
;;	    (register-callback win #'form-notify-prev :key-press 
;;			       :state :control :detail #\p))
;;	(cp (dolist (ch (children win))
;;		    (form-intern-set-vo ch form)))))
;;
;;(defmethod (setf visit-order) (val (self form))
;;  (setf (slot-value self 'visit-order) val)
;;  (dolist (ch val)
;;	  (form-intern-set-vo ch self))
;;  val)

(defmethod new-instance :after ((self form) 
				&key 
				(visit-order nil)
				(selectable nil)
				&allow-other-keys)
  (unless selectable (setq selectable visit-order))
  (when selectable
	(push-env self)
	(setq selectable (mapcar #'eval selectable))
        (mapc #'(lambda (f) (deactivate f)) selectable)
	(mapc #'(lambda (f) (handle-button f self)) selectable))
  (when visit-order
	(push-env self)
	(setq visit-order (mapcar #'eval visit-order))
	(mapc #'(lambda (f) (unless (member :key-press (event-mask f))
				    (push :key-press (event-mask f))))
	      visit-order)
	(mapl #'(lambda (l) (handle-next l self))
	      (append visit-order (list (car visit-order))))
	(mapl #'(lambda (l) (handle-prev l self))
	      (reverse (cons (car (last visit-order)) visit-order)))
        (mapc #'(lambda (f) (deactivate f)) visit-order)
	(setf (current-field self) (car visit-order))
	(pop-env)))

(defun handle-button (field form)
  (let ((default-fxn (lookup-event-mapping field '(:button-press 1))))
       (register-callback field 
			  #'(lambda (&rest args)
				    (if (activated field)
					(if default-fxn
					    (apply default-fxn args))
					(setf (current-field form) field)))
			  :button-press :detail :left-button)))

(defun handle-next (fields form)
  (let ((from (car fields))
	(to (cadr fields)))
       (when (and from to)
	     (register-callback from #'(lambda (&rest args)
					  (setf (current-field form) to))
				:key-press :state :control :detail #\n)
	     (register-callback from #'(lambda (&rest args)
					  (setf (current-field form) to))
				:key-press :detail #\tab))))

(defun handle-prev (fields form)
  (let ((from (car fields))
	(to (cadr fields)))
       (when (and from to)
	     (register-callback from #'(lambda (&rest args)
					  (setf (current-field form) to))
				:key-press :state :control :detail #\p)
	     (register-callback from #'(lambda (&rest args)
					  (setf (current-field form) to))
				:key-press :state :shift :detail #\tab))))

(defmethod invoke :after ((self form) &rest args)
  (declare (ignore args))
  (expose self))

(defun enter-form (form)
  (let ((cf (current-field form)))
       (if cf (safe-set-focus cf) (safe-set-focus form))))

(defun leave-form (form)
  (xlib:set-input-focus (res (display form))  :pointer-root :pointer-root))

(defmethod do-expose ((self form) &rest args &aux rf)
  (declare (ignore args))
  (call-next-method)
  (if (and (setq rf (current-repaint-func self)) (functionp rf))
      ;;  Send self an event as a reminder to repaint-currents after
      ;;  all other processing (on children) is done.
      (let ((res (res self)))
	   (xlib:send-event res :client-message nil
			    :type :clean-up :format 32 :data nil 
			    :window res :event-window res))))
