;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/gray-pop-button.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 20:39:04 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;;	A gray-pop-button is like any other button except that it pops up a menu
;;;	pane when selected which has user-specified behavior.  By default,
;;;	the menu-items just set the value of the button to their value.
;;; 	Pop buttons take a list of strings (:items '("hello" "there" . . .))
;;;	and a font along with all the other button arguments.  
;;;	Optionally, the :items may be a list of lists where each list has
;;;	an object and an expression to eval (code for the menu-entries)
;;;	For example
;;;		:items '(("hello" '(print "This is Great"))
;;;			 ("good-bye" `(print ',val))
;;;			 "welcome"
;;;			 ("cancel" nil))
;;;	

(defclass gray-pop-button (button)
  ((menu :initarg :menu :initform nil :type menu-pane :accessor menu)
   (gc-spec :initform '((gc-res "default") 
			(gc-white (:line-width 2 :foreground "white"
				  :cap-style :projecting))
			(gc-black (:line-width 2 :cap-style :projecting))))
   (gc-black :initform nil :type vector :reader gc-black)
   (gc-white :initform nil :type vector :reader gc-white)
   (items-font :initform nil :accessor items-font)
   (background :initform :parent-relative)
   (border-width :initform 0)
   (event-mask :initform '(:exposure :button-press :button-release))))

(defun make-gray-pop-button (&rest args)
  (apply #'make-instance 'gray-pop-button :allow-other-keys t args))

(defmacro gray-pop-button-p (pb)
  `(typep ,pb 'gray-pop-button))

(defmethod (setf release-func) (val (self gray-pop-button))
  (declare (ignore val))
  (warn "Can't set the release-func of a gray-pop-button"))

(defmethod (setf items) (items (self gray-pop-button))
  (destroy (menu self))
  (let ((mp (make-menu-pane :tearable nil))
	(items-font (items-font self)))
       (setf (menu self) mp)
       (setf (lexical-parent mp) #!po)
       (dolist (val items)
	       (if (atom val)
		   (make-menu-entry :parent mp
				    :center val
				    :font items-font
				    :code `(setf (value ',self) ,val))
		   (make-menu-entry :parent mp
				    :center (car val)
				    :font items-font
				    :code (cadr val))))
       (if (attached-p self) (attach mp))))

(defmethod (setf inverted) (val (self gray-pop-button) &aux gc oldval)
  (setq oldval (slot-value self 'inverted)
	gc (gc-res self))
  (setf (slot-value self 'inverted) val)
  (when (or (and (null val) oldval) (and val (null oldval)))
	(psetf (xlib:gcontext-foreground gc) (xlib:gcontext-background gc)
	       (xlib:gcontext-background gc) (xlib:gcontext-foreground gc))
	(xlib:force-gcontext-changes gc))
  (repaint self))

(defmethod new-instance ((self gray-pop-button)
			 &key 
			 (items nil)
			 (items-font "-b&h*bold-r*14*")
			 &allow-other-keys
			 &aux mp)
  (call-next-method)
  
  (setf (items-font self) items-font)
  ;;	Make menu-pane
  (setf (menu self)
	(setq mp (make-menu-pane :tearable nil))) 
  (setf (lexical-parent mp) #!po)
  
  ;;	Make menu-entries
  (dolist (val items)
	  (if (atom val) 
	      (make-menu-entry :parent mp :center val 
			       :font items-font
			       :code `(setf (value ',self) ,val))
	      (make-menu-entry :parent mp :center (car val) 
			       :font items-font
			       :code (cadr val))))

  ;;	Set function
  (setf (press-func self)
	'(progn
	  (setf (inverted self) t)
	  (clear self)
	  (do-repaint self)
	  (flush-display (display self))
	  (activate-pop-up-menu (menu self) event)
	  (setf (inverted self) nil
		(flag self) t
		(pushed self) nil)
	  (clear self)
	  (do-repaint self))))

(defmethod do-attach ((self gray-pop-button) &aux menu)
  (call-next-method)
  (when (setq menu (menu self))
	(attach menu)
	(setf (xlib:transient-for (res menu))
	      (res self))))

(defmethod do-detach ((self gray-pop-button) &aux menu)
  (call-next-method)
  (when (setq menu (menu self))
	(detach menu)))

(defmethod destroy ((self gray-pop-button) &aux menu)
  (call-next-method)
  (when (setq menu (menu self))
	(destroy menu)))

(defmethod do-repaint ((self gray-pop-button) &aux h w)
  (if (inverted self)
      (xlib:draw-arc (res self) (gc-black self) 
		     0 0 (width self) (height self) 0 (* 2 pi) t))
  (call-next-method)
  (setq h (height self)
	w (width self))
  (draw-curved-border self (gc-black self) (gc-white self) 
		      0 0 (width self) (height self)))

(defun draw-curved-border (win black-gc white-gc x y w h &key invert
			       &aux res r)
  (setq res (res win))
  (if invert
      (psetq black-gc white-gc
	     white-gc black-gc))
  (setq r (round (/ h 3)))
  (decf h)
  (decf w)
  (xlib:draw-arc res white-gc x y h h (/ pi 2) pi )
  (xlib:draw-arc res black-gc (- (+ x w) h) y h h (* 3 (/ pi 2)) pi)
  (xlib:draw-line res white-gc (+ x r) (1+ y) (- (+ x w) r) (1+ y))
  (xlib:draw-line res black-gc (+ x r) (+ y h) (- (+ x w) r) (+ y h))
  (xlib:draw-point res black-gc (+ x (- r 2)) (+ y h))
  (xlib:draw-point res black-gc (- (+ x w) r) (1+ y)))
