;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: konstan $
;;; $Source: RCS/button.cl,v $
;;; $Revision: 1.4 $
;;; $Date: 90/07/31 12:46:41 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; button class
;;;

(defclass button (widget)
  ((name :initform "A Button")
   (border-width :initform 1)
   (event-mask 
    :initform '(:exposure :button-press :button-release 
			  :leave-window :enter-window ))

   ;;  indicates whether or not button is default 
   (default :initarg :default :initform nil :type t :accessor default)
   (base-width :initform 0)
   (base-height :initform 0)

   ;; time to leave button pushed, after being selected and before
   ;; calling function.  CAN be fractional.
   (pause-seconds 
    :initarg :pause-seconds  
    :initform nil 
    :type t 
    :accessor pause-seconds)

   ;;  indicates whether button is currently pushed or not
   (pushed 
    :initform nil 
    :type t 
    :accessor pushed)

   ;;  used in event handling. . .
   (flag :initform nil :type t :accessor flag)

   (font :initform "-b&h*bold-r*14*")
   (gc-spec :initform '(gc-res "default"))

   (mask :initarg :mask :initform nil :type atom :reader mask)
   (data :initarg :data :initform nil :type t :accessor data)
   (press-func 
    :initarg :press-func  
    :initform nil 
    :type t 
    :accessor press-func)
   (release-func 
    :initarg :release-func  
    :initform nil 
    :type t 
    :accessor release-func)))

(defun make-button (&rest keys 
                    &key (pop nil) (gray nil) &allow-other-keys)
  (remf keys :gray)
  (cond  
   ((and pop gray) (apply #'make-gray-pop-button keys))
   (pop            (apply #'make-pop-button keys))
   (gray           (apply #'make-gray-button keys))
   (t              (apply #'make-instance 'button :allow-other-keys t keys))))

;;
;;	Determine whether or not an object is a button
;;

(defmacro button-p (self)
  `(typep ,self 'button))

;;;
;;; Accessor methods
;;;

(defmethod value ((self button))
  (car (slot-value self 'value)))

(defmethod (setf value) (val (self button) &aux sval)
  (unless (and (setq sval (slot-value self 'value))
	       (flag self))
	(setq sval (list nil :window self :width (width self) 
			 :height (height self) :gc (gc-res self) 
			 :mask (mask self) :dimmed (dimmed self))))
  (when (image-p val)
	(setf (getf (cdr sval) :bitmap-p) (bitmap-p val)))
  (setf (slot-value self 'value) 
	(rplaca sval val))
  (repaint self))

(defmethod (setf mask) (val (self button) &aux synth)
  (when (and (consp (setq synth (slot-value self 'value))) (second synth))
	(setf (getf (cdr synth) :mask) val)))

(defmethod (setf background) (val (self button))
  (call-next-method)
  (setf (mask self)
	(if (tile-p (background self)) 
	    nil
	    (color-display-p (display self)))))

(defmethod (setf dimmed) (new-value (self button) &aux sval)
  (setq sval (slot-value self 'dimmed)
	new-value (not (null new-value)))
  (unless (eq new-value sval)
	  (call-next-method)
	  (setf (getf (cdr (slot-value self 'value)) :dimmed) new-value)
	  (repaint self)))

(defmethod (setf inverted) (val (self button))
  (declare (ignore val))
  (call-next-method)
  (repaint self))

(defmethod func ((self button))
  (release-func self))

(defmethod (setf func) (value (self button))
  (setf (release-func self) value))

(defmethod (setf default-button) (val (self t))
  (declare (ignore val))
  t)

;;;
;;;	New-instance method
;;;

(defmethod new-instance ((self button)
			 &key 
			 (value		nil)
			 (base-size	nil)
			 (base-width	nil)
			 (base-height	nil)
			 (width		nil)
			 (height 	nil)
			 (mask 		(color-display-p))
			 (cursor 	(get-cursor "push" (display self)))
			 (func		nil)
			 &allow-other-keys)
  
  (call-next-method)

  (when base-size 
	(setq base-width (car base-size)
	      base-height (cadr base-size)))

  (when (and base-width (not width))
	(setq width base-width))

  (when (and base-height (not height))
	(setq height base-height))

  (setf (value self) value)
  (setf (cursor self) cursor
	(mask self) mask)
  (when func (setf (func self) func))
  (setf (flag self) t)
  (setf (font self) (font self))
  self)

(defun my-sleep (num &aux et)
  (dotimes (i (round (* num 600)))
	   (event-loop 
	    :hang nil
	    :handler
	    #'(lambda (&rest args)
		      (setq et (getf args :event-key))
		      (when (or (eq et :button-release) (eq et :leave-notify))
			    (apply #'dispatch-event args)
			    (return-from my-sleep))
		      nil))))

;;;
;;; Button event handlers
;;;

(defhandler select ((self button) &rest args 
		    &aux pause flag func et
		    &default :button-press)
  (declare (list args))
  (when (dimmed self)
	(return-from button-select))
  (setf (inverted self) t)
  (setq pause (pause-seconds self))
  (setq flag (flag self))
  (setf (pushed self) t
	(flag self) nil)

  ;;	Call press-func if necessary
  (when flag
	(flush-display (display self))
	(when (setq func (press-func self))
	      (execute 'press-func self args)))
  
  ;; Now, wait around until we get leave-window or button-released.
  ;; If (pause-seconds self) passes and we don't get one, call function
  ;; again.  All other events go to normal dispatcher.
  (when (and (numberp pause) func)
	(my-sleep pause)
	(loop
	 (event-loop 
	  :hang nil
	  :handler
	  #'(lambda (&rest args)
		    (setq et (getf args :event-key))
		    (when (or (eq et :button-release) (eq et :leave-notify))
			  (apply #'dispatch-event args)
			  (return-from button-select))
		    nil))
	 (execute 'press-func self args)))
  #|(if (numberp pause)
      (let ((ev-list nil)
	    (button-id (id self))
	    (cursor-id (id (if (cursor self) (cursor self) (default-cursor))))
	    (et 0))
	   (declare (integer button-id et)
		    (list ev-list))
	   (do ((done nil done))
	       (done)
	       (setq ev-list (x-next-event-nohang :count t))
	       (if (null ev-list) (sleep pause))
	       (dolist (ev ev-list)
		       (setq et (x-event-type ev))
		       (when (= (x-event-window ev) button-id)
			     (setq event ev
				   done (or (= et *button-released*) 
					    (= et *leave-window*)))
			     (dispatch-xcl-event ev)))
	       (when (not done)
		     (x-flush)
		     (execute 'press-func self event)))))|#
  )

(defhandler deselect ((self button) &rest args &aux func 
		      &default :button-release)
  (declare (list args))
  (when (dimmed self)
	(return-from button-deselect))
  ;; 	Pause for specified time
  (when (numberp (pause-seconds self))
	(sleep (pause-seconds self)))
  ;;	Un-invert the area enclosed within the border if pushed
  (setf (flag self) t)
  (when (pushed self)
	(setf (inverted self) nil)
	;;	Call release-function
	(flush-display (display self))
	(setf (pushed self) nil)
	(when (setq func (release-func self))
	      (execute 'release-func self args))))

(defhandler leave ((self button) &rest args 
		   &default :leave-window)
  (declare (ignore args))
  (when (pushed self)
	(setf (pushed self) nil)
	(setf (inverted self) nil)))

(defhandler enter ((self button) &rest args &aux pause flag
		   &default :enter-window)
  ;;	If button was pushed left and reentered
  (declare (ignore args))
  (when (not (flag self))
	(when (dimmed self)
	      (return-from button-enter))
	(setf (inverted self) t)
	(setq pause (pause-seconds self))
	(setq flag (flag self))
	(setf (pushed self) t
	      (flag self) nil)
	
	;;	Call press-func if necessary
	(if flag
	    (flush-display (display self)))))

(defmethod do-repaint ((self button))
  (call-next-method)
  (apply #'put (slot-value self 'value)))

(defmethod do-attach ((self button) &aux sval bsize bw bh)
  (setq bw (base-width self) bh (base-height self))
  (call-next-method)
  (if (default self) (setf (default-button #!po@self) self))
  (setq sval (slot-value self 'value))
  (when (gc-res self)
	(setf (getf (cdr sval) :gc) (gc-res self))
	(setf (slot-value self 'value) sval))
  (setq sval (car sval))
  (cond ((stringp sval)
	 (setq bsize
	       (list (+ (text-width sval :font (font self)) 20)
		     (+ (font-ascent (font self)) 10))))
	((image-p sval)
	 (setq bsize
	       (list (+ (width sval) 20) (+ (height sval) 10))))
	(t 
	 (setq bsize '(1 1))))
  (cond ((and (zerop bw) (zerop bh))
	 (setf (base-size self) bsize))
	((zerop bw) 
	 (setf (base-width self) (car bsize)))
	((zerop bh) 
	 (setf (base-height self) (cadr bsize)))))

(defmethod resize-window-handler ((self button) &aux sval)
  (setq sval (slot-value self 'value))
  (setf (getf (cdr sval) :width) (width self)
	(getf (cdr sval) :height) (height self))
  (setf (slot-value self 'value) sval))
