;;; -*- Mode: LISP; Package: PCL; Base: 10; Syntax: Common-lisp -*-
;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/store.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:37 $
;;;

(in-package 'pcl :use '(lisp libpq))

(defvar access-table (make-hash-table))
(defvar regexp-table (make-hash-table))

(defvar oid-table (make-hash-table))
(defvar object-table (make-hash-table))

(proclaim '(special access-table 
		    regexp-table 
		    oid-table 
		    object-table))

(defun reset ()
  (clrhash access-table)
  (clrhash regexp-table)
  (clrhash oid-table)
  (clrhash object-table))

(defmethod store-object ((self t))
  ;; return self
  self)

(defmethod object-id ((self t))
  ;; get object information
  (let* ((oid (gethash self object-table :failure)))
    ;; test if object has been stored
    (if (eq oid :failure)
	;; get class name and universal time
	(let ((cn (class-name (class-of self)))
	      (ut (get-universal-time)))
	  ;; generate object-id
	  (setq oid (read-from-string 
		     (symbol-name (gensym (format nil "~s.~x" cn ut)))))))
    ;; return oid
    oid))

(defmethod store-object ((self cons))
  ;; get cons information
  (let* ((oid (object-id self))
	 (regop (format nil "(let ((~s (cons" oid))
	 (setops nil)
	 (sval nil)
	 (soid nil))
    ;; insert self into access-table
    (setf (gethash self access-table) oid)
    ;; get car of cons
    (setq sval (car self))
    ;; get oid of car
    (setq soid (gethash soid access-table))
    ;; test if car has been seen
    (when (null soid)
	  ;; store sval
	  (store-object sval)
	  ;; get objectid of sval
	  (setq soid (gethash sval access-table)))
    ;; test if sval was accessed (i.e. a composite object)
    (when (not (null soid))
	  ;; push setf & load setop onto setops list
	  (push `(setf (car ,oid) (find-object ',soid)) setops)
	  (push `(load-object ',soid) setops)
	  ;; set value to nil
	  (setq sval nil))
    ;; add car to regop
    (setq regop (format nil "~a ~s" regop sval))
    ;; get cdr of cons
    (setq sval (cdr self))
    ;; get oid of cdr
    (setq soid (gethash soid access-table))
    ;; test if cdr has been seen
    (when (null soid)
	  ;; store sval
	  (store-object sval)
	  ;; get objectid of sval
	  (setq soid (gethash sval access-table)))
    ;; test if sval was accessed (i.e. a composite object)
    (when (not (null soid))
	  ;; push setf & load setop onto setops list
	  (push `(setf (cdr ,oid) (find-object ',soid)) setops)
	  (push `(load-object ',soid) setops)
	  ;; set value to nil
	  (setq sval nil))
    ;; add cdr to regop
    (setq regop (format nil "~a ~s" regop sval))
    ;; close let and append register expression
    (setq regop (format nil "~a))) (register-object '~s ~s)" regop oid oid))
    ;; test if setops is non-nil
    (if (not (null setops))
	;; append setops to regop
	(dolist (setop setops)
		;; append setop
		(setq regop (format nil "~a ~s" regop setop))))
    ;; close regop sexp
    (setq regop (format nil "~a ~s)" regop oid))
    ;; store regop
    #+libpq
    (let ((loid (symbol-name oid))
	  (pqop nil))
      (setq pqop (format
		  nil 
		  "delete local where local.loid = text[~s]" 
		  loid))
      (libpq::pqexec pqop)
      (setq pqop (format
		  nil 
		  "append local (loid = text[~s], make = text[~s])"
		  loid regop))
      (libpq::pqexec pqop))
    #-libpq
    (setf (gethash oid regexp-table) regop)
    ;; register object in memory
    (register-object oid self)
    ;; return oid
    oid))

(defmethod store-slots ((self object)
			&optional
			(slots nil slotsp))
  ;; test if slots are specified
  (if (null slotsp)
      ;; get slots from class
      (setq slots (class-instance-slots (class-of self)))
    ;; test slot list
    (if (not (listp slots))
	;; signal error
	(error "store-slots: invalid slot list ~s" slots)))
  ;; coerce list to all slot descriptors
  (do* ((i 0 (incf i))
	(s (nth i slots) (nth i slots))
	(d nil))
       ((= i (length slots)))
       ;; test if slot descriptor or slot name
       (when (symbolp s)
	     ;; get slot descriptor of this name
	     (setq d (find s (class-instance-slots (class-of self))
			   :key #'slotd-name))
	     ;; test if descriptor was found
	     (if (null d)
		 ;; signal error
		 (error "store-slots: can't find slotd named ~s" s))
	     ;; change name to slot descriptor
	     (setf (nth i slots) d)))
  ;; allocate locals for slot information
  (let ((sname nil)
	(svalue nil)
	(sclass nil)
	(soid nil))
    ;; store slot values
    (dolist (s slots)
	    ;; get slot name and value
	    (setq sname (slot-value s 'name))
	    (setq svalue (slot-value self sname))
	    ;; determine the class of the slot
	    (let ((stype (slot-value s 'type)))
	      ;; test if the slot's value is non-nil
	      (if (null svalue)
		  ;; attempt to determine the slot class
		  (if (listp stype)
		      ;; get least specific type
		      (setq sclass (class-named (car (last stype))))
		    ;; get class from type
		    (setq sclass (class-named (slot-value s 'type))))
		;; get class from value
		(setq sclass (class-of svalue))))
	    ;; test if svalue has been seen
	    (setq soid (gethash svalue access-table))
	    ;; test if svalue was accessed (i.e. not yet stored)
	    (when (null soid)
		  ;; store-object svalue
		  (store-object svalue)
		  ;; get objectid of svalue
		  (setq soid (gethash svalue access-table)))
	    ;; test if svalue was accessed (i.e. a composite object)
	    (when (not (null soid))
		  ;; push load setop onto setops list
		  (push `(setf (slot-value ,oid ',sname) 
			       (find-object ',soid)) setops)
		  (push `(load-object ',soid) setops)
		  ;; set svalue to nil
		  (setq svalue nil))
	    ;; add slot -name and -value to regop
	    (setq regop (format nil "~a :~s ~s" regop sname svalue)))))

(defmethod store-object ((self object))
  ;; get object information
  (let* ((class (class-of self))
	 (cname (class-name class))
	 (oid (object-id self))
	 (regop (format nil "(let ((~s (make-instance '~s" oid cname))
	 (setops nil))
    ;; make oid, regop and setops special
    (declare (special oid regop setops))
    ;; insert self into access-table
    (setf (gethash self access-table) oid)
    ;; store the slots
    (store-slots self)
    ;; close let and append register expression
    (setq regop (format nil "~a))) (register-object '~s ~s)" regop oid oid))
    ;; test if setops is non-nil
    (if (not (null setops))
	;; append setops to regop
	(dolist (setop setops)
		;; append setop
		(setq regop (format nil "~a ~s" regop setop))))
    ;; close regop sexp
    (setq regop (format nil "~a ~s)" regop oid))
    ;; store regop
    #+libpq
    (let ((loid (symbol-name oid))
	  (pqop nil))
      (setq pqop (format
		  nil 
		  "delete local where local.loid = text[~s]" 
		  loid))
      (libpq::pqexec pqop)
      (setq pqop (format
		  nil 
		  "append local (loid = text[~s], make = text[~s])"
		  loid regop))
      (libpq::pqexec pqop))
    #-libpq
    (setf (gethash oid regexp-table) regop)
    ;; register object in memory
    (register-object oid self)
    ;; return oid
    oid))

#+libpq
(defun load-object (oid)
  ;; allocate object
  (let ((object (gethash oid oid-table :failure)))
    ;; search for object in oid-table
    (if (eq object :failure)
	;; allocate portal
	(let* ((loid (symbol-name oid))
	       (pqop (format
		      nil
		      "retrieve (local.all) where local.loid = text[~s]"
		      loid))
	       (parray nil))
	  ;; exec pqop
	  (libpq::pqexec pqop)
	  ;; get portal array
	  (setq parray (libpq::pqparray ""))
	  ;; test for retrieve failure
	  (if (zerop (libpq::pqntuples parray))
	      ;; signal error
	      (error "load-object: can't find object ~s" oid))
	  ;; allocate regexp
	  (let* ((fnumber (libpq::pqfnumber parray 0 "make"))
		 (regexp (libpq::pqgetvalue parray 0 fnumber)))
	    ;; read regexp string
	    (setq regexp (read-from-string regexp))
	    ;; close portal
	    (libpq::pqfinish)
	    ;; evaluate regexp
	    (setq object (eval regexp)))))
    ;; return object
    object))

#-libpq
(defun load-object (oid)
  ;; allocate object
  (let ((object (gethash oid oid-table :failure)))
    ;; search for object in oid-table
    (if (eq object :failure)
	;; search for regexp in regexp-table
	(let ((regexp (gethash oid regexp-table :failure)))
	  ;; test for failure
	  (if (eq regexp :failure)
	      ;; signal error
	      (error "load-object: can't find object-id ~s" oid))
	  ;; evaluate regexp
	  (setq object (eval (read-from-string regexp)))))
    ;; return object
    object))

(defun find-object (oid &optional (default :failure))
  ;; allocate object
  (let ((object (gethash oid oid-table default)))
    ;; test for failure
    (if (eq object default)
	;; warn that object was not found
	(warn "find-object: object-id ~s not found" oid))
    ;; return object
    object))

(defun register-object (oid object)
  ;; test if object is already registered
  (when (eq (gethash oid oid-table :failure) :failure)
	;; put object in oid-table
	(setf (gethash oid oid-table) object)
	;; put oid in object-table
	(setf (gethash object object-table) oid))
  ;; return object
  object)

#-libpq
(defun find-regexp (oid &optional (default :failure))
  ;; allocate regexp
  (let ((regexp (gethash oid regexp-table default)))
    ;; test for failure
    (if (eq regexp :failure)
	;; warn that regexp was not found
	(warn "find-regexp: object-id ~s not found" oid))
    ;; return regexp
    regexp))

;;;
;;; Exports from PCL package
;;;
(export '(store-object
	  load-object
	  find-object
	  object-id
	  find-regexp
	  register-object)
	(find-package 'pcl))
