;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/slots.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:35 $
;;;

(in-package 'pcl :use '(lisp libpq))

;;;
;;; This file redifines functions and methods for slot accessing
;;;

;;;
;;; Whether the objid protocol is used
;;;
(defvar *objid* nil)

;;;
;;; Define the updating mode list and the default updating mode for dbobjects
;;; 
(defvar *dbmode-list* '(direct-update local-copy deferred-update))
(defvar *default-mode* 'local-copy)

(defun illegal-mode (dbmode)
  "Return T if dbmode is not one of the legal update modes"
  (or (null dbmode)
      (null (member dbmode *dbmode-list*))))

;;;
;;; Redefine make-instance which returns objid of the dbobject defined
;;; The update-mode of the instance created can be specified by the init-plist
;;; like ":dbmode 'direct-update"
;;;
(defun make-instance (class &rest init-plist)
  (when (symbolp class) (setq class (class-named class)))
  (let ((object (allocate-instance class)))
    (initialize object init-plist)
    ;; since we have not solved the problem with method invocation
    ;; we return the object instead of objid
    (if (and *objid* (eq (class-of class) (class-named 'dbclass)))
	;; the instance made is a dbobject - returns its objid
	(slot-value object 'objid)
      ;; the instance made is a standard object - returns the object
    object)))

;;;
;;; Redefine the initializing code for dbobject
;;; 
(defmethod initialize ((object dbobject) init-plist)
  ;; to do standard initialization of objects
  (call-next-method)
  ;; to do dbobject specific initialization
  (initialize-for-dbobject object init-plist))

;;;
;;; Dbobject specific initialization protocol
;;; 
(defmethod initialize-for-dbobject ((object dbobject) init-plist)
  (let ((dbmode (cadr (member ':dbmode init-plist))))
    (if (illegal-mode dbmode) (setq dbmode *default-mode*))
    ;; if a dbobject already has an objid (those in the database do)
    (cond ((slot-value object 'objid) 
	   ;; if objid protocol is not used, the slot 'objid actually
	   ;; stores the dbobject.  see the code in map.cl.
	   ;(if (null *objid*)
	   ;    (setf (slot-value object 'objid)
	;	     (slot-value (slot-value object 'objid) 'objid)))
	   ;; put it in the cache
	   ;; if we are using the objid protocol, updated will be recorded,
	   ;; so the initial modified flag is nil
	   (if *objid*
	       (put-dbobject-into-hashtable object t nil dbmode nil)
	     (put-dbobject-into-hashtable object t t dbmode nil)))
	  (t
	   ;; assign an objid to it (the newly created dbobject)
	   (setf (slot-value object 'objid) 
		 (make-objid (class-of object) dbmode))
	   ;; put the dbobject in the hashtable
	   (put-dbobject-into-hashtable object t t dbmode nil)
	   ;; if the dbmode is 'direct-update, store the instance into 
	   ;; the database
	   (if (eq dbmode 'direct-update)
	       (store-dbobject object))))))

;;;
;;; Make an objid for a dbclass
;;; This is a hack before we can preallocate a block of oid's
;;;

(defvar *prefix* (prin1-to-string (get-internal-run-time)))

(defmethod make-objid ((class dbclass) dbmode)
  (let ((class-name (class-name class))
	(relid (relid-cache class))
	;; generate a unique oid
	(fake-oid (symbol-name (gensym *prefix*))))
    (if (eq dbmode 'local-copy)
	;; local-copy dbobjects will not be stored in the database
	;; their oid is negative
	(make-instance 'objid 
		       :%id (list relid (- 0 (read-from-string fake-oid))))
      (let ((query1 (format nil "append ~A (objid = text[~S], halloween = text[\"0\"])"
			    (symbol-to-db class-name)
			    fake-oid))
	    (query2 (format nil "retrieve (~A.oid) where ~A.objid = text[~S]"
			    (symbol-to-db class-name)
			    (symbol-to-db class-name)
			    fake-oid)))
	(pqexec query1)
	(pqexec query2)
	(make-instance 'objid
		       :%id (list relid (read-from-string 
					 (pqgetvalue (pqparray "") 0 0))))))))

;;;
;;; This method returns the slot-value of a dbobject represented by the objid
;;;
(defmethod slot-value-using-class ((class objid-meta-class) object slot-name
				   &optional dont-call-slot-missing-p default)
  ;; if %id is accessed
  (if (eq slot-name '%id)
      ;; call the super method
      (call-next-method)
    ;; else we really want to access the dbobject represented by the objid
    (let* ((object (dbobject-cache object))
	   (class (class-of object)))
      ;; call the standard slot-accessing method to do the job
      ;; since the arguments are changed, we cannot use call-next-method
      (slot-value-using-class class object slot-name dont-call-slot-missing-p default))))

;;;
;;; This method setf the slot-value of a dbobject represented by the objid
;;;
(defmethod put-slot-using-class ((class objid-meta-class) object slot-name 
				 new-value &optional dont-call-slot-missing-p)
  ;; if %id is accessed
  (if (eq slot-name '%id)
      ;; call the super method
      (call-next-method)
    ;; else we really want to access the dbobject represented by the objid
    (let* ((object (dbobject-cache object))
	   (class (class-of object)))
      ;; call the standard slot-accessing method to do the job
      (put-slot-using-class class object slot-name new-value dont-call-slot-missing-p)))
  ;; unless relid or oid is accessed, we need to set flags or store objects
  (unless (eq slot-name '%id)
	  (progn
	    ;; change the modified flag to true
	    (setf (modified-cache object) t)
	    ;; if the updating mode is 'direct-update
	    (if (eq (mode-cache object) 'direct-update)
		;; update the dbobject in the database, 
		(store-dbobject object))))
  new-value)

;;;
;;; with-slots needs to be redifined too
;;;


;;;
;;; exports
;;;
(export '(make-instance
	  local-copy
	  deferred-update
	  direct-update)
	(find-package 'pcl))


