;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/map.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:33 $
;;;

(in-package 'pcl :use '(lisp libpq))

;;;
;;; this is a hack to get around the postgres limit on the lenght
;;; of class and attribute names.
;;;

#+longsymbol
(defvar *symbol-to-db* (make-hash-table :test #'eq))
#+longsymbol
(defvar *symbol-from-db* (make-hash-table :test #'eq))

#+longsymbol
(defun long-symbol-hack (symbol-list)
  (clrhash *symbol-to-db*)
  (clrhash *symbol-from-db*)
  (dolist (s symbol-list)
          (setf (gethash (car s) *symbol-to-db*) (cadr s))
	  (setf (gethash (cadr s) *symbol-from-db*) (car s))))

;;;
;;; POSTGRES does not allow the character '-' within a symbol, so we have to
;;; convert #\- in Commom Lisp symbol names to #\_ when storing to database
;;; and convert it back when fetching from database. 
;;; If you happen to use both '-' and '_', you LOSE.
;;;

#+longsymbol
(defun symbol-to-db (symbol)
  "Convert a Common Lisp \fBsymbol\fP to a POSTGRES string"
  (let ((dbsymbol (gethash symbol *symbol-to-db*)))
    (if (null dbsymbol)
	(setq dbsymbol symbol))
    (substitute #\_ #\- (string-downcase (symbol-name dbsymbol)))))
	 
#-longsymbol
(defun symbol-to-db (symbol)
  "Convert a Common Lisp \fBsymbol\fP to a POSTGRES string"
  (substitute #\_ #\- (string-downcase (symbol-name symbol))))
	 
#+longsymbol
(defun symbol-from-db (string)
  "Convert a POSTGRES \fBstring\fP back to a Common Lisp symbol"
  (let* ((dbsymbol (read-from-string (substitute #\- #\_ string)))
	 (symbol (gethash dbsymbol *symbol-from-db*)))
    (if symbol
	symbol
      dbsymbol)))

#-longsymbol
(defun symbol-from-db (string)
  "Convert a POSTGRES \fBstring\fP back to a Common Lisp symbol"
  (read-from-string (substitute #\- #\_ string)))

;;;
;;; The way SOH catalogs are stored depend on the POSTGRES catalog.
;;; For example, class named are stored in char16 at current time.
;;; We need functions to convert class names, etc. to the appropriate
;;; form to be stored in the database.  Consult the initialization
;;; database functions for the description of SOH catalogs.
;;;
(defun class-name-to-db (class-name)
  "Convert a class name to the format to be stored in the database"
  (symbol-to-db class-name))

;;;
;;; *OFTOPG* is a hashtable mapping CLOS types to POSTGRES types.
;;; Add an entry to this table for each type mapped to other than TEXT.
;;; In the future this table will be fetched from the database when the
;;; system is initialized.
;;;

(defvar *oftopg* (make-hash-table :test #'eq))

;(setf (gethash 'integer *oftopg*) "int4")

;;;
;;; All the slot types must be of either BUILT-IN-CLASS, CLASS, or DBCLASS.
;;; Otherwise it will be mapped to TEXT type.
;;; In PCL 5/22/87 release, only the following Common Lisp types have classes:
;;;
;;;  ARRAY BIT-VECTOR CHARACTER COMPLEX CONS FLOAT INTEGER LIST
;;;  NULL NUMBER RATIO RATIONAL SEQUENCE STRING SYMBOL T VECTOR
;;;

;;;
;;; OF-TYPE-TO-PG mapps an OBJFADS type (a class) to a POSTGRES type
;;;

(defmethod of-type-to-pg ((self built-in-class))
  "By default TEXT data type is used"
  (gethash (class-name self) *oftopg* "text"))

(defmethod of-type-to-pg ((self class))
  "For CLASS objects, a make function is stored in the TEXT form"
  (declare (ignore self))
  "text")

(defmethod of-type-to-pg ((self dbclass))
  "For DBCLASS objects (dbobjects), its objid is stored in the TEXT form"
  (declare (ignore self))
  "text")

(defmethod of-type-to-pg ((self objid-meta-class))
  "objid is stored in the TEXT form"
  (declare (ignore self))
  "text")

;;;
;;; OF-VALUE-TO-PG mapps an OBJFADS value to a POSTGRES value
;;;

;;;
;;; By default, the printed format is stored for plain values.
;;; For compiled functions, a mapping function is stored.
;;;
(defmethod of-value-to-pg (self)
  "Store compiled-function via function mapping.  
   By default store an OBJFADS value in its print form"
  (cond ((eq (type-of self) 'compiled-function)
	 (format nil "text[~S]"
		 (prin1-to-string (list 'fn-fetch-instance
					(list 'quote (fn-fetch-name self))))))
	(t
	 (format nil "text[~S]" (prin1-to-string self)))))


;(defmethod of-value-to-pg ((self string))
;  "Store string in POSTGRES as a string without the double quotes"
;  (format nil "text[~S]" self))

;(defmethod of-value-to-pg ((self integer))
;  "Integer is stored in the type int4 (4 byte integer)"
;  (prin1-to-string self))

(defmethod of-value-to-pg ((self object))
  "Store a make function for an object"
  (format nil "text[~S]" (build-make-function self)))

(defmethod of-value-to-pg ((self dbobject))
  "Store the objid for a dbobject"
  ;(store-dbobject self)
  ;; HACK
  (when (minusp (cadr (slot-value (slot-value self 'objid) '%id)))
        (setf (slot-value self 'objid)
              (make-objid (class-of self) 'deferred-update))
	(put-dbobject-into-hashtable self t t 'deferred-update nil))
  (format nil "text[~S]" 
	  (prin1-to-string  (slot-value (slot-value self 'objid) '%id))))

(defmethod of-value-to-pg ((self objid)) 
  "Store the objid"
  ;(store-dbobject self)
  (format nil "text[~S]" (prin1-to-string  (slot-value self '%id))))

(defmethod of-value-to-pg ((self cons))
  "Lists need special treatment"
  (format nil "text[~S]" (prin1-to-string (of-value-to-pg-internal self))))

(defmethod of-value-to-pg-internal ((self cons))
  (if (null self)
      nil
    (cons (of-value-to-pg-internal (car self))
	  (of-value-to-pg-internal (cdr self)))))

(defmethod of-value-to-pg-internal (self)
  self)

(defmethod of-value-to-pg-internal ((self dbobject))
  (when (minusp (cadr (slot-value (slot-value self 'objid) '%id)))
        (setf (slot-value self 'objid)
	      (make-objid (class-of self) 'deferred-update))
	(put-dbobject-into-hashtable self t t 'deferred-update nil))
  (slot-value (slot-value self 'objid) '%id))

;;;
;;; work to be done for of-value-to-pg:
;;; 1) we may need to check that the dbobject referenced is in DB
;;; this will be added later on
;;; 2) There is also a consistency problem here, what will happen
;;; if the dbobject is updated after its referee is stored to
;;; the database. This will be considered together with other 
;;; consistency problems.
;;; 3) for the time being, we assume objects do not referrence to
;;; dbobjects.
;;; 4) It might be a good idea to store a function 
;;; (dbobject-cache objid) instead of the objid
;;; Then if an object referrences to a dbobject, the make function
;;; will automatically get the dbobject.
;;;
#|
(defmethod build-make-function ((self object))
  "Build a make function for a CLOS object"
  (let* ((class (class-of self))
	 (instance-slots (class-instance-slots class))
	 (make-function ""))
    (when (not (null instance-slots))
	  (dolist (c instance-slots)
		  (setq make-function (format nil "~A :~A ~A"
					      make-function
					      (symbol-name (slotd-name c)) 
					      (build-make-function (slot-value self (slotd-name c)))))))
    (format nil "(make '~A ~A)" 
	    (symbol-name (class-name class))
	    make-function)))
|#

;;;
;;; the methods 'store is defined by D.C.Martin in store-object.cl
;;;
(defmethod build-make-function ((self object))
  (format nil "(load-object '~s)" (store-object self)))
 
(defmethod build-make-function (self)
  "By default build-make-function returns the value itself as a string"
  (format nil "~S" self))

(defmethod build-make-function ((self dbobject))
  "An Object is not allowed to reference dbobject for the current time"
  (error "An object which references dbobject(s) cannot be stored into the database"))

(defmethod build-make-function ((self objid))
  "An Object is not allowed to reference dbobject for the current time"
  (error "An object which references dbobject(s) cannot be stored into the database"))

;;;
;;; PG-VALUE-TO-OF converts a POSTGRES value to a certain type.
;;; If the type is not implemented as a class by PCL,
;;; (class-named slot-type) will signal an error.
;;;
;;; should write methods for this function
;;;
(defun pg-value-to-of (slot-type pg-value)
  (case (class-name (class-of (class-named slot-type)))
    ((built-in-class) (hack-value (read-from-string pg-value)))
    ((class) (hack-value (read-from-string pg-value)))
    ((objid-meta-class) (let* ((objid-list (read-from-string pg-value))
			       ;; if the objid already exists, return it
			       ;; otherwise make it
			       (objid (if (objid-exist-p objid-list)
					  (objid-cache objid-list)
					(make-instance
					 'objid :%id objid-list))))
			  ;; the only slot of this type is the objid slot
			  ;; return the objid
			  objid))
    ((dbclass) (let* ((objid-list (read-from-string pg-value))
		      ;; if the objid already exists, return it
		      ;; otherwise make it
		      (objid (if (objid-exist-p objid-list)
				 (objid-cache objid-list)
			       (make-instance 'objid :%id objid-list))))
		 ;; if objid protocol is used, return objid
		 ;; otherwise return the real object
		 (if *objid*
		     objid
		   (get-dbobject objid))))))

;;;
;;; Some forms need to be evaluated
;;;
(defun hack-value (value)
  (if (listp value)
      (if (or (equal (car value) 'make-instance)
	      (equal (car value) 'load-object)
	      (equal (car value) 'fn-fetch-instance))
	  (eval value)
	value)
    value))


;;;
;;; exports
;;;

#+longsymbol
(export '(*oftopg*
	  of-type-to-pg
	  of-value-to-pg
	  pg-value-to-of
	  build-make-function
	  *long-symbol-mappings*
	  long-symbol-hack)
	(find-package 'pcl))

#-longsymbol
(export '(*oftopg*
	  of-type-to-pg
	  of-value-to-pg
	  pg-value-to-of
	  build-make-function)
	(find-package 'pcl))

