;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/dbobject.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:23 $
;;;

(in-package 'pcl :use '(lisp libpq))

;;;
;;; STORE-DBOBJECT stores dbobjects into database: append if not already in, 
;;; otherwise replace only if modified.
;;;

(defmethod store-dbobject ((self dbobject))
  (store-dbobject (slot-value self 'objid)))

(defmethod store-dbobject ((self objid))
  ;; unless the dbobject is already in the database and not modified
  (unless (and (indb-cache self)
	       (not (modified-cache self)))
	  ;; flush the dbobject to the database
	  (flush-dbobject self)
	  ;; set the indb flag to be true
	  (setf (indb-cache self) t)
	  ;; if we are using the objid protocol, 
	  ;; set the modified flag to be false
	  (setf (modified-cache self) t))
  self)

(defmethod flush-dbobject ((self objid))
  (let* ((object (dbobject-cache self))
	 (class (class-of object))
	 (instance-slots (class-instance-slots class))
	 (indb-p (indb-cache self))
	 (query ""))
    ;; HACK
    ;; if the dbobject to be stored has mode 'local, and it is not
    ;; in the database yet, have to get objid for it.
    ;; we could also disallow local objects to be stored in database.
    (when (minusp (cadr (slot-value self '%id)))
	  (setf (slot-value object 'objid)
	        (make-objid (class-of object) 'deferred-update))
	  (put-dbobject-into-hashtable object t t 'deferred-update nil)
	  (setq indb-p nil))
    ;; HACK to get around the halloween problem
    ;; setf on object instead of self to avoid the recursive updating
    ;; since store-dbobject is called if mode is 'direct-update
    (setf (slot-value object 'halloween) 
	  (abs (1- (slot-value self 'halloween))))
    ;; build up the POSTGRES query 
    ;; instance-slots will never be NIL because each dbobject must 
    ;; have an objid
    (dolist (c instance-slots)
	    (setq query (format nil "~A~A = ~A, "
				query 
				(symbol-to-db (slotd-name c))
				;; convert OBJFADS values to POSTGRES
				(of-value-to-pg (slot-value object 
							    (slotd-name c))))))
    (setq query (format nil "~A ~A (~A) ~A"
			;; REPLACE if in database, otherwise append
			(if indb-p "replace" "append") 
			(symbol-to-db (class-name class))
			;; get rid of the extra ", " 
			(subseq query 0 (- (length query) 2))
			;; add a WHERE clause if REPLACE 
			(if indb-p 
			    (format nil "where ~A.oid = oid[~S] and ~A.halloween != ~A"
				    (symbol-to-db (class-name class))
				    (prin1-to-string (cadr (slot-value self '%id)))
    				    (symbol-to-db (class-name class))
				    (of-value-to-pg (slot-value self 'halloween)))
			  ;; nothing more is needed for APPEND
			  "")))
    (PQexec query)))

;;;
;;; GET-DBOBJECT returns the real dbobject given an objid:
;;; fetches it from the database if it is not already in the cache.
;;; 
(defmethod get-dbobject ((self objid) &optional no-error-p)
  "Returns the real dbobject given its objid"
  (or (dbobject-cache self t)
      (fetch-dbobject self no-error-p)))

;;;
;;; FETCH-DBOBJECT fetches a dbobject from the database.
;;;

(defmethod fetch-dbobject ((self objid) &optional no-error-p)
  "fetch a dbobject by objid"
  (let ((objid (slot-value self '%id)))
    (fetch-dbobject objid no-error-p)))

(defmethod fetch-dbobject ((self list) &optional no-error-p)
  ;; if the dbclass is not already defined in core, bring it in
  (let ((class (if (dbclass-cache (car self) t)
		   (class-named (dbclass-cache (car self)))
		 (defdbclass (car self))))
	(query (format nil "retrieve portal dbportal (x.all) from x in <~A>
                                     where x.oid = oid[~S]"
		       (car self)
		       (prin1-to-string (cadr self)))))
    (make-dbobject-from-database class query no-error-p)))

(defmethod fetch-dbobject ((self dbclass) slot-name slot-value &optional no-error-p)
  "fetch a dbobject by slot value"
  (let* ((class-name (class-name self))
	 (query (format nil "retrieve portal dbportal (~A.all) where ~A.~A=~A"
			(symbol-to-db class-name) 
			(symbol-to-db class-name)
			(symbol-to-db slot-name)
			(of-value-to-pg slot-value))))
    (make-dbobject-from-database self query no-error-p)))

;;; 
;;; MAKE-DBOBJECT-FROM-DATABASE makes an dbobject from the tuple fetched
;;; from the database.  The objid of this dbobject is returned.
;;;
(defmethod make-dbobject-from-database ((self dbclass) query &optional no-error-p)
  ;; query the instance relation for the dbobject
  (PQexec query)
  (pqexec "fetch all in dbportal")
  ;; if no qualified tuple returned
  (if (zerop (PQntuples (PQparray "dbportal"))) 
      ;; signal an error unless no-error-p is true
      (if no-error-p () (error "The dbobject is not in the database."))
    ;; otherwise build and evaluate the make-function to make the instance
    (let* ((portal-number (PQparray "dbportal"))
           (nfields (PQnfields portal-number 0))
           (class-name (class-name self))
	   (make-function (list 'make-instance (list 'quote class-name))))
      (dotimes (f nfields)
	       (setq make-function 
		     (append make-function 
			     (list (read-from-string (concatenate 'string  ":" (symbol-name (symbol-from-db (PQfname portal-number 0 f))))))
			     (list (list 'quote (pg-value-to-of 
						 (slot-type self (symbol-from-db (PQfname portal-number 0 f)))
						 (PQgetvalue portal-number 0 f)))))))
      (write make-function)
      (let ((new-object (eval make-function)))
	(ppi new-object)
	(pqexec "close dbportal")
	(setf (indb-cache new-object) t)
	(reinitialize new-object)))))

;;;
;;; SLOT-TYPE returns the slot-type given the class and slot-name.
;;; We need to know the type of each field to do the correct convertion
;;; of POSTGRES values from the backend.
;;; We are doing a sequential search of instance slots now.  
;;; There may be some better way to do it.
;;;
(defmethod slot-type ((class dbclass) slot-name)
  (do* ((instance-slots (class-instance-slots class) (cdr instance-slots))
	(slot-type nil))
    ((or slot-type (null instance-slots)) slot-type)
    (let ((c (car instance-slots)))
      (when (equal (slotd-name c) slot-name)
        (setq slot-type (slotd-type c))))))

;;;
;;; The reinitialize method
;;;
(defmethod reinitialize ((self dbobject))
  self)

;;;
;;; Pretty print an instance
;;;
(defmethod ppi ((self objid)
		&key
		(stream *standard-output*)
		(level nil))
  ;; let self be the dbobject represented by this objid
  (let ((self (dbobject-cache self)))
    ;; call the standard ppi method defined on object (not a standard PCL
    ;; method, but added by D.C.Martin)
    (ppi self)))

;;;
;;; exports
;;;
(export '(store-dbobject
	  get-dbobject
	  fetch-dbobject
	  reinitialize
	  ppi)
	(find-package 'pcl))
