;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/cache.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:17 $
;;;

(in-package 'pcl :use '(lisp libpq))

;;;
;;; Two caches are defined in this file:
;;; (1) a dbclass cache mapping between dbclasses and their relid's,
;;; (2) a cache which maps an objid to a dbobject.
;;;

;;;
;;; Define two hashtables which map between dbclass names (symbols) 
;;; and the POSTGRES oid's of the corresponding class instance 
;;; relations - relid's (integers).
;;;

(defvar *dbclass-cache-size* 1024)

(defun dbclass-cache-size ()
  *dbclass-cache-size*)

(defun setf-dbclass-cache-size (size)
  (setf *dbclass-cache-size* size))

(defsetf dbclass-cache-size setf-dbclass-cache-size)

(defvar *dbclass-cache* (make-hash-table :size *dbclass-cache-size* :test #'eq))

(defvar *relid-cache* (make-hash-table :size *dbclass-cache-size* :test #'eql))

(defmethod put-in-dbclass-cache ((self dbclass) relid)
  (setf (gethash (class-name self) *dbclass-cache*) relid)
  (setf (gethash relid *relid-cache*) (class-name self)))

(defmethod relid-cache ((self dbclass) &optional no-error-p)
  (or (gethash (class-name self) *dbclass-cache*)
      (if no-error-p nil (error "This dbclass is not in cache: ~S." self))))

(defun dbclass-cache (relid &optional no-error-p)
  (or (gethash relid *relid-cache*)
      (if no-error-p nil (error "This dbclass is not in cache: ~S." relid))))
  
;;;
;;; Define a hashtable which maps the print-form of an objid to a dbobject. 
;;; Also stored are four flags: indb, modified, mode (updating mode), pinned.
;;; Updating mode can be one of two values: 'local-copy or 'direct-update
;;;

(defvar *cache-size* 1024)

(defun cache-size ()
  *cache-size*)

(defun setf-cache-size (size)
  (setf *cache-size* size))

(defsetf cache-size setf-cache-size)

(defvar *cache-full* (/ *cache-size* 2)
  "hashtable is considered full if half of its entries are used")

(defvar *cache-count* 0
  "the number of entries in the hashtable")

(defvar *dbobject-cache* (make-hash-table :size *cache-size* :test #'equal))

;;;
;;; Define a method to put a dbobject in the hashtable 
;;;

(defmethod put-dbobject-into-hashtable ((self dbobject) indb modified mode pinned)
  ;; if hashtable is full, clean it by storing dbobjects to the database and
  ;; remove the entries from the hashtable.
  (when (> (incf *cache-count*) *cache-full*) 
	(clean-cache))
  (setf (gethash (slot-value (slot-value self 'objid) '%id) *dbobject-cache*) 
        (list self indb modified mode pinned)))

(defun clean-cache ()
  ;; removes all unmodified and unpinned entries
  (maphash #'(lambda (key val)
	       (when (and (cadr val) (not (caddr val)) (not (caddddr val)))
		     (remhash key *dbobject-cache*)
		     (decf *cache-count*)))
	   *dbobject-cache*)
  ;; if the hashtable is still full
  (when (> *cache-count* *cache-full*)
	;; store all dbobjects with the mode 'deferred-update and remove them
	;; from the hashtable
	(maphash #'(lambda (key val)
		     (when (and (eq (cadddr val) 'deferred-update) (not (caddddr val)))
			   (store-dbobject (car val))
			   (remhash key *dbobject-cache*)
			   (decf *cache-count*))))
	   *dbobject-cache*)
  ;; if the hashtable is still full (means the hashtable is too small)
  (when (> *cache-count* *cache-full*)
	;; force the hashtable size to be doubled
	(setq *cache-full* (* *cache-full* 2))))  

;;;
;;; Define methods that returns a dbobject or a flag given an objid.
;;;

(defmethod dbobject-and-flags-cache ((self objid) &optional no-error-p)
  (or (gethash (slot-value self '%id) *dbobject-cache*)
      (if no-error-p () (error "No dbobject has this id: ~S." self))))

(defmethod dbobject-cache ((self objid) &optional no-error-p)
  (first (or (gethash (slot-value self '%id) *dbobject-cache*)
	     (if no-error-p () (error "No dbobject has this id: ~S." self)))))

(defmethod indb-cache ((self objid) &optional no-error-p)
  (second (or (gethash (slot-value self '%id) *dbobject-cache*)
	      (if no-error-p () (error "No dbobject has this id: ~S." self)))))

(defmethod modified-cache ((self objid) &optional no-error-p)
  (third (or (gethash (slot-value self '%id) *dbobject-cache*)
             (if no-error-p () (error "No dbobject has this id: ~S." self)))))

(defmethod mode-cache ((self objid) &optional no-error-p)
  (fourth (or (gethash (slot-value self '%id) *dbobject-cache*)
	      (if no-error-p () (error "No dbobject has this id: ~S." self)))))

(defmethod pinned-cache ((self objid) &optional no-error-p)
  (fifth (or (gethash (slot-value self '%id) *dbobject-cache*)
	     (if no-error-p () (error "No dbobject has this id: ~S." self)))))

;;;
;;; Define methods that returns a flag given a dbobject.
;;; Error checking is redundant since all the dbobjects should be in the cache.
;;;

(defmethod indb-cache ((self dbobject) &optional no-error-p)
  (second (or (gethash (slot-value (slot-value self 'objid) '%id) 
		       *dbobject-cache*)
	      (if no-error-p () (error "Dbobject ~S not in cache." self)))))

(defmethod modified-cache ((self dbobject) &optional no-error-p)
  (third (or (gethash (slot-value (slot-value self 'objid) '%id) 
		      *dbobject-cache*)
	      (if no-error-p () (error "Dbobject ~S not in cache." self)))))

(defmethod mode-cache ((self dbobject) &optional no-error-p)
  (fourth (or (gethash (slot-value (slot-value self 'objid) '%id) 
		       *dbobject-cache*)
	      (if no-error-p () (error "Dbobject ~S not in cache." self)))))

(defmethod pinned-cache ((self dbobject) &optional no-error-p)
  (fifth (or (gethash (slot-value (slot-value self 'objid) '%id) 
		      *dbobject-cache*)
	      (if no-error-p () (error "Dbobject ~S not in cache." self)))))

;;;
;;; Define setf methods to modify flags given an objid.
;;;

(defmethod-setf indb-cache ((self objid)) (indb)
  (setf (second (or (gethash (slot-value self '%id) *dbobject-cache*)
		    (error "No dbobject has this id: ~S." self)))
	indb))

(defmethod-setf modified-cache ((self objid)) (modified)
  (setf (third (gethash (slot-value self '%id) *dbobject-cache* 
			;; This is a hack, since this method will be called 
			;; before an objid is assigned to a dbobject
			(list self t t t t))) modified)) 

(defmethod-setf mode-cache ((self objid)) (mode)
  (setf (fourth (or (gethash (slot-value self '%id) *dbobject-cache*)
		    (error "No dbobject has this id: ~S." self)))
	mode))

(defmethod-setf pinned-cache ((self objid)) (pinned)
  (setf (fifth (or (gethash (slot-value self '%id) *dbobject-cache*)
		   (error "No dbobject has this id: ~S." self)))
	pinned))

;;;
;;; Define setf methods to modify flags given a dbobject
;;;

(defmethod-setf indb-cache ((self dbobject)) (indb)
  (setf (second (gethash (slot-value (slot-value self 'objid) '%id) 
			     *dbobject-cache*)) 
	indb)) 

(defmethod-setf modified-cache ((self dbobject)) (modified)
  (setf (third (gethash (slot-value (slot-value self 'objid) '%id) 
			*dbobject-cache* 
			;; This is a hack, since this method will be called 
			;; before an objid is assigned
			(list self t t t))) 
	modified)) 
	
(defmethod-setf mode-cache ((self dbobject)) (mode)
  (setf (fourth (gethash (slot-value (slot-value self 'objid) '%id) 
			 *dbobject-cache*)) 
	mode))

(defmethod-setf pinned-cache ((self dbobject)) (pinned)
  (setf (fifth (gethash (slot-value (slot-value self 'objid) '%id) 
			*dbobject-cache*)) 
	pinned))

;;;
;;; miscellaneous routines
;;;

(defun objid-exist-p (objid)
  "Return T if objid (a list) is in *dbobject-cache*, NIL otherwise"
  (if (gethash objid *dbobject-cache*) T))

(defun objid-cache (objid)
  "Return the objid object given the objid (a list)"
  (slot-value (car (gethash objid *dbobject-cache*)) 'objid))

;;;
;;; these are the functions Larry want
;;;
(defmethod touch ((self objid))
  (setf (modified-cache self) t))

(defmethod touch ((self dbobject))
  (setf (modified-cache self) t))

;;;
;;; exports
;;;
(export '(dbclass-cache-size
	  cache-size
	  touch)
	(find-package 'pcl))







