;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/add-named-class.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 13:13:13 $
;;;

(in-package 'pcl :use '(lisp libpq))

;;;
;;; When we bring in the definition of a dbclass from the database, we need to
;;; call add-named-class to make an in-core definition.  However, we cannot use
;;; the method add-named-class defined for dbclass, because it will redefine
;;; this dbclass in the database.  Therefore, we store the function called by
;;; the method add-named-class defind for class in %add-named-class, and call
;;; it when we need to make an in-core definition for a dbclass brought from
;;; the database.  This is a HACK.
;;; 

(defun find-add-named-class-function ()
  "Return the function called by the method (add-named-class (proto class))"
  (let* ((generic-function (fboundp 'add-named-class))
       (method (car (slot-value generic-function 'methods))))
    (slot-value method 'function)))

#|
(defvar *loaded* nil)

(eval-when (compile load eval)
	   (when (null *loaded*)
		 (setf (symbol-function '%add-named-class)
		       (find-add-named-class-function))
		 (unintern 'add-named-class)
		 (setq *loaded* t)))
		 
(defmethod add-named-class ((proto class) name superclasses slots options)
  (%add-named-class proto name superclasses slots options))
|#

(defvar %add-named-class nil)
(setf %add-named-class (find-add-named-class-function))

;;;
;;; ADD-NAMED-CLASS is redefined here for DBCLASS.  It will store the
;;; definition of a dbclass into the database.
;;; For the time being, we do not consider redefinition of an existing
;;; dbclass. 
;;;
(defmethod ADD-NAMED-CLASS ((proto dbclass) name superclasses slots options)
  ;; first call the generic method add-named-class
  (let ((class (call-next-method)))
    ;; debugging
    (write class)
    ;; then add the class definition to the database
    (when class (add-class-to-database class superclasses slots options))
    class))

;;;
;;; ADD-CLASS-TO-DATABASE first checks if the class is already defined in the
;;; database.  It deletes it if it is true, otherwise does the following:
;;; 1) append the class definition to the relation DBCLASS
;;; 2) append the direct superclasses to the relation SUPERCLASSES
;;; 3) create an instance relation with the same name as the class
;;; 4) create a single tuple relation to store the class variables
;;;
;;; need to consider the order of queries to DB to make them cosistent.
;;; or use beginxact ... endxact facility.
;;;

;;;
;;; The POSTGRES queries in the file depend on the data types available and
;;; need to be consistent with POSTGRES catalog.  For example, dbclass is
;;; store as char16 right now because it is done so in pg_relation.
;;;

(defmethod add-class-to-database ((class dbclass) superclasses slots options)
  "Store the definition of \fBclass\fP into the database"
  (if (in-database-p class)
      (remove-class-from-database class))
  (add-class-definition-to-database class superclasses slots options)
  (add-superclasses-to-database class superclasses)
  (create-instance-relation class)
  (cache-relid class)
;  (create-class-slot-relation class)
)

(defmethod add-class-definition-to-database ((class dbclass) superclasses slots options)
  "Append the class definition of \fBclass\fP to the relation DBCLASS"
  (let ((query (format nil "append dbclass (name = ~S, owner = text[~S], superclasses = ~A, slots = ~A, options = ~A)"
		       (class-name-to-db (class-name class))
 		       (system:getenv "USER")
		       (of-value-to-pg superclasses)
		       (of-value-to-pg slots)
		       (of-value-to-pg options))))
    (pqexec query)))

(defmethod add-superclasses-to-database ((class dbclass) superclasses)
  "Append the direct superclasses (if any) to the relation SUPERCLASSES"
  (when (not (null superclasses))
	(dolist (s superclasses)
		(let ((query (format nil "append superclasses (class = ~S, superclass = ~S)"
				     (class-name-to-db (class-name class))
                                     (class-name-to-db s))))
		  (pqexec query)))))

(defmethod create-instance-relation ((class dbclass))
  "Create an instance relation for \fBclass\fP named (class-name \fBclass\fP)"
  (let ((instance-slots (class-instance-slots class))
	(query ""))
    (dolist (c instance-slots)
	    (setq query (format nil "~A~A = ~A, "
				query 
				(symbol-to-db (slotd-name c)) 
				;; convert the OBJFADS type to POSTGRES type
				;; Each type must have a class
				(of-type-to-pg (class-named (slotd-type c))))))
    (setq query (format nil "create ~A (~A)" 
			(symbol-to-db (class-name class))
			(subseq query 0 (- (length query) 2))))
    (pqexec query)))

#|
(defmethod create-class-slot-relation ((class dbclass))
  ;; if there are any class variables, create a relation to store them.
  ;; the relation is named class-name_cvars
  ;; if the class variables are inherited from a superclass, store a relid
  ;; of the superclass (we may even store a query there in the future)
  (let ((non-instance-slots (class-non-instance-slots class))
	(precedence-list (class-precedence-list class)))
    (when (not (null non-instance-slots))
	  ;; create a relation for non-instance-slots
	  (let ((query (concatenate 'string "create "
				    (symbol-to-db (class-name class)) 
				    "_cvars (")))
	    (dolist (c non-instance-slots)
		    (setq query (concatenate 'string query 
					   (symbol-to-db (slotd-name c))
					   " = " 
					   (of-type-to-pg (slotd-type c))
					   ", ")))
	    (setq query (concatenate 'string (subseq query 0 (- (length query) 2)) ")"))
	    (PQexec query))
	  ;; append the only tuple of this relation
	  ;; fill in the initform for local-class-slots
	  ;; fill in the super-relation name for non-local-class-slots
	  (let ((query (concatenate 'string "append "
))
|#

(defmethod cache-relid ((class dbclass))
  "Fetch relid of the instance relation for \fBclass\fP into *dbclass-cache*"
  (let* ((class-name (class-name class)) 
	 (query (format nil "retrieve (pg_relation.oid) where pg_relation.relname = ~S"
			(class-name-to-db class-name))))
    (pqexec query)
    (put-in-dbclass-cache class 
			  (read-from-string (pqgetvalue (pqparray "") 0 0)))))
  
;;;
;;; Query the database about a class. 
;;;
(defmethod in-database-p ((class dbclass))
  "Return T if \fBclass\fP is in the database"
  (let* ((class-name (class-name class))
	 (query (format nil "retrieve (pg_relation.relname) where pg_relation.relname = ~S"
			(class-name-to-db class-name))))
    (pqexec query)
    (if (plusp (pqntuples (pqparray "")))
	T)))

;;;
;;; Remove a class from the database
;;;
(defmethod remove-class-from-database ((class dbclass))
  "Remove the definition of \fBclass\fP from the database"
  (let* ((class-name (class-name class))
	 (delete-dbclass (format nil "delete dbclass where dbclass.name = ~S"
				 (class-name-to-db class-name)))
	 (delete-super (format nil "delete superclasses where superclasses.class = ~S"
			       (class-name-to-db class-name)))
	 (destroy-instance (format nil "destroy ~A"
				   (symbol-to-db class-name))))
    ;; need to add destroy class instance slot relation
    (pqexec delete-dbclass)
    (pqexec delete-super)
    (pqexec destroy-instance)))

;;;
;;; Get the definition of a dbclass from the database and define it in core.
;;; If any of its superclasses is not already in core, bring them in 
;;; recursively.
;;;

(defmethod defdbclass (class-name &optional no-error-p)
  "Define the dbclass \fBclass-name\fP from the definition in the database"
  (if (stringp class-name)
      (setq class-name (read-from-string class-name)))
  ;; if the class is already defined in memory, return it
  (if (class-named class-name)
      (class-named class-name)
    (let* ((query (format nil "retrieve (dbclass.all) where dbclass.name = ~S"
			  (class-name-to-db class-name)))
	   (class (defdbclass-internal query)))
      (if class
	  class
	(if no-error-p
	    nil
	  (error "Dbclass ~A is not in the database" class-name))))))

(defmethod defdbclass ((relid integer) &optional no-error-p)
  (let* ((query (format nil "retrieve (dbclass.all) where dbclass.name = pg_relation.relname and pg_relation.oid = oid[~S]"
			(prin1-to-string relid)))
	 (class (defdbclass-internal query)))
    (if class
	class
      (if no-error-p
	  nil
	(error "Dbclass with oid ~A is not in the database" relid)))))

(defun defdbclass-internal (query)
  (pqexec query)
  ;; if the dbclass does not exist, return nil
  (if (zerop (pqntuples (pqparray "")))
      nil
    ;; otherwise bring in the definitions from the database
    ;; and define it in core.
    (let* ((dbclass-proto (class-prototype (class-named 'dbclass)))
	   (parray (pqparray ""))
	   (class-name (symbol-from-db (pqgetvalue parray 0 (pqfnumber parray 0 "name"))))
	   (superclasses (portal-value parray "superclasses"))
	   (slots (portal-value parray "slots"))
	   (options (portal-value parray "options")))
      ;; if any of the superclasses is not already in core, bring it in
      (dolist (s superclasses)
	      (if (class-named s t)
		  nil
		(defdbclass s)))
      ;; actually define the dbclass in core
      (funcall %add-named-class 
	       dbclass-proto class-name superclasses slots options)
      ;; put this dbclass in the dbclass/relid cache
      (cache-relid (class-named class-name))
      (class-named class-name))))
    
(defun portal-value (parray field-name)
  "Get a field value from the portal \fBparray\fP"
  (read-from-string (pqgetvalue parray 0 (pqfnumber parray 0 field-name))))
      
;;;
;;; Get the definition of a dbclass from the database and define it in core,
;;; without checking for the existence of its superclasses.
;;; 
(defun defdbclass-self (class-name &optional no-error-p)
  "Define the dbclass \fBclass-name\fP from the definition in the database"
  (if (stringp class-name)
      (setq class-name (read-from-string class-name)))
  (let ((dbclass-proto (class-prototype (class-named 'dbclass)))
	(query (format nil "retrieve  (dbclass.all) 
			    where dbclass.name = ~S"
		       (class-name-to-db class-name))))
    (pqexec query)
    ;; if the dbclass exists, bring the definitions from the database
    ;; and define it in core.
    (if (plusp (pqntuples (pqparray "")))
	(let* ((parray (pqparray ""))
	       (superclasses (portal-value parray "superclasses"))
	       (slots (portal-value parray "slots"))
	       (options (portal-value parray "options")))
	  ;; if any of the superclasses is not already in core, bring it in
;	  (dolist (s superclasses)
;		  (if (class-named s t)
;		      nil
;		    (defdbclass s)))
	  ;; actually define the dbclass in core
	  (funcall %add-named-class 
		   dbclass-proto class-name superclasses slots options)
	  ;; put this dbclass in the dbclass/relid cache
	  (cache-relid (class-named class-name))
	  (class-named class-name))
      ;; otherwise return nil or signal error
      (if no-error-p
	  nil
	(error "Class ~A does not exist in the database" class-name)))))


;;;
;;; exports symbols
;;;
(export '(defdbclass
	  defdbclass-self)
	(find-package 'pcl))



