;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/tile.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 14:16:58 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; tile class
;;;

(defclass tile ()
  ((res
    :initform nil
    :type vector
    :reader res)
   (name
    :initarg :name 
    :initform "a tile"
    :type string
    :accessor name)
   (window
    :initarg :window 
    :initform nil
    :type t
    :reader window)
   (width 
    :initarg :width  
    :initform 0
    :type t
    :reader width)
   (height 
    :initarg :height  
    :initform 0
    :type t
    :reader height)
   (depth 
    :initarg :depth  
    :initform 0
    :type t
    :reader depth)
   (image
    :initform nil
    :type t)
   (foreground
    :initarg :foreground 
    :initform "black"
    :type t
    :reader foreground)
   (background
    :initarg :background 
    :initform "white"
    :type t
    :reader background)))

;;;	
;;;	Accessors. . .
;;;

(defmethod (setf foreground) (val (self tile) &aux color)
  (cond ((attached-p self)
	 (warn "tile.setf.foreground: can't set foreground of an attached tile"))
	((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.foreground: illegal color ~S~%" val)
	     (setf (slot-value self 'foreground) color)))
	((color-p val)
	 (setf (slot-value self 'foreground) color))))

(defmethod (setf background) (val (self tile) &aux color)
  (cond ((attached-p self)
	 (warn "tile.setf.background: can't set background of an attached tile")) 
	((stringp val) 
	 (setq color (get-color val))
	 (if (not (color-p color))
	     (warn "cursor.setf.background: illegal color ~S~%" val)
	     (setf (slot-value self 'background) color)))
	((color-p val)
	 (setf (slot-value self 'background) color))))

;;;
;;; 	When possible and needed, attaches self
;;;

(defmethod (setf window) (val (self tile) &key (warn-p t))
  (case (slot-value self 'window)
	(nil 
	 (cond ((window-p val) 
		(setf (slot-value self 'window) val)
		(when (attached-p val) (attach self)))
	       (t 
		(when warn-p 
		      (warn "tile.setf.window: illegal window \`~s\`." val)))))
	(val t)
	(t (cond ((attached-p self) 
		  (when warn-p 
			(warn "tile.setf.window: can't share attached tile.")))
		 (t 
		  (when 
		   warn-p 
;;		   (warn 
;;		    "tile.setf.window: sharing of tiles is not reccommended.")
		   )
		  (setf (slot-value self 'window) val)
		  (when (attached-p val) (attach self)))))))

(defun make-tile (&rest keys)
  (apply #'make-instance 'tile :allow-other-keys t keys))

;;;
;;; tile initialization method
;;;

(defmethod new-instance ((self tile)
			 &key
			 ;; window to associate tile
			 (window nil)
			 (dest nil)  ;; same as window
			 ;; generate tile from file
			 (file nil)
			 ;; generate tile from image
			 (image nil)
			 ;; generate tile from color
			 (color nil)
			 ;; generate tile from window
			 (source nil)
			 (x-offset 0)
			 (y-offset 0)
			 (width nil)
			 (height nil)
			 (depth nil)
			 (foreground "black")
			 (background "white")
			 ;; allow specification of slots
			 &allow-other-keys)
  (when window (setq dest window))
  (setf (foreground self) foreground)
  (setf (background self) background)
  ;; switch on keywords
  (cond (image nil)
	(file
	 (setq file (find-library-file file))
	 (setf (slot-value self 'image)
	       (setq image (xlib:read-bitmap-file file))))
	(color
	 (if (stringp color)
	     (let ((temp (get-color color)))
		  (when (null temp) 
			(warn "tile.new-instance: bad color ~S" color)
			(setq temp (get-color "white")))
		  (setq color temp)))
	 (do-attach color)
	 (setq image (xlib:create-pixmap :width 1 :height 1 
					 :depth (xlib:drawable-depth (res dest))
					 :drawable (res dest)))
	 (let ((gc (xlib:create-gcontext :drawable image
					 :foreground (pixel color))))
	      (xlib:draw-point image gc 0 0)
	      (xlib:free-gcontext gc)))
	(source
	 ;; test x-offset
	 (if (not (and (numberp x-offset)
		       (plusp x-offset)
		       (< x-offset (width source))))
	     ;; signal error
	     (error "tile.new-instance: invalid x-offset ~s" x-offset))
	 ;; test y-offset
	 (if (not (and (numberp y-offset)
		       (plusp y-offset)
		       (< y-offset (height source))))
	     ;; signal error
	     (error "tile.new-instance: invalid y-offset ~s" y-offset))
	 ;; test width
	 (if (not (and (numberp width)
		       (plusp width)
		       (< width (height source))))
	     ;; signal error
	     (error "tile.new-instance: invalid width ~s" width))
	 ;; test height
	 (if (not (and (numberp height)
		       (plusp height)
		       (< height (height source))))
	     ;; signal error
	     (error "tile.new-instance: invalid height ~s" height))
	 ;; test depth
	 (if (not (and (numberp depth)
		       (plusp depth)
		       (> depth (xlib:drawable-depth (res source)))))
	     (setq depth (xlib:drawable-depth (res source))))
	 ;; allocate resource id
	 (setf (slot-value self 'image)
	       (setq image (xlib:get-image (res source)
					   x-offset
					   y-offset
					   width
					   height
					   depth)))
	 ;; test for failure
	 (unless image
		 ;; signal error
		 (error "tile.new-instance: can't create tile?"))))

  ;; store resource id
  (when (and dest (attached-p dest))
	(if (not (typep image 'xlib:pixmap))
	    (let* ((fg (foreground self)) 
		   (bg (background self)) 
		   (gc nil))
		  (do-attach fg)
		  (do-attach bg)
		  (setq gc (xlib:create-gcontext :drawable (res dest)
						 :foreground (pixel fg)
						 :background (pixel bg)))
		  (setq image
			(xlib:image-pixmap
			 (res dest) (res image) :gcontext gc
			 :depth (xlib:drawable-depth (res dest))))
		  (xlib:free-gcontext gc)))
	(setf (slot-value self 'res) image))
  ;; return self
  self)

;;;
;;;	Attach method makes image into pixmap
;;;

(defmethod do-attach ((self tile) &aux dest image w h d res fg bg)
  (when (res self)
	(return-from do-attach))
  (unless (setq dest (window self))
	  (error "tile.attach: can't attach a tile without a destination"))
  (setq fg (foreground self)
	bg (background self))
  (do-attach fg)
  (do-attach bg)
  (setq fg (pixel fg))
  (setq bg (pixel bg))
  (setq image (slot-value self 'image))
  (if image 
      (progn
       (setq image (res image))
       (setq w (xlib:image-width image)
	     h (xlib:image-height image)
	     d (xlib:drawable-depth (res dest)))
       (let ((gc (xlib:create-gcontext :drawable (res dest)
				       :foreground fg
				       :background bg)))
	    (setq res (xlib:image-pixmap (res dest) image
					 :gcontext gc :width w
					 :height h :depth d))
	    (setf (slot-value self 'width) w
		  (slot-value self 'height) h
		  (slot-value self 'depth) d)
	    (xlib:free-gcontext gc)))
      (progn
       (setq w (width self)
	     h (height self)
	     d (depth self))
       (setq res (xlib:create-pixmap :width w :height h
				     :depth d :drawable (res dest)))))
  (setf (slot-value self 'res) res))

(defmethod do-detach ((self tile) &aux res fg bg)
  (setq fg (foreground self)
	bg (background self))
  (do-detach fg)
  (do-detach bg)
  (when (setq res (res self)) 
	(xlib:free-pixmap (res self)))
  (setf (slot-value self 'window) nil)
  (setf (slot-value self 'res) nil))
