;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/color.cl,v $
;;; $Revision: 1.3 $
;;; $Date: 90/07/25 16:59:50 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

(defclass color ()
  ((res
    :initform nil
    :type vector
    :reader res)
   (pixel
    :initarg :pixel
    :initform -1
    :type integer
    :reader pixel)
   (name
    :initarg :name
    :initform ""
    :type string
    :reader name)
   (colormap
    :initarg :colormap
    :initform nil
    :type colormap
    :reader colormap)
   (red 
    :initarg :red
    :initform 0 
    :type integer
    :reader red)
   (green 
    :initarg :green
    :initform 0 
    :type integer
    :reader green)
   (blue 
    :initarg :blue
    :initform 0 
    :type integer
    :reader blue)
   (ref-count
    :initarg :ref-count
    :initform 0
    :type integer)))

(defun make-color (&rest args
			 &key
			 (name nil namep)
			 (colormap nil mapp)
			 (attach-p nil)
			 red green blue
			 &allow-other-keys 
			 &aux color)
  (unless (and mapp (colormap-p colormap))
	  (setq colormap (default-colormap)))
  (setf (getf args :colormap) colormap)
  (if (and 
       (setq color (get-color name colormap))
       (not (or red green blue))) 
      (progn 
       (when attach-p (do-attach color))
       color)
      (apply #'make-instance 'color :allow-other-keys t args)))

(defmethod new-instance ((self color)
			 &key
			 (name nil namep)
			 (colormap nil mapp)
			 (red 0 rp)
			 (green 0 gp)
			 (blue 0 bp)
			 (attach-p nil) ;; attach right now
			 (lookup-p t) ;; indicates that it should be looked-up
			 (pixel 0 pixelp) ;; only for advanced x-users
			 (hardware-color-p nil) ;; use hardware-color
			 &allow-other-keys 
			 &aux sc hc)
  (unless (and mapp (colormap-p colormap))
	  (setf (slot-value self 'colormap)
		(setq colormap (default-colormap))))
  (when (or rp gp bp) 
	(setq lookup-p nil))
  (if pixelp 
      (setf (slot-value self 'pixel) pixel)
      (progn
       (cond (lookup-p
	      (when (and namep (stringp name)) 
		    (multiple-value-setq 
		     (sc hc) 
		     (xlib:lookup-color (res colormap) name)) 
		    (if hardware-color-p 
			(setf (slot-value self 'res) sc
			      (slot-value self 'red) (xlib:color-red sc)
			      (slot-value self 'green) (xlib:color-green sc)
			      (slot-value self 'blue) (xlib:color-blue sc) 
			      (gethash name (hashtab colormap)) self)
			(setf (slot-value self 'res) hc
			      (slot-value self 'red) (xlib:color-red hc)
			      (slot-value self 'green) (xlib:color-green hc)
			      (slot-value self 'blue) (xlib:color-blue hc)
			      (gethash name (hashtab colormap)) self))))
	     ((and (<= 0 red 1) (<= 0 green 1) (<= 0 blue 1))
	      (setq hc (xlib:make-color :blue blue :green green :red red)) 
	      (setf (slot-value self 'res) hc
		    (slot-value self 'red) (xlib:color-red hc)
		    (slot-value self 'green) (xlib:color-green hc)
		    (slot-value self 'blue) (xlib:color-blue hc))
	      (when name
		    (if (get-color name colormap)
			(warn "color.new-instance: name \'~s\' already exists in colormap. Aborted." name)
			(setf (gethash name (hashtab colormap)) self))))
	     (t
	      (warn "color.new-instance: bad rgb values (~s ~s ~s)" 
		    red green blue)))))
  
  ;; attach when desired
  (when attach-p
	(setf (slot-value self 'pixel)
	      (xlib:alloc-color (res colormap) (res self))
	      (slot-value self 'ref-count) 1))

  ;; return self
  self)

;;;
;;;	Do not normally detach colors
;;;

(defmethod do-detach ((self color))
  ;; don't detach if ref-count is 0
  (when (> (slot-value self 'ref-count) 1) 
	(decf (slot-value self 'ref-count))
	(return-from do-detach))
  (let ((p (pixel self)))
       (cond ((< p 0)
	      (warn "Color ~S already detached" self))
	     (t
	      (xlib:free-colors (res (colormap self)) (list (pixel self)))
	      (setf (slot-value self 'pixel) -1
		    (slot-value self 'ref-count) 0)))))

(defmethod do-attach ((self color))
  ;; don't reattach if ref-count is positive
  (when (> (slot-value self 'ref-count) 0)
	(incf (slot-value self 'ref-count))
	(return-from do-attach))
  ;; check colormap
  (unless (colormap self)
	  (error "Can't attach color ~S with no colormap" self))

  ;; attach. . .
  (setf (slot-value self 'pixel)
	(xlib:alloc-color (res (colormap self)) (res self))
	(slot-value self 'ref-count) 1))

(defmacro color-attach (self)
  `(do-attach ,self))

(defmacro color-detach (self)
  `(do-detach ,self))
