;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/propagator.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 20:20:27 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

(defun dumphash (tab)
  (maphash #'(lambda (k v) (format t "~S:  ~S~%" k v)) tab))

(defvar *prop-table* (make-hash-table :test #'equal))
(defvar *func-table* (make-hash-table :test #'equal))
(defvar *meth-table* (make-hash-table :test #'equal))
(defvar *unbind-table* (make-hash-table :test #'equal))

;;; Update, expressions are expected to be without any remaining local
;;; variable references.  To accomplish this, they should generally
;;; be created in backquote form.  for instance, 
;;;
;;; (bind-slot 'sname win1 `(concatenate 'string (var name ,win2)))
;;;

(defmacro bind-var (first second &rest others)
  (let* ((third (car others))
	 (thirdp (and third (not (eql third :receipt))))
	 (receipt (cadr (member :receipt others))))
	(if thirdp
	    `(bind-var-internal ,first ,second ,third :receipt ,receipt)
	    (let ((name nil)
		  (reference nil)
		  (expression third))
		 (if (valid-var-expression first)
		     (progn
		      (if (eql (car first) 'value) (setq first (cadr first)))
		      `(bind-var-internal ,(cadr first) 
					  ,(or (caddr first) 
					       '(lexical-environment))
					  ,second :receipt ,receipt))
		     `(warn "Bind-var:  Invalid arguments"))))))

(defun bind-var-internal (name reference expression &key (receipt nil))
  (bind-slot 'value (find-var name reference) (var-adjust expression reference) 
	     :receipt receipt))

(defun var-adjust (expression reference)
  (cond ((atom expression) expression)
	((var-p expression) (append expression (list :ref reference)))
	(t (cons (var-adjust (car expression) reference)
		 (var-adjust (cdr expression) reference)))))

(defun bind-slot (slot object expression &key (receipt nil))
  (let* ((dependees (get-dependees expression object))
	 (function (po-get-function expression object))
	 (ftab-entry (cons function (mapcar #'cdr dependees)))
	 (key `(,slot ',object)))
	
	;; create appropriate methods for virtual slots
	(unless (slot-exists-p object slot)
		(format t "Debug: binding virtual slot (~s ~s)~%" slot object)
		(eval `(defmethod ,slot ((self (eql ',object)))
				  (if (next-method-p) (call-next-method)
				      ,function)))
		;;; (compile-methods-named slot)
		(eval `(defmethod (setf ,slot) (value (self (eql ',object)))
				  (declare (ignore value))
				  (if (next-method-p) (call-next-method))))
		;;; (compile-setf-methods-named slot)
		)
	
	;; process the dependee list
	(when (null dependees)
	      (unbind-from-deplist-internal key nil)
	      (unless (null function) 
		      (warn "Bind-slot should not be used with
			    constants")))
	
	(dolist (d dependees)
		(let ((new-d (cdr d)))
		     
		     ;; add the location to the hash table
		     
		     (pushnew key (gethash new-d *prop-table*) :test #'equal)
		     
		     ;; handle checking for methods
		     (if (= (incf (gethash new-d *meth-table* 0)) 1)
			 (case (car d)
			       (slot 
				(eval 
				 `(defmethod (setf ,(car new-d)) :around
					     (value (self (eql ,(cadr new-d))))
					     (unless (equal (slot-value self ',(car new-d))
							    value)
						     (call-next-method)
						     (propagate (gethash ',new-d *prop-table*)
								',new-d))))
				;;; (compile-setf-methods-named (car new-d))
				)
			       (vslot
				(eval
				 `(defmethod (setf ,(car new-d)) :around
					     (value (self (eql ,(cadr new-d))))
					     (declare (ignore value))
					     (if (next-method-p) (call-next-method))
					     (propagate (gethash ',new-d *prop-table*)
							',new-d)))
				;;; (compile-setf-methods-named (car new-d))
				)))
		     
		     ;;;  Now, let's deal with the actual function
		     (let* ((fkey (cons key new-d))
			    (func-list (gethash fkey *func-table*))
			    (unbind-list 
			     (mapcan #'(lambda (func) 
					       (cond 
						((replaces (cdr func) 
							   (mapcar #'cdr
								   dependees))
						 (list (cons key func)))))
				     func-list)))
			   (pushnew ftab-entry (gethash fkey *func-table*) 
				    :test #'equal)
			   (setq unbind-list (delete (cons key ftab-entry) 
						     unbind-list 
						     :test #'equal))
			   (mapc #'unbind-internal unbind-list)
			   (when receipt 
				 (setq receipt (gensym))
				 (setf (gethash receipt *unbind-table*) 
				       (cons key ftab-entry))))))
	(eval `(setf (,slot ',object) ,function)))
  receipt)

(defun unbind-internal (receipt)
  (let* ((dest (car receipt))
	 (pair (cdr receipt))
	 (places (cdr pair))
	 (loc (cons dest nil)))
	(dolist (place places)
		(rplacd loc place)
		(setf (gethash loc *func-table*)
		      (delete pair (gethash loc *func-table*) :test #'equal))
		(unless (gethash loc *func-table*)
			(setf (gethash place *prop-table*)
			      (delete dest (gethash place *prop-table*) 
				      :test #'equal))
			(unless (gethash place *prop-table*)
				(remhash place *prop-table*))
			(remhash loc *func-table*)))))

(defun unbind-fast (receipt)
  (unbind-internal (gethash receipt *unbind-table*))
  (remhash receipt *unbind-table*))

(defun unbind-from-deplist-internal (dest source-list
					&key
					(unbind-supersets t)
					(unbind-subsets t))
  (prog (unbind-receipts)
	(maphash #'(lambda (key value)
		     (if (equal (car key) dest)
			 (dolist (f value)
			     (let ((sub (subsetp (cdr f) source-list
						 :test #'equal))
				   (sup (subsetp source-list (cdr f)
						 :test #'equal)))
				  (when 
				   (or (and unbind-subsets sub)
				       (and unbind-supersets sup)
				       (and sub sup))
				   (pushnew (cons dest f) unbind-receipts))))))
		 *func-table*)
	(mapc #'unbind-internal unbind-receipts)))

(defun unbind-var (name reference expr &key
			(unbind-supersets t)
			(unbind-subsets t))
  (unbind-slot 'value (find-var name reference)
	       (var-adjust expr reference)
	       :unbind-supersets unbind-supersets
	       :unbind-subsets unbind-subsets))

(defun unbind-slot (slot object expr &key
			 (unbind-supersets t)
			 (unbind-subsets t))
  ;;  takes a slot name, object, and any expression (or list) containing
  ;;  var expressions.  The effect is to remove ANY bindings for the slot of
  ;;  the specified object which have the same set of dependees, and, 
  ;;  unless specified otherwise, remove bindings for proper supersets and 
  ;;  subsets.  The expression must be in "bind" form (var specifications).
  (unbind-from-deplist-internal `(,slot ',object) 
			      (mapcar #'cdr (get-dependees expr object))
			      :unbind-supersets unbind-supersets
			      :unbind-subsets unbind-subsets))

(defun propagate (dlist from)
  (dolist (d dlist)
	  (eval `(setf ,d ,(caar (gethash (cons d from) *func-table*))))))

;;; To be called as (do-propagate 'title foo-obj) or (do-propagate #?var)

(defun do-propagate (var-slot &optional (object nil object-p)
			      &aux desc)
  (unless object-p
    (setq object var-slot var-slot 'value))
  (setq desc `(,var-slot ',object))
  (propagate (gethash desc *prop-table*) desc))

(defun get-dependees (expression reference)
  (cond ((atom expression) nil)
	((slot-p expression)
	 (if (real-slot-p expression)
	     (if (variable-p (second expression))
		 `((var ,(second expression) ',(third expression)))
		 `((slot ,(second expression) ',(third expression))))
	     `((vslot ,(second expression) ',(third expression)))))
	((var-ex-p expression)
	 (list (process-var-ex (cadr expression) reference)))
	((var-ex-with-ref-p expression)
	 (list (process-var-ex (cadr expression) (fourth expression))))
	((var-p expression)
	 (list (process-var expression reference)))
	((non-local-var-p expression)
	 (list (process-non-local-var expression)))
	(t (append (get-dependees (car expression) reference)
		   (get-dependees (cdr expression) reference)))))


(defun slot-p (expr)
  (and (eql (car expr) 'var)
       (= (length expr) 3)))

(defun real-slot-p (expr)
  (slot-exists-p (third expr) (second expr)))

(defun var-ex-p (expr)
  (and (eql (car expr) 'var)
       (= (length expr) 2)
       (valid-var-expression (cadr expr))))

(defun var-ex-with-ref-p (expr)
  (and (eql (car expr) 'var)
       (= (length expr) 4)
       (valid-var-expression (cadr expr))))

(defun var-p (expr)
  (and (eql (car expr) 'var)
       (= (length expr) 2)))

(defun non-local-var-p (expr)
  (and (= (length expr) 4)
       (eql (car expr) 'var)
       (eql (third expr) :ref)))

(defun process-var-ex (expr ref)
  (if (eql (car expr) 'value) (setq expr (cadr expr)))
  (cond ((= (length expr) 2)
	 `(var value ',(find-var (eval (cadr expr)) ref)))
	((= (length expr) 3)
	 `(var value ',(eval expr)))
	(t (error "process-var-ex: unresolvable expression ~s" expr))))

(defun process-var (expr ref)
  `(var value ',(find-var (second expr) ref)))

(defun process-non-local-var (expr)
  `(var value ',(find-var (second expr) (fourth expr))))

(defun po-get-function (expr ref)
  (cond ((atom expr) expr)
	((slot-p expr) `(,(second expr) ',(third expr)))
	((var-ex-p expr) (cdr (process-var-ex (cadr expr) ref)))
	((var-ex-with-ref-p expr) 
	    (cdr (process-var-ex (cadr expr) (fourth expr))))
	((var-p expr) (cdr (process-var expr ref)))
	((non-local-var-p expr) (cdr (process-non-local-var expr)))
	(t (cons (po-get-function (car expr) ref)
		 (po-get-function (cdr expr) ref)))))


;;; (bind-function 'a f2 '(+ (slot a f1) (slot b f1)))

(defun replaces (deplist1 deplist2)
  (or (subsetp deplist1 deplist2 :test #'equal)
      (subsetp deplist2 deplist1 :test #'equal)))

(defun dump-prop ()
  (format t "~%METHOD TABLE~%")
  (dumphash *meth-table*)
  (format t "~%UNDO TABLE~%")
  (dumphash *unbind-table*)
  (format t "~%PROP TABLE~%")
  (dumphash *prop-table*)
  (format t "~%FUNC TABLE~%")
  (dumphash *func-table*)
  t)

(defun clear-prop ()
  (clrhash *unbind-table*)
  (clrhash *prop-table*)
  (clrhash *func-table*))

(defvar *dummy-prop-var* (make-variable :value nil))

(defmacro set-trigger (s-or-v o-or-f &optional (func nil func-p))
   (if func-p
       `(bind-slot 'value *dummy-prop-var*
          '(progn (var ,(eval s-or-v) ,(eval o-or-f)) ,(eval func)))
       `(bind-slot 'value *dummy-prop-var*
          '(progn (var ,s-or-v :ref ,(lexical-environment)) ,(eval o-or-f)))))

(defun bind-helper (let-clause)
    (if (valid-var-expression (second let-clause))
	`(list ',(car let-clause)
	       (list 'var ',(second let-clause)))
	`(list ',(car let-clause)
	       (list 'var ',(car (second let-clause)) 
			  ,(cadr (second let-clause))))))

(defun bind-helper-2 (let-clause)
  `(list ',(car let-clause) ,(second let-clause)))

(defun repair-damage (form)
  (cond ((atom form) form)
	((eql (car form) 'let)
		  (cons (car form) 
			(cons (repair-let-clauses (cadr form))
			      (cddr form))))
	((cons (repair-damage (car form)) (repair-damage (cdr form))))))

(defun repair-let-clauses (list)
  (mapcar #'repair-one-let-clause list))

(defun repair-one-let-clause (cl)
  (if (and (listp (cadr cl)) (eql (caadr cl) 'var))
      cl
      `(,(car cl) ',(cadr cl))))

        
  

(defmacro blet (what &rest stuff &aux v w bool form)
  (setq v (mapcar #'bind-helper (cadr (member :var stuff))))
  (setq w (mapcar #'bind-helper-2 (cadr (member :with stuff))))
  (setq bool nil)
  (dolist (s stuff)
	  (if bool
	      (setq bool nil)
	      (if (member s '(:var :with))
		  (setq bool t)
		  (setq form (cons (list 'quote s) form)))))
  (setq form `(list 'quote 
		    (list 'let (list ,@(append v w)) ,@(nreverse form))))

  (if (valid-var-expression what)
      (setq form  `(list 'bind-var ',what ,form))
      (setq form  `(list 'bind-slot '',(car what) ',(cadr what) ,form)))
  (repair-damage (eval form)))
  
(defmacro bind (this that)
  `(blet ,this
	 :var ((x ,that))
	 x))
   
  
