;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/lazy-eval.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 20:19:40 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;  This file provides functions to support lazy evaluation slots on a
;;;  class-wide basis.  The implementation of these slots is simple:
;;;
;;;  1.  The setf method for this slot does not actually do any setting, 
;;;      instead it merely places the keyword :invalid in the slot.  
;;;
;;;  2.  A :before method is created to support checking the cache (slot)
;;;      and bringing it up to date as needed.

(defvar *lazy-table* (make-hash-table :test #'equal))

(defmacro invalid-p (thing)
  `(eq ,thing :invalid))

(defmacro lazy-slot-p (slot class-name)
  `(member ,slot (get ,class-name :lazy)))

(defmacro unlazy-slot-p (slot class-name)
  `(member ,slot (get ,class-name :unlazy)))

(defun lazy-p-internal (slot class)
  (let ((classes (pcl::class-class-precedence-list class)))
       (dolist (cl classes)
	       (if (lazy-slot-p slot (class-name cl))
		   (return-from lazy-p-internal t))
	       (if (unlazy-slot-p slot (class-name cl))
		   (return-from lazy-p-internal nil))))
  nil)

(defun lazy-instance-p (slot object)
  (gethash (cons slot object) *lazy-table*))

(defun lazy-p (slot object)
  (or (lazy-instance-p slot object)
      (lazy-p-internal slot (class-of object))))

(defun make-slot-lazy-for-class (slot class-name before-expression)
  (eval
   `(defmethod (setf ,slot) (value (self ,class-name))
	       (declare (ignore value))
	       (setf (slot-value self ',slot) :invalid)))
  ;;; (compile-setf-methods-named slot)
  (eval 
   `(defmethod ,slot :before ((self ,class-name))
	       (if (invalid-p (slot-value self ',slot))
		   (setf (slot-value self ',slot)
			 ,before-expression))))
  ;;; (compile-setf-methods-named slot)
  (pushnew slot (get class-name :lazy)))

(defun make-slot-unlazy-for-subclass (slot class-name)
  (unless (lazy-p-internal slot (find-class class-name))
	  (error "Slot ~S in Class Named ~S is not lazy" slot class-name))
  (eval `(defmethod (setf ,slot) (value (self ,class-name))
		    (setf (slot-value self ',slot) value)))
  ;;; (compile-setf-methods-named slot)
  (pushnew slot (get class-name :unlazy)))


(defun make-slot-lazy-for-instance (slot object before-expression)
  (eval
   `(defmethod (setf ,slot) (value (self (eql ',object)))
	       (declare (ignore value))
	       (setf (slot-value self ',slot) :invalid)))
  ;;; (compile-setf-methods-named slot)
  (eval
   `(defmethod ,slot :before ((self (eql ',object)))
	       (if (invalid-p (slot-value self ',slot))
		   (setf (slot-value self ',slot)
			 ,before-expression))))
  ;;; (compile-setf-methods-named slot)
  (setf (gethash (cons slot object) *lazy-table*) t))
