;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/macros.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 13:56:21 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; determine if an object is a funcallable object
;;;

(defmacro function-p (value)
  "determine if the specified argument is either a function or bound to a 
function"
  `(if (symbolp ,value)
      (fboundp ,value)
    (functionp ,value)))

;;;
;;; current display and other objects
;;;

;;
;; global display variables
;;

(defvar *current-display* nil)
(defvar *current-root* nil)
(defvar *current-screen* nil)
(defvar *active-displays* nil)
(defvar *display-table* (make-hash-table :size 10 :rehash-size 5 :test #'eq))

(defmacro current-display ()
  "Return the current display instance or nil if no display"
  `*current-display*)

(defmacro default-display ()
  `(current-display))

(defmacro current-screen (&optional display)
  "Return the current screen instance or nil if no screen"
  `(cond ((null ,display) *current-screen*)
	 ((display-p ,display)
	  (primary-screen ,display))
	 (,display
	  (error "current-screen: invalid display \`~s\`." ,display))
	 (t *current-screen*)))

(defmacro default-screen ()
  `(current-screen))

(defmacro root-window (&optional screen)
  "Return the current root-window or nil if no root-window"
  `(cond ((null ,screen) *current-root*)
	 ((screen-p ,screen)
	  (root ,screen))
	 ((display-p ,screen)
	  (root (primary-screen ,screen)))
	 (t (error "root-window: invalid source \`~s\`." ,screen))))

(defmacro current-root ()
  `(root-window))

(defmacro default-root ()
  `(root-window))

(defmacro current-colormap (&optional source)
  `(cond ((null ,source) 
	  (when *current-root*
		(colormap *current-root*)))
	 ((screen-p ,source)
	  (colormap (root ,source)))
	 ((display-p ,source)
	  (colormap (root (primary-screen ,source))))
	 ((window-p ,source)
	  (colormap ,source))
	 (t (error "current-colormap: invalid source \`~s\`." ,source))))

(defmacro default-colormap (&optional source)
  `(current-colormap ,source))

;;;
;;; resource macros
;;;

(defconstant *resource-types* '(color image tile font image))
(defmacro resource-p (obj)
  `(member (class-name (class-of ,obj)) *resource-types*))

(defconstant *paint-types* '(color image tile))
(defmacro paint-p (obj)
  `(member (class-name (class-of ,obj)) *paint-types*))


(defmacro current-font-table (&optional display)
  `(if ,display
       (font-table ,display)
       (font-table (current-display))))

(defmacro font-name-p (name &optional display)
  `(not (null
	 (if ,display
	     (xlib:list-font-names (res ,display) ,name)
	     (xlib:list-font-names (res (current-display)) ,name)))))

;;
;; look-up resource macros
;;

(defmacro find-screen (num &optional display)
  `(if ,display
       (get-screen ,num ,display)
       (get-screen ,num)))

(defmacro find-font (name &optional display)
  `(if ,display
       (get-font ,name ,display)
       (get-font ,name)))

(defmacro find-color (name &optional colormap)
  `(if ,colormap
       (get-color ,name ,colormap)
       (get-color ,name)))

(defmacro find-paint (name &optional display)
  `(if ,display
       (get-paint ,name ,display)
       (get-paint ,name)))

(defmacro find-image (name)
  `(get-image ,name))

;;
;; default cached white & black pixels for optimal clx routines
;;

(defmacro white-pixel (&optional display)
  `(cond ((null ,display)
	  (xlib:screen-white-pixel (res (current-screen))))
	 ((display-p ,display)
	  (xlib:screen-white-pixel 
	   (xlib:display-default-screen (res ,display))))
	 ((screen-p ,display)
	  (xlib:screen-white-pixel (res ,display)))
	 (t (error "white-pixel: invalid argument \`~s\`." ,display))))

(defmacro black-pixel (&optional display)
  `(cond ((null ,display)
	  (xlib:screen-black-pixel (res (current-screen))))
	 ((display-p ,display)
	  (xlib:screen-black-pixel 
	   (xlib:display-default-screen (res ,display))))
	 ((screen-p ,display)
	  (xlib:screen-black-pixel (res ,display)))
	 (t (error "black-pixel: invalid argument \`~s\`." ,display))))
;;;
;;; macro to map clx windows to Picasso window objects
;;;

(defvar *global-display-hashtab*
    (make-hash-table :size 5 :rehash-size 5 :test #'eq))

(defvar *global-window-table-hashtab*
    (make-hash-table :size 5 :rehash-size 5 :test #'eq))

(defmacro find-display (display-res)
  `(gethash ,display-res *global-display-hashtab*))

(defmacro find-window-table (display-res)
  `(gethash ,display-res *global-window-table-hashtab*))

(defmacro quick-find-window (window-res display-res)
  `(gethash ,window-res (find-window-table ,display-res)))

(defmacro find-window (window-res &optional display)
  "return the window object with the specified resource id"
  `(gethash
    ,window-res
    (cond ((null ,display)
	   (find-window-table
	    (xlib:window-display ,window-res)))
	  ((display-p ,display) (window-hash-table ,display))
	  ((screen-p ,display) (window-hash-table (display ,display)))
	  (t (error "find-window: invalid argument \`~s\`." ,display)))))

(defmacro find-from-objid (objid)
  `(gethash ,objid *objid-window-table*))

;;;
;;; macros to determine the class of an CLX event
;;;

(defvar *type-to-class-table* nil
  "hash table for storing event classes on CLX type")
