;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/rubber-gm.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 20:02:00 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;  rubber-gm is a simple geometry manager which merely handles resizing 
;;;  children to keep each interval in the collection at a fixed proportion
;;;  of the parent.  The geom-spec field of the child is used to hold the 
;;;  relative x, y, height, and width coordinates (in 0-1 range) of the
;;;  children.  If :center is in the geom-spec of the child, the child will
;;;  not be resized, but centered inside the area it should occupy (specified
;;;  by the proportions in the geom-spec).  Similarly, if :center-x or 
;;;  :center-y is in the geom-spec, the field will be centered only in the
;;;  horizontal or vertical direction respectively.

(defconstant *rubber-gm-magic-id* '*rubber-gm-magic-id*)

(defmethod gm-spec-changed ((gm (eql 'rubber-gm)) self child)
  (if (managed-p child)
      (r-gm-reshape-one-child self child)))

(defmethod gm-calculate-min-size ((gm (eql 'rubber-gm)) self)
  (let ((min-w 0)
	(min-h 0)
	(gs nil))
       (dolist (ch (managed-of (children self)))
	       (setq gs (r-gm-parse-spec ch self))
	       (setq min-w (max min-w (/ (base-width ch) (third gs))))
	       (setq min-h (max min-h (/ (base-height ch) (fourth gs)))))
       (setf (min-size self) (list min-w min-h))))

(defmethod gm-repack ((gm (eql 'rubber-gm)) self)
  (let ((nh (height self))
	(nw (width self))
	(children (managed-of (children self))))
       (dolist (ch children)
	       (let* ((gs (r-gm-parse-spec ch self))
		      (pct-x (first gs))
		      (pct-y (second gs))
		      (pct-w (third gs))
		      (pct-h (fourth gs))
		      (chw (width ch))
		      (chh (height ch))
		      (x (round (* pct-x nw)))
		      (y (round (* pct-y nh)))
		      (w (round (* pct-w nw)))
		      (h (round (* pct-h nh)))
		      (centering (fifth gs))
		      (cent (eq centering :center))
		      (centw (eq centering :center-x))
		      (centh (eq centering :center-y)))
		     (cond (cent
			    (move ch 
				  (round (+ x (/ (max 0 (- w chw)) 2)))
				  (round (+ y (/ (max 0 (- h chh)) 2)))))
			   (centw
			    (reshape ch
				  (round (+ x (/ (max 0 (- w chw)) w))) 
				  y
				  chw
				  h))
			   (centh
			    (reshape ch
				  x
				  (round (+ y (/ (max 0 (- h chh)) 2)))
				  w
				  chh))
			   (t
			    (reshape ch x y w h)))))
       (repaint self)))

(defun r-gm-reshape-one-child (self child)
  (let* ((nh (height self))
	 (nw (width self))
	 (gs (r-gm-parse-spec child self))
	 (old-x (x-offset child))
	 (old-y (y-offset child))
	 (old-w (width child))
	 (old-h (height child))
	 (pct-x (first gs))
	 (pct-y (second gs))
	 (pct-w (third gs))
	 (pct-h (fourth gs))
	 (x (round (* pct-x nw)))
	 (y (round (* pct-y nh)))
	 (w (round (* pct-w nw)))
	 (h (round (* pct-h nh)))
	 (rp-x (min old-x x))
	 (rp-y (min old-y y))
	 (rp-w (max old-w w 
		    (+ w (- x old-x)) 
		    (+ old-w (- old-x x))))
	 (rp-h (max old-h h 
		    (+ h (- y old-y)) 
		    (+ old-h (- old-y y))))
	 (centering (fifth gs))
	 (cent (eq centering :center))
	 (centw (eq centering :center-x))
	 (centh (eq centering :center-y)))
	(cond (cent
	       (move child 
		     (round (+ x (/ (max 0 (- w old-w)) 2)))
		     (round (+ y (/ (max 0 (- h old-h)) 2)))))
	      (centw
	       (reshape child
		     (round (+ x (/ (max 0 (- w old-w)) w))) 
		     y
		     old-w
		     h))
	      (centh
	       (reshape child
		     x
		     (round (+ y (/ (max 0 (- h old-h)) 2)))
		     w
		     old-h))
	      (t
	       (reshape child x y w h)))
	(if (and (exposed-p child) (not (x-window-p child)))
	    (repaint-region self rp-x rp-y rp-w rp-h))))


(defun r-gm-parse-spec (self parent)
  (let ((gs (geom-spec self)))
       (if (and (consp gs) (eql (car gs) *rubber-gm-magic-id*))
	   (cdr gs)
	   (if (and (listp gs) 
		    (or 
		     (and (eql (length gs) 4) (every #'r-gm-valid gs))
		     (and (eql (length gs) 5) (every #'r-gm-valid (butlast gs))
			  (member (fifth gs) '(:center :center-x :center-y)
				  :test #'eq))))
	       (progn
		(push *rubber-gm-magic-id* (geom-spec self))
		gs)
	       (let* ((pw (width parent))
		      (ph (height parent))
		      (cw (max (width self) (base-width self)))
		      (ch (max (height self) (base-height self)))
		      (cx (x-offset self))
		      (cy (y-offset self))
		      (pct-x (/ cx pw))
		      (pct-y (/ cy ph))
		      (pct-w (min (/ cw pw) (- 1 pct-x)))
		      (pct-h (min (/ ch ph) (- 1 pct-y)))
		      (new-gs (if (listp gs) gs (list gs)))
		      (cnt-x (member :center-x new-gs :test #'eq))
		      (cnt-y (member :center-y new-gs :test #'eq))
		      (cnt (or (and cnt-x cnt-y)
			       (member :center new-gs :test #'eq)))
		      (cnt-spec (if cnt :center
				    (if cnt-x :center-x
					(if cnt-y :center-y)))))
		     (setf (slot-value self 'geom-spec) 
			   (if cnt-spec
			       (list *rubber-gm-magic-id* 
				     pct-x pct-y pct-w pct-h cnt-spec)
			       (list *rubber-gm-magic-id* 
				     pct-x pct-y pct-w pct-h)))
		     (cdr (geom-spec self)))))))

(defun r-gm-valid (gs)
  (and (>= gs 0) (<= gs 1)))
