;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/matrix-gm.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 20:01:23 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;  matrix-gm is a special purpose matrix geometry manager.  It works in a
;;;  manner which is appropriate for table fields and other matrices which
;;;  are organized into rows and columns.  The invariant feature will be that
;;;  each row will be at least as tall as the size required by the tallest 
;;;  element and each column will similarly be at least as wide as
;;;  the width needed by the widest element.

;;;  The size of the matrix field must be preset at initialization and may not
;;;  be changed.  Also, the max-visible rows and columns, as well as the row and
;;;  column indices, should be specified.  These
;;;  are values that
;;;  tell the gm the maximum number of rows and columns of gadgets the gm
;;;  can display at any one time, and where in the table of gadgets the upper
;;;  left corner is.  It is possible that less than that many will
;;;  be accommodated, but not more.  Also, row and column maximum and minimum
;;;  dimensions can be input to put restrictions on how large and how small
;;;  columns and rows can grow.  By default, a row's and column's minimum is
;;;  the largest base height and width (respectively) of the elements contained
;;;  in that row and column, and the maximums are infinite (no restrictions). 

;;;  Children, when added to a matrix-gm, are assumed to have a list in their
;;;  geom-spec whose first element is the row and second is the column it
;;;  should be placed in.

;;;  Row and column maximums are kinda strange.  They aren't the largest size
;;;  the gadget can grow to, but are actually the largest size the minimum size
;;;  can grow to.  What this means is that if a huge gadget is added, if a
;;;  reasonable maximum has been defined the column will no become
;;;  proportionately huge compared to other columns.

(defconstant *matrix-gm-magic-id* '*matrix-gm-magic-id*)

(defstruct (matrix-detail  (:conc-name md-))
  (num-rows 0 :type integer)     ;;;  Determine size of
  (num-cols 0 :type integer)     ;;;  table this GM contains.
  (row-index 0 :type integer)    ;;;  The number of the upper-left exposed
  (col-index 0 :type integer)    ;;;  cell in the matrix.
  (visible-rows 0 :type integer) ;;;  The number of rows and columns exposed
  (visible-cols 0 :type integer) ;;;  at this time.  
  (max-visible-rows 0 :type integer);;the maximum # rows to show
  (max-visible-cols 0 :type integer);;the maximun # cols to show
  (row-mins nil :type array)     ;;;  User-defined size minimums and maximums
  (row-maxs nil :type array)
  (col-mins nil :type array)
  (col-maxs nil :type array)
  (row-current nil :type array)  ;;;  Widths and heights of rows & col's in
  (col-current nil :type array)  ;;;  current display.
  (inter-col-pad 0 :type integer) ;;; How much to space between children
  (inter-row-pad 0 :type integer) 
  (conform nil :type atom)
  (children nil :type array))    ;;;  Pointers to children in table form.


(defmethod gm-initialize ((gm (eql 'matrix-gm)) self)
  (setf (gm-data self) (make-matrix-detail)))

;;;
;;; gm-matrix-init must be called to initialize the geometry manager.
;;; Rows & columns are the maximum number of rows and columns in this matrix
;;;

(defun gm-matrix-init (self &key 
			    (rows 		(rows self)) 
			    (cols		(cols self))
			    (max-visible-rows 	nil)
			    (max-visible-cols 	nil)
			    (row-index		0)
			    (col-index		0)
			    (row-mins		nil)
			    (col-mins		nil)
			    (row-maxs		nil)
			    (col-maxs		nil)
			    (inter-row-pad	0) 
			    (inter-col-pad	0)
			    (conform		nil)
			    &aux md)
  
  (setq md (gm-data self))
  
  (setf (md-num-rows md) rows
	(md-num-cols md) cols 
	(md-max-visible-rows md) (if max-visible-rows max-visible-rows rows) 
  	(md-max-visible-cols md) (if max-visible-cols max-visible-cols cols)
	(md-row-index md) row-index
	(md-col-index md) col-index
	(md-row-mins md) (if row-mins 
			     (make-array (list rows) 
					 :element-type 'integer
					 :initial-contents row-mins)
			     (make-array (list rows) 
					 :element-type 'integer
					 :initial-element 0)) 
	(md-col-mins md) (if col-mins 
			     (make-array (list cols) 
					 :element-type 'integer
					 :initial-contents col-mins)
			     (make-array (list cols) 
					 :element-type 'integer
					 :initial-element 0))
	(md-row-maxs md) (if row-maxs 
			     (make-array (list rows) 
					 :element-type 'integer
					 :initial-contents row-maxs)
			     (make-array (list rows) 
					 :element-type 'integer
					 :initial-element 10000)) 
	(md-col-maxs md) (if col-maxs 
			     (make-array (list cols) 
					 :element-type 'integer
					 :initial-contents col-maxs)
			     (make-array (list cols) 
					 :element-type 'integer
					 :initial-element 10000)) 
	(md-row-current md) (make-array (list rows)
					:element-type 'integer
					:initial-element 0)
	(md-col-current md) (make-array (list cols)
					:element-type 'integer
					:initial-element 0)
	(md-children md) (make-array (list rows cols)
				     :initial-element nil)
	(md-inter-col-pad md) inter-col-pad
	(md-inter-row-pad md) inter-row-pad
	(md-conform md) conform))

(defmethod gm-add-child ((gm (eql 'matrix-gm)) self child)
  (cond ((or (= (md-num-rows (gm-data self)) 0)
	     (= (md-num-cols (gm-data self)) 0))
	 (error "matrix.add-child: matrix not initialized"))
	((not (consp (geom-spec child)))
	 (error "matrix.add-child:  Child added with incorrect geom-spec~%"))
	(t
	 (let* ((gs (geom-spec child))
		(md (gm-data self))
		(bw (base-width child))
		(bh (base-height child))
		(row (car gs))
		(col (cadr gs))
		(bd (border-width child))
		(bdw nil)
		(bdh nil))
	       (if (listp bd)
		   (setq bdw (+ (first bd) (third bd))
			 bdh (+ (second bd) (fourth bd)))
		   (setq bdw
			 (setq bdh (* 2 bd))))
	       (setf (aref (md-children md) row col) child)
	       (push child (slot-value self 'children))
	       (cond ((and (> (+ bh bdh) (aref (md-row-current md) row))
			   (> (+ bw bdw) (aref (md-col-current md) col)))
		      (setf (aref (md-row-current md) row)
			    (+ bh bdh))
		      (setf (aref (md-col-current md) col)
			    (+ bw bdw))
		      (if (or (and (>= row (md-row-index md))
				   (< row (+ (md-row-index md)
					     (md-max-visible-rows md))))
			      (and (>= col (md-col-index md))
				   (< col (+ (md-col-index md)
					     (md-max-visible-cols md)))))
			  ;;; Only repack if child alters max-visible row or col
			  (do-repack self)))
		     ((> (+ bh bdh)
			 (aref (md-row-current md) row))
		      (setf (aref (md-row-current md) row)
			    (+ bh bdh))
		      (if (and (>= row (md-row-index md))
			       (< row (+ (md-row-index md)
					 (md-max-visible-rows md))))
			  ;;; Only repack if child alters max-visible row
			  (do-repack self)))
		     ((> (+ bw bdw)
			 (aref (md-col-current md) col))
		      (setf (aref (md-col-current md) col)
			    (+ bw bdw))
		      (if (and (>= col (md-col-index md))
			       (< col (+ (md-col-index md)
					 (md-max-visible-cols md))))
			  ;;; Only repack if child alters max-visible col
			  (do-repack self)))
		     (t
		      (if (and (>= row (md-row-index md))
			       (< row (+ (md-row-index md)
					 (md-max-visible-rows md)))
			       (>= col (md-col-index md))
			       (< col (+ (md-col-index md)
					 (md-max-visible-cols md))))
			  ;;; Even though the child doesn't alter row or column
			  ;;; mins, we still have to set it's position if it is
			  ;;; max-visible.
			  (do-repack self))))))))

(defmethod gm-delete-child ((gm (eql 'matrix-gm)) self child)
  (cond ((or (= (md-num-rows (gm-data self)) 0)
	     (= (md-num-cols (gm-data self)) 0))
	 (error "matrix.delete-child must call gm-setsize first"))
	(t
	 (let* ((gs (geom-spec child))
		(md (gm-data self))
		(bw (base-width child))
		(bh (base-height child))
		(row (car gs))
		(col (cadr gs))
		(bd (border-width child))
		(bdw nil)
		(bdh nil))
	       (if (listp bd)
		   (setq bdw (+ (first bd) (third bd))
			 bdh (+ (second bd) (fourth bd)))
		   (setq bdw
			 (setq bdh (* 2 bd))))
	       (setf (aref (md-children md) row col) nil)
	       (setf (slot-value self 'children)
		     (delete child (children self)))
	       (cond ((and (= (+ bh bdh)
			      (aref (md-row-current md) row))
			   (= (+ bw bdw)
			      (aref (md-col-current md) row)))
		      (setf (aref (md-row-current md) row)
			    (recomp-row self row))
		      (setf (aref (md-col-current md) col)
			    (recomp-col self col))
		      (if (or (and (>= row (md-row-index md))
				   (< row (+ (md-row-index md)
					     (md-max-visible-rows md))))
			      (and (>= col (md-col-index md))
				   (< col (+ (md-col-index md)
					     (md-max-visible-cols md)))))
			  ;;; Only repack if child alters max-visible row or col
			  (do-repack self)))
		     ((> (+ bh bdh)
			 (aref (md-row-current md) row))
		      (setf (aref (md-row-current md) row)
			    (recomp-row self row))
		      (if (and (>= row (md-row-index md))
			       (< row (+ (md-row-index md)
					 (md-max-visible-rows md))))
			  ;;; Only repack if child alters max-visible row
			  (do-repack self)))
		     ((> (+ bw bdw)
			 (aref (md-col-current md) row))
		      (setf (aref (md-col-current md) col)
			    (recomp-col self col))
		      (if (and (>= col (md-col-index md))
			       (< col (+ (md-col-index md)
					 (md-max-visible-cols md))))
			  ;;; Only repack if child alters max-visible col
			  (do-repack self)))
		     (t
		      (if (and (>= row (md-row-index md))
			       (< row (+ (md-row-index md)
					 (md-max-visible-rows md)))
			       (>= col (md-col-index md))
			       (< col (+ (md-col-index md)
					 (md-max-visible-cols md))))
			  ;;; Since child doesn't alter row or column mins, only
			  ;;; repaint if child is max-visible.
			  (repaint self))))))))

(defun recomp-row (self row &aux temp md bd)
  (setq temp 0
	md (gm-data self))
  (do ((index 0 (1+ index))
       (child nil)
       (h 0))
      ((= index (md-num-cols md)))
      (setf child (aref (md-children md) row index))
      (setf h (if (null child)
		  0
		  (progn
		   (setq bd (border-width child))
		   (setq bd 
			 (if (listp bd)
			     (+ (second bd) (fourth bd))
			     (* 2 bd)))
		   (+ (base-height child) bd))))
      (if (< temp h)
	  (setf temp h)))
  (setf (aref (md-row-current md) row) temp))

(defun recomp-col (self col &aux temp md bd)
  (setq temp 0
	md (gm-data self))
  (do ((index 0 (1+ index))
       (child nil)
       (h 0))
      ((= index (md-num-rows md)))
      (setf child (aref (md-children md) index col))
      (setf h (if (null child)
		  0
		  (progn
		   (setq bd (border-width child))
		   (setq bd 
			 (if (listp bd)
			     (+ (first bd) (third bd))
			     (* 2 bd)))
		   (+ (base-width child) bd))))
      (if (< temp h)
	  (setf temp h)))
  (setf (aref (md-col-current md) col) temp))

(defmethod gm-resize-hint-changed ((gm (eql 'matrix-gm)) self child)
  (unless (md-conform (gm-data self))
	  (return-from gm-resize-hint-changed))
  (let* ((gs (geom-spec child))
	 (md (gm-data self))
	 (row (car gs))
	 (col (cadr gs)))
	(recomp-row self row)
	(recomp-col self col)
	(if (or (and (>= row (md-row-index md))
		     (< row (+ (md-row-index md)
			       (md-max-visible-rows md))))
		(and (>= col (md-col-index md))
		     (< col (+ (md-col-index md)
			       (md-max-visible-cols md)))))
	    (do-repack self)))) ;;; Only repack if child alters displayed area

(defmethod gm-status-changed ((gm (eql 'matrix-gm)) self child)
  (let* ((gs (geom-spec child))
	 (md (gm-data self))
	 (row (car gs))
	 (col (cadr gs)))
    (if (and (>= row (md-row-index md))
	     (>= col (md-col-index md))
	     (< row (+ (md-row-index md)
		       (md-max-visible-rows md)))
	     (< col (+ (md-col-index md)
		       (md-max-visible-cols md))))
	(repaint self)))) ;;; Only repaint if child is in displayed area

(defmethod gm-calc-min-size ((gm (eql 'matrix-gm)) self)
  (let* ((md (gm-data self))
	 (width (* (md-max-visible-cols md) (md-inter-col-pad md)))
	 (height (* (md-max-visible-rows md) (md-inter-row-pad md))))
    (do ((index (md-col-index md) (1+ index)))
	((= index (+ (md-max-visible-cols md) (md-col-index md))))
	(setf width (+ width (max (min (aref (md-col-maxs md) index)
				       (aref (md-col-current md) index))
				  (aref (md-col-mins md) index)))))
    (do ((index (md-row-index md) (1+ index)))
	((= index (+ (md-max-visible-rows md) (md-row-index md))))
	(setf height (+ height (max (min (aref (md-row-maxs md) index)
				       (aref (md-row-current md) index))
				  (aref (md-row-mins md) index)))))
    (setf (min-size self) (list width height))))

(defmethod gm-repack ((gm (eql 'matrix-gm)) self &aux widthscale heightscale)
  (if (or (= 0 (md-num-rows (gm-data self)))
	  (= 0 (md-num-cols (gm-data self))))
      (error "matrix.repack: attempting to repack an incomplete matrix ~%"))
  (let* ((md (gm-data self))
	 (ri (md-row-index md))
	 (ci (md-col-index md))
	 (mr (md-max-visible-rows md))
	 (mc (md-max-visible-cols md))
	 (mdrn (md-row-mins md))
	 (mdrc (md-row-current md))
	 (mdrx (md-row-maxs md))
	 (mdcn (md-col-mins md))
	 (mdcc (md-col-current md))
	 (mdcx (md-col-maxs md))
	 (irp (md-inter-row-pad md))
	 (icp (md-inter-col-pad md))
	 (tw 0) ;;; total width and height of the children's base-sizes
	 (th 0)
	 (ws (width self))
	 (hs (height self))
	 (mdch (md-children md))
	 (ch nil)
	 (bd nil)
	 (bdw nil)
	 (bdh nil)
	 (w nil)
	 (h nil)
	 (rows 0)                  ;;; The number of actually max-visible rows &
	 (cols 0))                 ;;; columns

	;;	Calculate how many cols we can fit
	(do ((index ci (1+ index))
	     (minw 0)
	     (maxw 0)
	     (cur 0)
	     (val 0))
	    ((= index mc))
	    (setf minw (aref mdcn index))
	    (setf maxw (aref mdcx index))
	    (setf cur (aref mdcc index))
	    (setf val (max (min maxw cur) minw))
	    (if (> (+ tw val (* (- index ci -2) icp))
		   ws)
		(return)
		(progn
		 (setf tw (+ tw val))
		 (setf cols (1+ cols)))))
	(when (zerop tw) (setf tw (aref mdcc ci)))

	;;	Calculate how many rows we can fit
	(do ((index ri (1+ index))
	     (minh 0)
	     (maxh 0)
	     (cur 0)
	     (val 0))
	    ((= index mr))
	    (setf minh (aref mdrn index))
	    (setf maxh (aref mdrx index)) 
	    (setf cur (aref mdrc index))
	    (setf val (max (min maxh cur) minh))
	    (if (> (+ th val (* (- index ri -2) irp))
		   (height self))
		(return)
		(progn
		 (setf th (+ th val))
		 (setf rows (1+ rows)))))
	(when (zerop th) (setf th (aref mdrc ri)))

	;;	Resize children to fit
	(setq widthscale (/ (- ws (* (1+ (if (zerop cols) 1 cols)) icp)) 
			    (max tw 1))
	      heightscale (/ (- hs (* (1+ (if (zerop rows) 1 rows)) irp)) 
			     (max th 1)))
	(do ((index ri (1+ index))
	     (ypos icp))
	    ((= index (+ ri rows))) 
	    (setq h 
		  (round 
		   (* (max (min (aref mdrc index)
				(aref mdrx index))
			   (aref mdrn index))
		      heightscale)))
	    (do* ((index2 ci (1+ index2))
		  (xpos icp)) 
		 ((= index2 (+ ci cols)))  	;;; set children to their sizes
		 (setq ch (aref mdch index index2))
		 (setq w (round (* (max (min (aref mdcc index2)
					     (aref mdcx index2))
					(aref mdcn index2))
				   widthscale)))
		 (cond ((listp ch)
			(setf (getf ch :x) xpos
			      (getf ch :y) ypos
			      (getf ch :width) w
			      (getf ch :height) h))
		       (t
			(setq bd (border-width ch))
			(if (listp bd)
			    (setq bdw (+ (first bd) (third bd))
				  bdh (+ (second bd) (fourth bd)))
			    (setq bdw
				  (setq bdh (* 2 bd))))
			(reshape ch
				 xpos
				 ypos 
				 (- w bdw)
				 (- h bdh))))
		 (setq xpos (+ xpos icp w)))
	    (setq ypos (+ ypos irp h)))

	;;	Make sure we have at least one row/col in there
	(when (and (> ws 0) (zerop cols))
	      (do ((index ri (1+ index))
		   (ypos irp))
		  ((= index (+ ri rows)))
		  ;;	Set children to their sizes
		  (setq ch (aref mdch index ci))
		  (setq h (round (* (max (min (aref mdrc index)
					      (aref mdrx index))
					 (aref mdrn index))
				    heightscale))) 
		  (cond ((listp ch)
			 (setf (getf ch :x) icp
			       (getf ch :y) ypos
			       (getf ch :width) (- ws icp icp)
			       (getf ch :height) h))
			(t
			(setq bd (border-width ch))
			(if (listp bd)
			    (setq bdw (+ (first bd) (third bd))
				  bdh (+ (second bd) (fourth bd)))
			    (setq bdw
				  (setq bdh (* 2 bd))))
			 (reshape ch
				  icp
				  ypos 
				  (- ws icp icp bdw)
				  (- h bdh))))
		  (setq ypos (+ ypos icp h)))
	      (setf cols 1))
	(when (and (> hs 0) (zerop rows))
	      (do ((index ci (1+ index)) 
		   (xpos icp)) 
		  ((= index (+ ci cols)))
		  ;;	Set children to their sizes
		  (setq ch (aref mdch ri index))
		  (setq w (round (* (max (min (aref mdcc index)
					      (aref mdcx index))
					 (aref mdcn index))
				    widthscale)))
		  (cond ((listp ch)
			 (setf (getf ch :x) xpos
			       (getf ch :y) irp
			       (getf ch :width) w
			       (getf ch :height) (- hs irp irp)))
			(t
			(setq bd (border-width ch))
			(if (listp bd)
			    (setq bdw (+ (first bd) (third bd))
				  bdh (+ (second bd) (fourth bd)))
			    (setq bdw
				  (setq bdh (* 2 bd))))
			 (reshape ch
				  xpos
				  irp
				  (- w bdw)
				  (- hs irp irp bdh))))
		  (setq xpos (+ xpos icp w)))
	      (setf rows 1))

	;;	Pend invisible children
	(do ((index 0 (1+ index)))
	    ((= index ci))
	    (do ((index2 0 (1+ index2)))
		((= index2 mr))
		(if (listp (setq ch (aref mdch index2 index)))
		    (setf (getf ch :exposed) nil)
		    (pend ch))))
	(do ((index2 0 (1+ index2)))
	    ((= index2 ri))
	    (do ((index ci (1+ index)))
		((= index mc))
		(if (listp (setq ch (aref mdch index2 index)))
		    (setf (getf ch :exposed) nil)
		    (pend ch))))
	(if (< rows mr)
	    (do ((index (+ ri rows) (1+ index)))
		((= index mr))
		(do ((index2 ci (1+ index2)))
		    ((= index2 mc))
		    (if (listp (setq ch (aref mdch index index2)))
			(setf (getf ch :exposed) nil)
			(pend ch)))))
	(if (< cols mc)
	    (do ((index (+ ci cols) (1+ index)))
		((= index mc))
		(do ((index2 ri (1+ index2)))
		    ((= index2 mr))
		    (if (listp (setq ch (aref mdch index2 index)))
			(setf (getf ch :exposed) nil)
			(pend ch)))))

	;;	Unpend visible previously pended children
	(let ((c (+ ci cols))
	      (r (+ ri rows)))
	     (do ((index ci (1+ index)))
		 ((>= index c))
		 (do ((index2 ri (1+ index2)))
		     ((>= index2 r))
		     (if (listp (setq ch (aref mdch index2 index)))
			 (setf (getf ch :exposed) t)
			 (unpend ch)))))

	(setf (md-visible-rows md) rows
	      (md-visible-cols md) cols)
	(when (typep self 'matrix-field)
	      (mf-resize-update self rows cols))))

;;;
;;;	Swaps two rows in a matrix
;;;

(defun gm-matrix-col-exchange (self col1 col2 
				    &aux md mins maxs curs el1 el2 mdch mr)
  (setq md (gm-data self))
  (setq mins (md-col-mins md)
	maxs (md-col-maxs md)
	curs (md-col-current md)
	mdch (md-children md)
	mr (md-max-visible-rows md))

  ;;	Swap size info
  (psetf (aref mins col1) (aref mins col2)
	 (aref mins col2) (aref mins col1))
  (psetf (aref maxs col1) (aref maxs col2)
	 (aref maxs col2) (aref maxs col1))
  (psetf (aref curs col1) (aref curs col2)
	 (aref curs col2) (aref curs col1))

  ;;	Swap children & geom-specs
  (do ((row 0 (1+ row)))
      ((= row mr))
      (psetf (aref mdch row col1) (setq el1 (aref mdch row col2))
	     (aref mdch row col2) (setq el2 (aref mdch row col1)))
      (psetf (geom-spec el1) (geom-spec el2)
	     (geom-spec el2) (geom-spec el1))))

(defun gm-matrix-row-exchange (self row1 row2 
				    &aux md mins maxs curs el1 el2 mdch mc)
  (setq md (gm-data self))
  (setq mins (md-row-mins md)
	maxs (md-row-maxs md)
	curs (md-row-current md)
	mdch (md-children md)
	mc (md-max-visible-cols md))

  ;;	Swap size info
  (psetf (aref mins row1) (aref mins row2)
	 (aref mins row2) (aref mins row1))
  (psetf (aref maxs row1) (aref maxs row2)
	 (aref maxs row2) (aref maxs row1))
  (psetf (aref curs row1) (aref curs row2)
	 (aref curs row2) (aref curs row1))

  ;;	Swap children & geom-specs
  (do ((col 0 (1+ col)))
      ((= col mc))
      (psetf (aref mdch row1 col) (setq el1 (aref mdch row2 col))
	     (aref mdch row2 col) (setq el2 (aref mdch row1 col)))
      (psetf (geom-spec el1) (geom-spec el2)
	     (geom-spec el2) (geom-spec el1))))

;;;
;;;	Updates row/col max/min/cur data to reflect "scoot" operation
;;;

(defun gm-matrix-scoot-rows (self l1 l2 &aux md mins maxs curs nr newmins 
				  	newmaxs newcurs row)
  (setq md (gm-data self))
  (setq mins (md-row-mins md)
	maxs (md-row-maxs md)
	curs (md-row-current md)
	nr (md-num-rows md)
	newmins (make-array (list nr) :element-type 'integer)
	newmaxs (make-array (list nr) :element-type 'integer)
	newcurs (make-array (list nr) :element-type 'integer)
	row 0)

  ;;	Reconstruct mins/maxs/curs
  (dolist (cur l1)
	  (setf (aref newmins row) (aref mins cur)
		(aref newmaxs row) (aref maxs cur)
		(aref newcurs row) (aref curs cur))
	  (setq row (1+ row)))
  (dolist (cur l2)
	  (setf (aref newmins row) (aref mins cur)
		(aref newmaxs row) (aref maxs cur)
		(aref newcurs row) (aref curs cur))
	  (setq row (1+ row)))

  ;;	Replace old arrays with new ones
  (setf (md-row-mins md) newmins
	(md-row-maxs md) newmaxs
	(md-row-current md) newcurs))

(defun gm-matrix-scoot-cols (self l1 l2 &aux md mins maxs curs nc newmins 
				  	newmaxs newcurs col)
  (setq md (gm-data self))
  (setq mins (md-col-mins md)
	maxs (md-col-maxs md)
	curs (md-col-current md)
	nc (md-num-cols md)
	newmins (make-array (list nc) :element-type 'integer)
	newmaxs (make-array (list nc) :element-type 'integer)
	newcurs (make-array (list nc) :element-type 'integer)
	col 0)

  ;;	Reconstruct mins/maxs/curs
  (dolist (cur l1)
	  (setf (aref newmins col) (aref mins cur)
		(aref newmaxs col) (aref maxs cur)
		(aref newcurs col) (aref curs cur))
	  (setq col (1+ col)))
  (dolist (cur l2)
	  (setf (aref newmins col) (aref mins cur)
		(aref newmaxs col) (aref maxs cur)
		(aref newcurs col) (aref curs cur))
	  (setq col (1+ col)))
  
  ;;	Replace old arrays with new ones
  (setf (md-col-mins md) newmins
	(md-col-maxs md) newmaxs
	(md-col-current md) newcurs))
