;;;
;;; POSTGRES/Common LISP Object System
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/portal.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 12:50:51 $
;;;

(in-package 'pgclos :nicknames '(postclos) :use '(lisp pcl libpq))

;;;
;;; imports from PCL package
;;;
(import '(pcl::class-slots
	  pcl::slotd-name
	  pcl::slotd-initform)
	(find-package 'pgclos))

;;;
;;; definition of the portal class
;;;
(defclass portal (object)
  ((database
    :initform nil
    :type string
    :reader portal-database
    :allocation :class)
   (name
    :initform ""
    :type string
    :accessor portal-name)
   (array
    :initform nil
    :type array
    :accessor portal-tuple-array)
   (index
    :initform -1
    :type integer
    :accessor portal-tuple-index)
   (count
    :initform 0
    :type integer
    :accessor portal-tuple-count)))

(defun make-portal (&rest keys)
  (apply #'make-instance (cons 'portal keys)))

;;;
;;; initialize an instance of the portal class
;;;

(defmethod new-instance ((self portal)
			 &key
			 (database (current-database))
			 (name "")
			 (target nil targetp)
			 (where nil)
			 &allow-other-keys)
  "Make a new portal instance.  The database specifies the POSTGRES database;
the name specifies the name of the portal; the target specifies the target list
for the retrieve; and where specifies the where-clause."
  ;; test specified database
  (if (or (null database) (not (stringp database)))
      ;; signal error
      (error "portal.new-instance: no/invalid database ~s" database)
    ;; set database
    (libpq:pqsetdb database))
  ;; test specified name
  (if (or (null name) (not (stringp name)))
      ;; signal error
      (error "portal.new-instance: invalid portal name ~s" name))
  ;; allocate query string
  (let ((query nil))
    ;; test target and where clause
    (cond ((or (null targetp) (not (stringp target)))
	   ;; signal error
	   (error "portal.new-instance: invalid target list - ~s" target))
	  ((null where)
	   ;; make query string w/ no where clause
	   (setq query (format nil "retrieve portal ~a ( ~a )" name target)))
	  ((not (stringp where))
	   ;; signal error
	   (error "portal.new-instance: invalid where clause - ~s" where))
	  (t
	   ;; make query string w/ where clause
	   (setq query (format nil "retrieve portal ~a ( ~a ) where ~a" 
			       name 
			       target 
			       where))))
    ;; execute query
    (libpq:pqexec query))
  ;; return self
  self)

;;;
;;; setf methods for portal slots
;;;

(defmethod (setf portal-tuple-index) ((value integer) (self portal))
  "setf method for changing the tuple-index"
  ;; get current index
  (let ((count (slot-value self 'count)))
    ;; test if new value is out of bounds
    (if (or (< value 0) (>= value count))
	;; signal error
	(error "portal.setf-portal-tuple-index: invalid index ~s" value))
    ;; set new value
    (setf (slot-value self 'index) value)))

;;;
;;; portal methods
;;;

(defmethod rewind-portal ((self portal))
  "rewind the portal to the first tuple and return it"
  ;; set the portal-tuple-index to zero
  (setf (portal-tuple-index self) 0)
  ;; return tuple
  (current-tuple self))

(defmethod close-portal ((self portal))
  "close the portal and end communications to the backend"
  ;; set the index slot to negative one
  (setf (slot-value self 'index) -1)
  ;; get portal-array
  (let ((array (slot-value self 'array))
	(count (slot-value self 'count)))
    ;; set all array references to nil
    (dotimes (i count)
	     (setf (aref array i) nil)))
  ;; set the count slot to zero
  (setf (slot-value self 'count) 0)
  ;; end communications to the backend
  (libpq:pqfinish)
  ;; return self
  self)

;;;
;;; database specification methods
;;;

(defun current-database ()
  "Return the name of the current database, or nil if no current database"
  ;; get class slots for the display class
  (let* ((class (find-class 'portal))
	 (class-slots (class-slots class))
	 (current (find 'database class-slots :key #'slotd-name)))
    ;; test if the current display slot is defined
    (if (null current)
	;; signal error
	(error "current-database: can't find class slot for database"))
    ;; return the value of the current database
    (slotd-initform current)))

(defun setf-current-database (name)
  "Set the current database to name, or signal error is already defined"
  ;; get class slots for the display class
  (let* ((class (find-class 'portal))
	 (class-slots (class-slots class))
	 (current (find 'database class-slots :key #'slotd-name)))
    ;; test if the current display slot is defined
    (if (null current)
	;; signal error
	(error "current-database: can't find class slot for database"))
    ;; test for invalid name
    (if (and name (not (stringp name)))
	;; signal error
	(error "setf-current-display: invalid database ~s" name))
    ;; test if null/reset
    (if (null name)
	;; finish 
	(libpq:pqfinish)
      ;; set  database name
      (libpq:pqsetdb name))
    ;; change the value of the current database
    (setf (slotd-initform current) name)))

(defsetf current-database setf-current-database)

;;;
;;; fetch tuples from POSTGRES back-end
;;;

(defun pg-to-cl-type (pgtype)
  "Determine the Common LISP type which best matches the POSTGRES type"
  ;; declare argument(s)
  (declare (ignore pgtype))
  ;; return string
  (quote string))

(defmethod fetch-tuples ((self portal)
			 &key
			 (direction :forward)
			 (count :all)
			 &allow-other-keys)
  "Fetch tuples from the back-end into the front-end portal array.  The
direction may be either :forward or :backward - :forward is the default. The
count may be any positive integer and specifies the maximum number of tuples
to fetch from the backend."
  ;; get name
  (let ((pname (portal-name self)))
    ;; test direction
    (if (eq direction :forward)
	;; make direction a string
	(setq direction "forward")
      ;; test if backward
      (if (eq direction :backward)
	  ;; make direction a string
	  (setq direction "backward")
	;; signal error
	(error "portal.fetch: invalid direction ~s" direction)))
    ;; test if count numeric
    (if (and (numberp count) (plusp count))
	;; exec fetch w/ number
	(libpq:pqexec (format nil "fetch ~a ~d in ~a" direction count pname))
      ;; test if count :all
      (if (eq count :all)
	  ;; exec fetch for all tuples
	  (libpq:pqexec (format nil "fetch ~a all in ~a" direction pname))
	;; signal error
	(error "portal.fetch: invalid count ~s" count)))
    ;; get portal information
    (let* ((parray (libpq:pqparray pname))
	   (ntuples (libpq:pqntuples parray))
	   (nfields nil)
	   (slots nil)
	   (fname nil)
	   (rdr nil))
      ;; test for no tuples
      (if (= ntuples 0)
	  ;; signal error
	  ;; (error "portal.fetch: no tuples returned")
	  ;; return a empty list. this makes more sense than an error.
	  (return-from fetch-tuples nil)
	;; determine number of fields -- should be the same for all tuples
	(setq nfields (libpq:pqnfields parray 0)))
      ;; set portal count and index slots to initial values
      (setf (slot-value self 'count) ntuples)
      (setf (slot-value self 'index) 0)
      ;; get tuple/instance information
      (dotimes (i nfields)
	       ;; get field name
	       (setq fname (read-from-string (libpq:pqfname parray 0 i)))
	       ;; build accessor
	       (setq rdr (read-from-string (format nil "~a-~s" pname fname)))
	       ;; make new slot description
	       (push (list fname 
			   :initform nil
			   :type (pg-to-cl-type (libpq:pqftype parray 0 i))
			   :reader rdr)
		     slots))
      ;; reverse slot list
      (setq slots (reverse slots))
      ;; allocate class name, tuple class and portal array
      (let* ((cname (read-from-string (symbol-name (gensym "tuple-"))))
	     ;; A bug in PCL: defclass always returns nil.
	     (nil-tclass (eval `(defclass ,cname (object) ,slots)))
	     (tclass (find-class cname))
	     (array (make-array ntuples)))
	;; initialize array
	(dotimes (i ntuples)
		 ;; allocate instances for each tuple
		 (setf (aref array i) (make-instance tclass))
		 ;; initialize slot values
		 (dotimes (j nfields)
			  ;; set slot value
			  (setf (slot-value (aref array i) (car (nth j slots)))
				(libpq:pqgetvalue parray i j))))
	;; set up portal array
	(setf (slot-value self 'array) array)))))

(defmethod current-tuple ((self portal))
  "Return the current tuple as indicated by the tuple-index"
  (aref (slot-value self 'array) (slot-value self 'index)))

(defmethod next-tuple ((self portal))
  "Make the next tuple in the portal the current tuple and return it"
  ;; increment tuple-index of portal
  (incf (portal-tuple-index self))
  ;; return tuple
  (current-tuple self))

(defmethod previous-tuple ((self portal))
  "Make the previous tuple in the portal the current tuple and return it"
  ;; decrement tuple-index of portal
  (decf (portal-tuple-index self))
  ;; return tuple
  (current-tuple self))

;;;
;;; Exports from PGCLOS
;;;
(export '(make-portal
	  close-portal
	  rewind-portal
	  current-tuple
	  next-tuple
	  previous-tuple
	  fetch-tuples
	  current-database
	  portal-tuple-count
	  portal-tuple-index
	  portal-tuple-array
	  portal-database
	  portal-name)
	(find-package 'pgclos))
