;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/clos.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 19:57:28 $
;;;

;;; This file contains modifications to CLOS used by picasso.
;;; Among these is a modification to the parse-defmethod function,
;;; called by the defmethod macro, to add profiling code.  The code
;;; keeps track of the number of calls to a method and the approximate
;;; time spent in the method.  The resolution of this clock is about 20
;;; milleseconds.  Note that this profiling data applies only to method
;;; calls, not function calls.
;;;
;;; The basic interface to the profiler is simple.  :profiler should be
;;; on the *features* list to enable the profiling code to be generated.
;;; The basic overhead of this code with profiling off is very low (two
;;; compares with global variable and a couple of surrounding prog
;;; structures), and so shouldn't present a problem for debugging use.
;;; All modules to be profiled must be recompiled with :profiler on
;;; the features list.  Once this is done, generation of profiling
;;; statistics can be turned on and off using the prof:*profile*
;;; global variable;  if this variable is non-nil, statistics are gathered,
;;; otherwise they are not.  This mechanism allows fine tuned gathering of
;;; profiling data under programmer control.  For example, to figure out
;;; how time is spent inside the "foo" method, set prof:*profile* to "t"
;;; as the first line of the method, and back to it's original value at
;;; the exit point of the method.
;;;
;;; The report generating facilities of the profiler are modeled after
;;; the allegro profiler. The most useful functions are:
;;;	method-call-report (&key number-to-report sort-by)
;;;	method-call-clear ()
;;; The first of these functions produces a sorted list showing the
;;; number of times a method is called, the total time spent in a method,
;;; and the average time spent in a method. The number-to-report keyword
;;; controls how many records are printed (default is all), the keyword
;;; sort-by controls the ordering of the sorting and should be one of
;;; the values :total-time :average-time :calls (default is :total-time).
;;; If given :total-time, the function sorts the list by the total time
;;; spent in the method, if given :average-time, the function sorts the
;;; list by the average time spent in the method, and if given :calls,
;;; the function sorts the list by the number of calls to the method.
;;;
;;; The method-call-clear function simply resets the data structures
;;; holding the profiling statistics.  It is used to seperate profiling
;;; runs.  Note that unlike the allegro profiler, this function is not 
;;; automatically called after method-call-report.
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

(in-package 'profiler :nicknames '(prof) :use '(lisp))

(pushnew :profile *features*)
(defvar *profile* nil "Turn this on for profiling.")
(defvar *call-hashtab* (make-hash-table :test #'eq))

;;
;; Records are of the form (call-count elapsed-time start-time)
;; start-time isa list of start-times.
;;
(defun start-call (name-qualifiers-classes)
  (let ((rec (gethash name-qualifiers-classes *call-hashtab*)))
       (when (null rec)
	     (setq rec (list 0 0 nil))
	     (setf (gethash name-qualifiers-classes *call-hashtab*) rec))
       ;; Record the start time.
       (setq rec (cddr rec))
       (push (get-internal-run-time) (car rec))))

(defun end-call (name-qualifiers-classes)
  (let ((end-time (get-internal-run-time))
	(start-time 0)
	(rec (gethash name-qualifiers-classes *call-hashtab*)))
       (if (null rec)
	   (return-from end-call))
       ;; Get the end time
       (setq start-time (pop (caddr rec)))
       (when (integerp start-time)
	     ;; Increment the call count...
	     (incf (car rec))
	     ;; Increment the elaaapsed run time
	     (incf (cadr rec) (- end-time start-time)))))

(defun report-one (k v)
  (let* ((num-calls (car v))
	 (ticks (cadr v))
	 (ms (round (* 1000.0 (/ ticks internal-time-units-per-second)))))
	(format t "~d	~d	~d	~s~%" num-calls ms 
		(round (/ ms num-calls)) k)))

(defun calls (kv-pair)
  (let* ((v (cdr kv-pair))
	 (num-calls (car v)))
	num-calls))

(defun total-time (kv-pair)
  (let* ((v (cdr kv-pair))
	 (ticks (cadr v)))
	ticks))

(defun average-time (kv-pair)
  (let* ((v (cdr kv-pair))
	 (num-calls (car v))
	 (ticks (cadr v)))
	(/ ticks num-calls)))

(defun average-time-sort (a b)
  (> (average-time a) (average-time b)))

(defun total-time-sort (a b)
  (> (total-time a) (total-time b)))

(defun calls-sort (a b)
  (> (calls a) (calls b)))

(defun method-call-report (&key (number-to-report 1000000)
				(sort-by :total-time))
  (let ((count 0)
	(sorted-list nil)
	(test (cond 
	       ((eq sort-by :average-time) #'average-time-sort)
	       ((eq sort-by :calls) #'calls-sort)
	       (t #'total-time-sort))))
       ;; Build up the list and count the items...
       (maphash #'(lambda (k v)
			  (push (cons k v) sorted-list)
			  (incf count)) *call-hashtab*)
       (setq number-to-report (min count number-to-report))
       ;; Sort the list by the correct item
       (setq sorted-list (sort sorted-list test))
       ;; Print it out!
       (format t "Method names are (Name Qualifiers Classes)~%")
       (format t "All times reported in milleseconds~%~%")
       (format t "Times	Total	Avg	Method~%")
       (format t "Called	Time	Time	Name~%")
       (format t "--------------------------------~%")
       (do* ((i 0 (1+ i))
	     (sl sorted-list (cdr sl))
	     (item (car sl) (car sl)))
	    ((or (>= i number-to-report) (null sl)))
	    (report-one (car item) (cdr item)))))

(defun method-call-clear ()
  (clrhash *call-hashtab*))

(export '(*profile* start-call end-call method-call-report method-call-clear))

(in-package 'pcl :use '(lisp))

;;;
;;; Return the classes in a spec-ll
;;;
(defun classes-of-spec-ll (spec-ll)
  (let ((rv nil))
       (dolist (spec spec-ll)
	       (if (consp spec)
		   (push (cadr spec) rv)
		   (if (member spec lambda-list-keywords)
		       (return))))
       (nreverse rv)))

(defmacro doc-or-decl-p (form)
  "Return t if argument is a declaration or a doc-string."
  `(or (stringp ,form) (and (consp ,form) (eq (car ,form) 'declare))))

(defmacro skip-decl-doc (body)
  "Skip over the declarations and doc-strings in a body of code."
  `(do ()
       ((or (null ,body) (not (doc-or-decl-p (car ,body)))))
       (setq ,body (cdr ,body))))

(defun add-profile-code (name qualifiers spec-ll cdr-of-form)
  (let ((nqc (list name qualifiers (classes-of-spec-ll spec-ll)))
	(start-of-body cdr-of-form)
	(doc-decl nil))
       ;; March down the declarations/doc-string and body and separate
       ;; the two into doc-decl and start-of-body.
       (skip-decl-doc start-of-body)
       (if (and (null start-of-body) (stringp (last cdr-of-form)))
	   (setq start-of-body (last cdr-of-form)))
       (unless (eq start-of-body cdr-of-form)
	       (do ()
		   ((eq (cdr cdr-of-form) start-of-body))
		   (push (car cdr-of-form) doc-decl)
		   (setq cdr-of-form (cdr cdr-of-form)))
	       (push (car cdr-of-form) doc-decl)
	       (setq doc-decl (nreverse doc-decl)))
       `(,@doc-decl
	 (unwind-protect
	  (progn
	   (if prof:*profile* (prof:start-call ',nqc))
	   ,@start-of-body)
	  (if prof:*profile* (prof:end-call ',nqc)))
	 )))

;;;
;;; Change this pcl function for profiling...
;;;
(defun parse-defmethod (cdr-of-form)
  (declare (values name qualifiers specialized-lambda-list body))
  (let ((name (pop cdr-of-form))
	(qualifiers ())
	(spec-ll ()))
       (loop (if (and (car cdr-of-form) (symbolp (car cdr-of-form)))
		 (push (pop cdr-of-form) qualifiers)
		 (return (setq qualifiers (nreverse qualifiers)))))
       (setq spec-ll (pop cdr-of-form))
       (if (member :profile *features*)
	   (setq cdr-of-form
		 (add-profile-code name qualifiers spec-ll cdr-of-form)))
       (values name qualifiers spec-ll cdr-of-form)))

;;;
;;; Top-level new-instance function
;;;
(defmethod new-instance ((self t) &key (ignore nil) &allow-other-keys)
  (declare (ignore ignore))
  ;; do nothing
  self)

;;;
;;; Initialize an object
;;;
(defun make-instance (class &rest init-plist)
  (when (symbolp class) (setq class (find-class class)))
  (let ((object (allocate-instance class)))
       (initialize object init-plist)
       (apply #'new-instance object init-plist)
       object))

;;;
;;; Pretty print an instance
;;;
(defmethod ppi (self
		&optional
		(stream *standard-output*)
		(level nil))
  (let* ((class (class-of self))
	 (all-slots (class-slots class)))
    (format stream "~A CLASS~%" (class-name (class-of self)))
    (when (not (null all-slots))
	  (format stream "~8,8tVARIABLES:~%")
	  (dolist (c all-slots)
		  (format stream "~16,8t~a~40,8t" (slotd-name c))
		  (write (slot-value self (slotd-name c))
			 :stream stream
			 :circle nil
			 :pretty t
			 :level level
			 :length 1)
		  (terpri)
		  (force-output)))
    )
  self)

;;;
;;; Change print function to print out named objects better.
;;;
(defmethod print-object ((self t) stream)
  (if (and (slot-exists-p self 'pt::name)
	   (slot-boundp self 'pt::name))
      (format stream "<~s ~s ~o>" 
	      (class-name (class-of self))
	      (slot-value self 'pt::name)
	      (excl::pointer-to-fixnum self))
      (format stream "<~s ~o>" 
	      (class-name (class-of self))
	      (excl::pointer-to-fixnum self))))

;;;
;;; exports from PCL
;;;

(export '(new-instance
	  ppi)
	(find-package 'pcl))
