;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/x-window.cl,v $
;;; $Revision: 1.3 $
;;; $Date: 90/07/25 17:25:38 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; event-map methods
;;;

(defun find-entry (table dlist)
  "Return the map entry for the specified descriptor from the specified map"
  ;; declare arguments
  (declare (type hash-table table) (list dlist))
  ;; look for match ...
  (cond
   ;; ... table empty -- no match
   ((null table)
    (return-from find-entry nil))
   ;; ... on full specification
   ((gethash dlist table))
   ;; ignoring state
   ((gethash (list (first dlist) (third dlist)) table))
   ;; ... ignoring detail
   ((gethash (list (first dlist) (second dlist)) table))
   ;; ignoring state and detail
   ((gethash (first dlist) table))))

(defun setf-find-entry (table dlist value)
  "Set the map entry for the specified descriptor from the specified map"
  ;; declare arguments
  (declare (type hash-table table) (list dlist) (t value))
  ;; determine if
  (cond
   ;; ... state and detail ignored
   ((and (zerop (second dlist)) (eq (third dlist) 0))
    (setf (gethash (first dlist) table) value))
   ;; ... state ignored
   ((zerop (second dlist))
    (setf (gethash (list (first dlist) (third dlist)) table) value))
   ;; ... detail ignored
   ((eq (third dlist) 0)
    (setf (gethash (list (first dlist) (second dlist)) table) value))
   ;; ... nothing ignored
   ((setf (gethash dlist table) value))))

(defsetf find-entry setf-find-entry)

;;;
;;; x-window hash table utilities
;;;

(defun window-hash-table (display &aux wt)
  "return the window hash table or create it if it is nil"
  ;; test display 
  (setq wt (if display (slot-value display 'window-table) nil))
  (if display
      ;; test current window table
      (if wt
	  ;; return window table of current display
	  wt
	  ;; make the hash table
	  (progn 
	   (setf (gethash (res display) *global-window-table-hashtab*)
		 (setf (slot-value display 'window-table) (make-hash-table)))))
      ;; signal error
      (error "window-hash-table: no current display.")))

(defun append-window (w &aux display)
  "Append the window object to the hash table"
  (setq display (slot-value w 'display))
  (setf (gethash (res w) (window-hash-table display)) w)
  (do ((sup (parent w) (parent w)))
      ((or (root-window-p sup) (null sup)))
      (setq w sup)))

(defun delete-window (window-res &optional (display (current-display)))
  "return the window object with the specified resource"
  (remhash window-res (window-hash-table display)))

;;;
;;; x-window keywords
;;;

(defvar :tile-mode-absolute "tile with respect to window origin")
(defvar :tile-mode-relative "tile with respect to parent origin")
(defvar :clip-mode-clipped "suppress output into areas covered by children")
(defvar :clip-mode-drawthru "output into areas covered by children")

;;;
;;; x-window class
;;;

(defclass x-window (window)
  ((name
    :initform "An X-Window"
    :type string)
   (cursor
    :initarg :cursor 
    :initform nil
    :type cursor
    :reader cursor)
   (event-mask
    :initarg :event-mask 
    :initform '( :no-event )
    :type list
    :reader event-mask)
   (event-map
    :initform nil
    :type event-map
    :accessor event-map)
   (class-event-map
    :initform nil
    :type event-map
    :accessor class-event-map
    :allocation :class)
   (position-specified
    :initform nil
    :type atom)
   (mf-selectable-widget :initform nil)
   (intern-gc :initform nil)))

(defun make-x-window (&rest keys)
  (apply #'make-instance 'x-window :allow-other-keys t keys))

;;;
;;; event mapping for x-windows
;;;

(defmethod make-event-map ((self x-window))
  "Return the class-event-map for the window class"
  ;; attemp to locate event-map
  (let ((map (event-map self)))
       ;; test if map has been created
       (if (or (null map) (not (hash-table-p map)))
	   ;; create new map
	   (setf (event-map self) (make-hash-table :test #'equal))
	   ;; return map
	   map)))

;;;
;;; event mapping functions
;;;

(defun make-class-event-map-entry (window &key (descriptor nil) 
					  (function nil functionp) 
					  &allow-other-keys)
  ;; insert mapping into table if function was specified and valid
  (if (and functionp (function-p function))
      ;; set function mapping
      (setf (find-entry (class-event-map window) descriptor) function)
      ;; signal error
      (error "no mapping specified for this event")))

(defun make-event-map-entry (window
				 &key
				 (descriptor nil)
				 (function nil functionp)
				 &allow-other-keys)
  (if (null (event-map window))
      (make-event-map window))

  ;; insert mapping into table if function was specified and valid
  (if (and functionp (function-p function))
      ;; set function mapping
      (setf (find-entry (event-map window) descriptor) function)
      ;; signal error
      (error "no mapping specified for this event")))

(defun convert-event-type (event-type)
  (case event-type
	(:enter-window :enter-notify)
	(:leave-window :leave-notify)
	(:expose-region :exposure)
	(t event-type)))

(defun convert-detail (event-type detail)
  (typecase detail
	    (keyword 
	     (case detail 
		   ((left-button button-1 :left-button :button-1) 1) 
		   ((middle-button button-2 :middle-button :button-2) 2) 
		   ((right-button button-3 :right-button :button-3) 3)
		   (t detail)))
	    (character detail) 
	    (string
	     (let ((char (character detail)))
		  (if char char detail)))
	    (null 0)
	    (t detail)))

(defun convert-state-key (key)
  (setq key (excl::make-keyword key))
  (case key 
	(:left-button :button-1)
	(:middle-button :button-2)
	(:right-button :button-3)
	(:meta :mod-1)
	(t key)))

(defun convert-state (state)
  (unless (listp state) (setq state (list state)))
  (mapcar #'convert-state-key state))

(defun register-callback (self callback event-type 
			       &key 
			       (state nil) 
			       (detail nil) 
			       &allow-other-keys)
  (if (null (event-map self))
      (make-event-map self))
  (let ((desc (list (convert-event-type event-type) 
		    (if state (apply #'xlib:make-state-mask 
				     (convert-state state)) 0)
		    (convert-detail event-type detail))))
       (make-event-map-entry self :function callback :descriptor desc)))

(defun lookup-event-mapping (window descriptor)
  "Return the mapping (if any) for the specified descriptor"
  ;; search the hash-table associated with the instance
  (declare (list descriptor))
  (let ((map nil)
	(em (event-map window)))
       (declare (type hash-table em) (t map))
       (if (null em)
	   (find-entry (class-event-map window) descriptor)
	   (progn
	    (setq map (find-entry em descriptor))
	    ;; test if mapping is null
	    (if map
		;; return map
		(return-from lookup-event-mapping map)
		;; test if the class has a mapping defined
		(find-entry (class-event-map window) descriptor))))))

;;;
;;; x-window initialization method
;;;

(defmethod new-instance ((self x-window)
			 &key
			 (cursor nil)
			 x-offset y-offset region location
			 &allow-other-keys)
  "Initialize a new instance of the x-window class"
  (declare (ignore cursor))
  (call-next-method)

  (if (or x-offset y-offset region location)
      (setf (slot-value self 'position-specified) t))

  ;; make the class event mapping
  (if (null (class-event-map self))
      (make-class-event-map self))

  ;; return self
  self)

;;;
;;; Setf methods for slots in the x-window class
;;;

(defmethod (setf x-offset) (value (self window))
  (configure self :x-offset value)
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf y-offset) (value (self window))
  (configure self :y-offset value)
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf location) (value (self window))
  (configure self :x-offset (car value) :y-offset (cadr value))
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf region) (value (self window))
  (configure self 
	     :x-offset (first value) 
	     :y-offset (second value)
	     :width (third value)
	     :height (fourth value))
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf parent) (value (self x-window))
  (if (attached-p self)
      (progn
       (unless (window-p value)
	       (error "x-window.setf.parent: invalid window \`~s\`." value))
       (detach self)
       (when (attached-p value)
	     (attach self))
       (when (parent self)
	     (delete-child (parent self) self)
	     (setf (slot-value self 'parent) nil))
       (when value
	     (setf (slot-value self 'parent) value)
	     (add-child value self))
       (update-state self))
      (call-next-method)))

(defun convert-keyword-list-to-mask (kwlist)
  ;; allocate return value
  (let ((value 0))
       ;; get value for all event keywords
       (dolist (kw kwlist)
	       ;; add value to event mask
	       (setq value (logior value (keyword-value kw))))
       ;; return value
       value))

(defun parse-event-mask (em)
  (if (member :no-event em) 
      nil
      (substitute :exposure :expose-region em)))

(defun final-parse-event-mask (em)
  (mapcan #'(lambda (x) (when (typep x 'xlib::event-mask-class)
			      (list x)))
	  em))

(defmethod (setf event-mask) ((value list) (self x-window))
  ;; if :no-event is specified w/ other event keywords
  ;; remove :no-event from list
  (setq value (parse-event-mask value))
  (if (and (> (length value) 1) (member :no-event value))
      (setq value (delete :no-event value)))

  ;; if :all-events is specified w/ other event keywords
  ;; remove :all-events from list
  (if (and (> (length value) 1) (member :all-events value))
      (setq value (delete :all-events value)))

  ;; create clx event-mask 
  (if (attached-p self)
      (setf (xlib:window-event-mask (res self)) 
	    (if value
		(apply #'xlib:make-event-mask 
		       (final-parse-event-mask value))
		0)))
  (setf (slot-value self 'event-mask) value))

(defmethod (setf event-mask) (value (self x-window))
  ;; call setf with list of value
  (setf (event-mask self) (list value)))

(defmethod set-resize-hint ((self x-window) 
			    &key
			    (base-width (base-width self))
			    (base-height (base-height self))
			    (width-increment (width-increment self))
			    (height-increment (height-increment self))
			    &allow-other-keys)
  (call-next-method)
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties
       (res self)
       :name (name self)
       :min-width base-width
       :min-height base-height
       :width-inc width-increment
       :height-inc height-increment)))

(defmethod fix-location ((self x-window)
			 &key x y)
  (unless x (setq x (x-offset self)))
  (unless y (setq y (y-offset self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-position-p t
				    :x x :y y)))

(defmethod fix-size ((self x-window)
		     &key width height)
  (unless width (setq width (width self)))
  (unless height (setq height (height self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-size-p t
				    :width width :height height)))

(defmethod fix-region ((self x-window)
		       &key x y width height)
  (unless x (setq x (x-offset self)))
  (unless y (setq y (y-offset self)))
  (unless width (setq width (width self)))
  (unless height (setq height (height self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-position-p t
				    :user-specified-size-p t
				    :x x :y y
				    :width width :height height)))

(defmethod (setf size) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self (car val) (cadr val))
      (call-next-method)))

(defmethod (setf width) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self val (height self))
      (call-next-method)))

(defmethod (setf height) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self (width self) val)
      (call-next-method)))

(defmethod (setf cursor) (value (self x-window) &aux old)
  ;; test cursor 
  (unless (cursor-p value) 
	  (error "x-window.setf.cursor: invalid cursor \`~s\`." value))
  (when (cursor-p (setq old (slot-value self 'cursor)))
	(detach old))
  (when (attached-p self)
	(attach value)
	(setf (xlib:window-cursor (res self)) (res value)))
  (setf (slot-value  self 'cursor) value))

;;;	*** optimize configure? ***
(defmethod configure ((self x-window)
		      &key
		      (x-offset 0 x-offset-p)
		      (y-offset 0 y-offset-p)
		      &allow-other-keys)
  (call-next-method)
  (if (attached-p self)
      (let* ((sup (parent self)) 
	     (res (res self))
	     (rwp (root-window-p sup)))
	    (let
	     ((x (if rwp 
		     (if x-offset-p x-offset (xlib:drawable-x res))
		     (+ (x-offset self) (repaint-x sup))))
	      (y (if rwp 
		     (if y-offset-p y-offset (xlib:drawable-y res))
		     (+ (y-offset self) (repaint-y sup))))
	      (w (width self))
	      (h (height self)))
	     (if (and (> w 0) (> h 0))
		 (xlib:with-state (res) 
		    (setf (xlib:drawable-x res) x 
			  (xlib:drawable-y res) y 
			  (xlib:drawable-width res) w 
			  (xlib:drawable-height res) h)))))))

(defmethod repaint-x ((self x-window))
  0)

(defmethod repaint-y ((self x-window))
  0)

(defun server-x-offset (self)
  (if (root-window-p (parent self)) (query-region self))
  (+ (slot-value self 'x-offset) (repaint-x (parent self))))

(defun server-y-offset (self)
  (if (root-window-p (parent self)) (query-region self))
  (+ (slot-value self 'y-offset) (repaint-y (parent self))))

;;;
;;; Window query methods
;;;
(defun query-region (self)
  ;; query the server and cache the results
  (when (attached-p self)
	(let* ((res (res self))
	       (x (xlib:drawable-x res))
	       (y (xlib:drawable-y res)))
	      (multiple-value-setq (x y)
				   (xlib:translate-coordinates
				    res x y (res (root-window))))
	      (setf (slot-value self 'x-offset) x
		    (slot-value self 'y-offset) y
		    (slot-value self 'width) (xlib:drawable-width res) 
		    (slot-value self 'height) (xlib:drawable-height res))))
  ;; let region method construct the region
  (region self))

(defun root-coords (x y win)
  (xlib:translate-coordinates (res win) x y (res (root-window))))

;;;
;;; window methods that effect the windows status
;;;

;;	*** optimize do-detach (event-cond) ***
(defmethod do-detach ((self x-window) &aux res)
  ;; detach foreground, background, etc.
  (setq res (res self))
  (setf (slot-value self 'res) nil)
  (call-next-method)
  ;; detach cursor
  (if (cursor self)
      (detach (cursor self)))
  ;; Kill any pending events for this window.
  (xlib:display-finish-output (res (display self)))
  (event-sync :display (display self) :windows self :discard-p t)
  ;; Clean up any shared gc's
  (cleanup-shared-gcs self)
  ;; destroy x resource
  (xlib:destroy-window res)
  ;; delete window from the hash table
  (delete-window res)
  ;; set the resource to nil
  (flush-display (display self)))

(defmethod do-pend ((self x-window))
;;  (call-next-method) replaced below
  (setf (state self) (logior (state self) *pended*))
  (if (attached-p self)
      (xlib:unmap-window (res self))))

(defmethod do-unpend ((self x-window))
;;  (call-next-method) replaced below
  (setf (state self) (logandc2 (state self) *pended*))
  (when (exposed-p self)
      (xlib:map-window (res self))))

;;	*** incomplete do-conceal: can't handle 'transparent' keyword ***
(defmethod do-conceal ((self x-window) 
		       &key 
		       (transparent nil) 
		       (x-unmap t) 
		       &allow-other-keys)
  ;;  map resource
  (when (and x-unmap (attached-p self))
	(xlib:unmap-window (res self)))
  
  (call-next-method))

(defmethod do-expose ((self x-window) 
		      &key 
		      (x-map t) 
		      &allow-other-keys) 
  (call-next-method)

  ;;  map resource
  (if (and x-map (exposed-p self))
      (xlib:map-window (res self))))

(defmethod do-make-invisible ((self x-window)
			      &key (x-unmap t))
;;  (call-next-method) replaced below
  (setf (state self) (logior (state self) *not-visible*))

  ;;  unmap resource
  (if (and (attached-p self) x-unmap)
      (xlib:unmap-window (res self))))

(defmethod do-make-uninvisible ((self x-window))
;;  (call-next-method) replaced below
  (setf (state self) (logandc2 (state self) *not-visible*))

  ;;  map resource
  (if (exposed-p self)
      (xlib:map-window (res self))))

(defmethod conceal-inferiors ((self x-window))
  (if (attached-p self)
      (xlib:unmap-subwindows (res self))))

(defmethod expose-inferiors ((self x-window))
  (if (attached-p self)
      (xlib:map-subwindows (res self))))

;;	XXXXXXX  cant find clx functions raise and lower XXXXXXX
(defmethod raise ((self x-window))
  "Raise the specified window instance to the top of the occlusion stack"
  (if (attached-p self)
      (xlib:circulate-window-up (res self))))

(defmethod lower ((self x-window))
  "Lower the specified window instance to the bottom of the occlusion stack"
  (if (attached-p self)
      (xlib:circulate-window-down (res self))))

(defmethod circle-up ((self x-window))
  (if (attached-p self)
      (xlib:circulate-window-up (res self))))

(defmethod circle-down ((self x-window))
  (if (attached-p self)
      (xlib:circulate-window-down (res self))))

(defmethod warp-mouse ((self x-window)
		       &key
		       (x 0)
		       (y 0)
		       (location nil))
  ;; test if location is specified
  (if (attached-p self)
      (if location
	  (xlib:warp-pointer (res self) (first location) (second location))
	  (xlib:warp-pointer (res self) x y))))

(defmethod warp-mouse-if ((self x-window)
			  &key
			  (x 0)
			  (y 0)
			  (location nil)
			  (in-window nil)
			  (in-region nil))
  ;; if region isn't specified, get in-window region
  (if (null in-region)
      (setq in-region (region in-window)))
  
  ;; if location is specified, warp mouse using location
  ;; otherwise warp mouse using x and y
  (if (attached-p self)
      (if location
	  (xlib:warp-pointer-if-inside (res self)
				 (first location) (second location)
				 (res in-window)
				 (first in-region) (second in-region)
				 (third in-region) (fourth in-region))
	  (xlib:warp-pointer-if-inside (res self)
				 x y
				 (res in-window)
				 (first in-region) (second in-region)
				 (third in-region) (fourth in-region)))))

(defmethod grab-mouse ((self x-window)
		       &key
		       (cursor nil cursor-p)
		       (event-mask nil))
  (unless (attached-p self)
	  (return-from grab-mouse nil))
  
  ;; validate specified cursor
  (if (not (and cursor-p (cursor-p cursor)))
      ;; if cursor defined for window, use window cursor
      (if (and (cursor self))
	  (setq cursor (cursor self))
	  (error "window.grab-mouse: no cursor defined for grab")))
  
  ;; allocate event mask value
  ;; validate specified event mask
  (when (not (and event-mask (listp event-mask)))
	;; if event mask defined for window, use it
	(if (and (event-mask self) (listp (event-mask self)))
	    (setq event-mask (event-mask self))
	    (error "window.grab-mouse: no event mask defined for grab")))
  
  ;; convert event mask to integer value
  
  ;; attempt to grab mouse
  (unless (xlib:grab-pointer (res self) event-mask :cursor (res cursor))
	  (warn "window.grab-mouse: couldn't grab mouse"))
  self)

;;;
;;; determine if two windows are children of the same root-window
;;;

(defmethod related-p ((self x-window) (parent x-window))
  "Determine if two windows are children of the same root-window."
  (eq (do ((pw (parent self) (parent pw))
	   (w self pw))
	  ((null pw) w))
      (do ((pw (parent parent) (parent pw))
	   (w parent pw))
	  ((null pw) w))))

;;;
;;; window functions
;;;

(defun ungrab-mouse (&optional (display (current-display) displayp))
  ;; test is display specified
  (if displayp
      ;; test if specified display is valid
      (if (display-p display) 
	  (xlib:ungrab-pointer (res display))
	  ;; invalid display
	  (error "force-window-output: invalid display ~s" display))
      ;; use current display
      (when display
	    (xlib:ungrab-pointer (res display))))
  ;; return nil
  nil)
