;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/x-window-def.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/25 17:24:48 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; x-window keywords
;;;

(defvar :tile-mode-absolute "tile with respect to window origin")
(defvar :tile-mode-relative "tile with respect to parent origin")
(defvar :clip-mode-clipped "suppress output into areas covered by children")
(defvar :clip-mode-drawthru "output into areas covered by children")

;;;
;;; x-window class
;;;

(defclass x-window (window)
  ((name
    :initform "An X-Window"
    :type string)
   (cursor
    :initarg :cursor 
    :initform nil
    :type cursor
    :reader cursor)
   (event-mask
    :initarg :event-mask 
    :initform '( :no-event )
    :type list
    :reader event-mask)
   (event-map
    :initform nil
    :type event-map
    :accessor event-map)
   (class-event-map
    :initform nil
    :type event-map
    :accessor class-event-map
    :allocation :class)
   (position-specified
    :initform nil
    :type atom)
   (mf-selectable-widget :initform nil)
   (intern-gc :initform nil)))

(defun make-x-window (&rest keys)
  (apply #'make-instance 'x-window :allow-other-keys t keys))


;;;
;;; Setf methods for slots in the x-window class
;;;

(defmethod (setf x-offset) (value (self window))
  (configure self :x-offset value)
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf y-offset) (value (self window))
  (configure self :y-offset value)
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf location) (value (self window))
  (configure self :x-offset (car value) :y-offset (cadr value))
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf region) (value (self window))
  (configure self 
	     :x-offset (first value) 
	     :y-offset (second value)
	     :width (third value)
	     :height (fourth value))
  (if (root-window-p (parent self))
      (setf (slot-value self 'position-specified) t))
  value)

(defmethod (setf parent) (value (self x-window))
  (if (attached-p self)
      (progn
       (unless (window-p value)
	       (error "x-window.setf.parent: invalid window \`~s\`." value))
       (detach self)
       (when (attached-p value)
	     (attach self))
       (when (parent self)
	     (delete-child (parent self) self)
	     (setf (slot-value self 'parent) nil))
       (when value
	     (setf (slot-value self 'parent) value)
	     (add-child value self))
       (update-state self))
      (call-next-method)))

(defun convert-keyword-list-to-mask (kwlist)
  ;; allocate return value
  (let ((value 0))
       ;; get value for all event keywords
       (dolist (kw kwlist)
	       ;; add value to event mask
	       (setq value (logior value (keyword-value kw))))
       ;; return value
       value))

(defun parse-event-mask (em)
  (if (member :no-event em) 
      nil
      (substitute :exposure :expose-region em)))

(defun final-parse-event-mask (em)
  (mapcan #'(lambda (x) (when (typep x 'xlib::event-mask-class)
			      (list x)))
	  em))

(defmethod (setf event-mask) ((value list) (self x-window))
  ;; if :no-event is specified w/ other event keywords
  ;; remove :no-event from list
  (setq value (parse-event-mask value))
  (if (and (> (length value) 1) (member :no-event value))
      (setq value (delete :no-event value)))

  ;; if :all-events is specified w/ other event keywords
  ;; remove :all-events from list
  (if (and (> (length value) 1) (member :all-events value))
      (setq value (delete :all-events value)))

  ;; create clx event-mask 
  (if (attached-p self)
      (setf (xlib:window-event-mask (res self)) 
	    (if value
		(apply #'xlib:make-event-mask 
		       (final-parse-event-mask value))
		0)))
  (setf (slot-value self 'event-mask) value))

(defmethod (setf event-mask) (value (self x-window))
  ;; call setf with list of value
  (setf (event-mask self) (list value)))

(defmethod set-resize-hint ((self x-window) 
			    &key
			    (base-width (base-width self))
			    (base-height (base-height self))
			    (width-increment (width-increment self))
			    (height-increment (height-increment self))
			    &allow-other-keys)
  (call-next-method)
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties
       (res self)
       :name (name self)
       :min-width base-width
       :min-height base-height
       :width-inc width-increment
       :height-inc height-increment)))

(defmethod fix-location ((self x-window)
			 &key x y)
  (unless x (setq x (x-offset self)))
  (unless y (setq y (y-offset self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-position-p t
				    :x x :y y)))

(defmethod fix-size ((self x-window)
		     &key width height)
  (unless width (setq width (width self)))
  (unless height (setq height (height self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-size-p t
				    :width width :height height)))

(defmethod fix-region ((self x-window)
		       &key x y width height)
  (unless x (setq x (x-offset self)))
  (unless y (setq y (y-offset self)))
  (unless width (setq width (width self)))
  (unless height (setq height (height self)))
  (if (and (attached-p self) (root-window-p (parent self)))
      (xlib:set-standard-properties (res self) 
				    :user-specified-position-p t
				    :user-specified-size-p t
				    :x x :y y
				    :width width :height height)))

(defmethod (setf size) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self (car val) (cadr val))
      (call-next-method)))

(defmethod (setf width) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self val (height self))
      (call-next-method)))

(defmethod (setf height) (val (self x-window))
  (if (root-window-p (parent self))
      (root-shape self (width self) val)
      (call-next-method)))

(defmethod (setf cursor) (value (self x-window) &aux old)
  ;; test cursor 
  (unless (cursor-p value) 
	  (error "x-window.setf.cursor: invalid cursor \`~s\`." value))
  (when (cursor-p (setq old (slot-value self 'cursor)))
	(detach old))
  (when (attached-p self)
	(attach value)
	(setf (xlib:window-cursor (res self)) (res value)))
  (setf (slot-value  self 'cursor) value))
