;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/window-def.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/31 18:57:50 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; Window states: A window can be in one of three states: exposed,
;;; concealed or pending. These are returned by the status method. If
;;; a window is pending, it can be for one of three reasons: 
;;;
;;;	a) it's not attached
;;;	b) it's been shrunk to zero size by a geometry manager("pended")
;;;	c) it's parent is not visible.
;;;
;;; These are orthogonal blocks -- they can be caused by the server, the
;;; geometry manager or the application, respectively. We can use the
;;; attached-p macro to detect conditiopn (a), and (b) and (c) are 
;;; stored as a bit-vector in the window "state" slot. The "status" slot
;;; holds one of :exposed or :concealed, and the status method returns one
;;; of :concealed, :exposed or :pending by combining the values stored in
;;; the state and status slots appropriately.
;;;
;;; The macros "invisible-p", "exposed-p", "attached-p", "pending-p",
;;; "concealed-p", "pended-p" should be used to determine the status of
;;; a window.
;;;

(defvar :pending :pending "window is exposed, but not visible for other reason")
(defvar :exposed :exposed "window is exposed")
(defvar :concealed :concealed "window is concealed (not exposed)")

;;;
;;; window class
;;;

(defclass window ()
  (
   ;; Information to support the window-hierarchy

   (res 
    :initform nil
    :type vector
    :reader res)
   (parent
    :initarg :parent 
    :initform nil
    :type window
    :accessor parent)
   (display
    :initarg :display 
    :initform nil
    :type display
    :reader display)
   (screen
    :initarg :screen 
    :initform nil
    :type screen
    :reader screen)
   (lexical-parent
    :initarg :lexical-parent 
    :initform nil
    :type window
    :accessor lexical-parent)
   
   ;; Information about a window geometry. The setf methods on these slots
   ;; cause a repaint method to be invoked.
   (x-offset
    :initarg :x-offset 
    :initform 0
    :type integer
    :reader x-offset)
   (y-offset
    :initarg :y-offset 
    :initform 0
    :type integer
    :reader y-offset)
   (width
    :initarg :width 
    :initform 1
    :type integer
    :reader width)
   (height
    :initarg :height 
    :initform 1
    :type integer
    :reader height)
   
   ;; The windows resize-hint.  Used by geometry managers. The setf methods
   ;; on these slots cause the resize-hint-changed method to be invoked.
   (base-width
    :initarg :base-width 
    :initform 1
    :type integer
    :reader base-width)
   (base-height
    :initarg :base-height 
    :initform 1
    :type integer
    :reader base-height)
   (width-increment
    :initarg :width-increment 
    :initform 1
    :type integer
    :reader width-increment)
   (height-increment
    :initarg :height-increment 
    :initform 1
    :type integer
    :reader height-increment)
   (width-height-ratio
    :initarg :width-height-ratio 
    :initform nil
    :type float
    :reader width-height-ratio)
   
   ;; Drawing information about a window.  Used by most any drawing function.
   (gc-spec			;; list of gc-specifications where each spec
    :initform nil		;; is (<slot-name> . <gc-description>).  The
    :type list			;; described gc is created and stuffed into
    :reader gc-spec		;; the named slot automatically when the window
    :allocation :class)		;; is attached.

   (gc-res			;; xlib gc to use for all graphics ops
    :initform nil		;; since a window doesn't have a resource
    :type vector		;; of its own (unless it's an x-window)
    :reader gc-res)
   (font 
    :initarg :font  
    :initform nil
    :type t
    :accessor font)
   (background
    :initarg :background 
    :initform "white"
    :type color
    :reader background)
   (inverted-background
    :initarg :inverted-background 
    :initform "black"
    :type color
    :reader inverted-background)
   (dimmed-background
    :initarg :dimmed-background 
    :initform "white"
    :type color
    :reader dimmed-background)
   (foreground
    :initarg :foreground 
    :initform "black"
    :type color
    :reader foreground)
   (dimmed-foreground
    :initarg :dimmed-foreground 
    :initform "gray50"
    :type color
    :reader dimmed-foreground)
   (inverted-foreground
    :initarg :inverted-foreground 
    :initform "white"
    :type color
    :reader inverted-foreground)

   (dimmed
    :initarg :dimmed 
    :initform nil
    :type t
    :accessor dimmed)
   (inverted
    :initarg :inverted 
    :initform nil
    :type t
    :accessor inverted)
   (colormap
    :initarg :colormap 
    :initform nil 
    :type colormap
    :accessor colormap)
   
   ;; Info to optimize repainting...
   (repaint-flag
    :initarg :repaint-flag 
    :initform t
    :type t
    :accessor repaint-flag)
   (partial-repaint-p
    :initform nil
    :type atom
    :reader partial-repaint-p)
   (intern-res-cache
    :initform nil
    :type list)
   (intern-gc
    :initform t
    :type vector)
   (cached-background
    :initform nil
    :type integer)

   ;; Misc information.
   (value
    :initarg :value
    :initform nil
    :type t
    :accessor value)
   (name
    :initarg :name 
    :initform "A Window"
    :type string
    :accessor name)
   (doc 
    :initarg :doc  
    :initform ""
    :type string
    :accessor doc)
   (objid
    :initarg :objid 
    :initform nil
    :type t
    :reader objid)
   (state
    :initarg :state 
    :initform 0
    :type integer
    :accessor state)
   (status
    :initarg :status 
    :initform :exposed
    :type symbol
    :accessor status)
   (attach-when-possible
    :initarg :attach-when-possible 
    :initform t
    :type atom
    :accessor attach-when-possible)
   (geom-spec
    :initarg :geom-spec 
    :initform nil
    :type t
    :accessor geom-spec)
   (mf-selectable-widget
    :initarg :mf-selectable-widget 
    :initform t
    :type atom
    :accessor mf-selectable-widget)

   ;; Label info
   (label
    :initarg :label 
    :initform nil
    :type t
    :reader label)
   ;; Currently One of (:left :frame)
   (label-type
    :initarg :label-type 
    :initform nil
    :type symbol
    :reader label-type)
   (label-x
    :initarg :label-x 
    :initform nil
    :type integer
    :reader label-x)
   (label-y
    :initarg :label-y 
    :initform nil
    :type integer
    :reader label-y)
   (label-attributes
    :initarg :label-attributes 
    :initform nil
    :type list
    :reader label-attributes)
   (intern-label-gc
    :initform t 
    :type vector)

   ;; Border info
   (border-width
    :initarg :border-width 
    :initform 0
    :type t
    :reader border-width)
   ;; One of (:box :frame :shadow :black-frame nil)
   (border-type
    :initarg :border-type 
    :initform nil
    :type symbol
    :reader border-type)
   (border-attributes
    :initarg :border-attributes 
    :initform nil
    :type list
    :reader border-attributes)
   (intern-border-gcs
    :initform t
    :type t)))

(defun make-window (&rest keys)
  (apply #'make-instance 'window :allow-other-keys t keys))

(defvar *objid-window-table* (make-hash-table :test #'equal))

;;;
;;; Setf methods for slots in the window class
;;;

(defmethod (setf status) (value (self window))
  (let ((old-status (managed-p self))
	(new-status nil))
       (setf (slot-value self 'status) value)
       (setq new-status (managed-p self))
       (if (not (eq new-status old-status))
	   (status-changed self))
       ;; return status
       new-status))

;;;
;;; Setf methods that affect the resize-hint.
;;;

(defmethod (setf base-width) (value (self window))
  (set-resize-hint self :base-width value)
  (slot-value self 'base-width))

(defmethod (setf base-height) (value (self window))
  (set-resize-hint self :base-height value)
  (slot-value self 'base-height))

(defmethod (setf width-increment) (value (self window))
  (set-resize-hint self :width-increment value)
  (width-increment self))

(defmethod (setf height-increment) (value (self window))
  (set-resize-hint self :height-increment value)
  (height-increment self))

(defmethod (setf base-size) (value (self window))
  (set-resize-hint self :base-width (first value)
		   :base-height (second value))
  (list (slot-value self 'base-width) (slot-value self 'base-height)))

(defmethod (setf increment-size) (value (self window))
  (set-resize-hint self :width-increment (first value)
		   :height-increment (second value))
  (increment-size self))

(defmethod (setf width-height-ratio) (value (self window))
  (set-resize-hint self :width-height-ratio value)
  (width-height-ratio self))

(defmethod (setf resize-hint) (value (self window))
  (set-resize-hint self 
		   :base-width (first value)
		   :base-height (second value)
		   :width-increment (third value)
		   :height-increment (fourth value)
		   :width-height-ratio (fifth value)))

(defmethod (setf geom-spec) :after (value (self window))
  (declare (ignore value))
  (geom-spec-changed self))

;;;
;;; Setf methods to futz with a windows geometry.
;;;

(defmethod (setf res) (value (self window))
  ;; warn
  (declare (ignore value))
  (warn "Can't manually change the resource of a window"))

(defmethod (setf x-offset) (value (self window))
  (configure self :x-offset value)
  value)

(defmethod (setf y-offset) (value (self window))
  (configure self :y-offset value)
  value)

(defmethod (setf width) (value (self window))
  (configure self :width value)
  value)

(defmethod (setf height) (value (self window))
  (configure self :height value)
  value)

(defmethod (setf location) (value (self window))
  (configure self
	     :x-offset (first value)
	     :y-offset (second value))
  value)

(defmethod (setf size) (value (self window))
  (configure self
	     :width (first value)
	     :height (second value))
  value)

(defmethod (setf region) (value (self window))
  (configure self
	     :x-offset (first value)
	     :y-offset (second value)
	     :width (third value)
	     :height (fourth value))
  value)

;;;
;;;	Setf methods on colors update gc-res if attached. . .
;;;

(defmethod (setf font) (value (self window))
  (cond ((attached-p self) 
	 (let ((old (slot-value self 'font))) 
	      (when (font-p old) 
		    (font-detach old))) 
	 (when value
	       (unless 
		(or (and (stringp value) 
			 (setq value (make-font :name value 
						:display (display self)))) 
		    (font-p value))
		(error "window.setf.font: invalid font ~s." value)) 
	       (setf (slot-value self 'font) value)
	       (font-attach value)
	       (let ((gc (gc-res self)))
		    (when (xlib:gcontext-p gc) 
			  (setf (xlib:gcontext-font gc) (res value))))))
	(t (if (or (stringp value) (font-p value) (null value))
	       (setf (slot-value self 'font) value)
	       (error "window.setf.font: invalid font ~s." value)))))

(defmethod (setf background) (val (self window) &aux gc old attp) 
  (setq old (slot-value self 'background)
	attp (attached-p self))
  (if (xlib:gcontext-p (setq gc (slot-value self 'intern-gc)))
      (cond ((stringp val)
	     (if (not attp)
		 (setf (slot-value self 'background) val)
		 (if (setq old (get-paint val self))
		     (setf (background self) old)
		     (error "window.setf.background: can't find paint \`~s\`."
			    val))))
	    ((color-p val) 
	     (when attp
		   (color-attach val)
		   (setf (xlib:gcontext-foreground gc) (pixel val) 
			 (xlib:gcontext-fill-style gc) :solid)
		   (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
			 (setf (xlib:gcontext-background gc) (pixel val))
			 (xlib:force-gcontext-changes gc))
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'background) val)
	     (clear self))
	    ((integerp val)
	     (when attp
		   (setf (xlib:gcontext-foreground gc) val 
			 (xlib:gcontext-fill-style gc) :solid)
		   (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
			 (setf (xlib:gcontext-background gc) val)
			 (xlib:force-gcontext-changes gc))
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'background) val)
	     (clear self))
	    ((image-p val)
	     (setq val (make-tile :image val :window self))
	     (when attp
		   (setf (xlib:gcontext-tile gc) (res val)
			 (xlib:gcontext-fill-style gc) :tiled))
	     (setf (slot-value self 'background) val)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    ((tile-p val) 
	     (setf (window val) self)
	     (when attp
		   (setf (xlib:gcontext-stipple gc) (res val)
			 (xlib:gcontext-foreground gc) (pixel (foreground val))
			 (xlib:gcontext-background gc) (pixel (background val))
			 (xlib:gcontext-fill-style gc) :opaque-stippled))
	     (setf (slot-value self 'background) val)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    ((null val) 
	     (when (and attp (setq val (slot-value self 'cached-background))
			(xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
		   (setf (xlib:gcontext-background gc) val))
	     (setf (slot-value self 'background) nil)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    (t
	     (error "window.setf.background: invalid background \`~s\`." 
		    val)))
      (setf (slot-value self 'background) val)))

(defmethod (setf inverted-background) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self) (null value)
	(eq :parent-relative value))
    (setf (slot-value self 'inverted-background) value))
   (t
    (error "window.setf-inverted-background: invalid color ~s" value))))

(defmethod (setf dimmed-background) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self) (null value)
	(eq :parent-relative value))
    (setf (slot-value self 'dimmed-background) value))
   (t
    (error "window.setf-dimmed-background: invalid color ~s" value))))

(defmethod (setf foreground) (val (self window) &aux gc old attp) 
  (setq old (slot-value self 'foreground)
	attp (attached-p self))
  (if (xlib:gcontext-p (setq gc (gc-res self)))
      (cond ((stringp val)
	     (if (not attp)
		 (setf (slot-value self 'foreground) val)
		 (if (setq old (get-paint val self)) 
		     (setf (foreground self) old) 
		     (error 
		      "window.setf.foreground: couldn't find paint \`~s\`." 
		      val))))
	    ((color-p val) 
	     (when attp
		   (color-attach val)
		   (setf (xlib:gcontext-foreground gc) (pixel val) 
			 (xlib:gcontext-fill-style gc) :solid)
		   (xlib:force-gcontext-changes gc)
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'foreground) val))
	    ((integerp val)
	     (when attp
		   (setf (xlib:gcontext-foreground gc) val 
			 (xlib:gcontext-fill-style gc) :solid
			 (xlib:gcontext-tile gc) nil)
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'foreground) val))
	    ((image-p val)
	     (setq val (make-tile :image val :window self))
	     (when attp
		   (setf (xlib:gcontext-tile gc) (res val)
			 (xlib:gcontext-fill-style gc) :tiled))
	     (setf (slot-value self 'foreground) val)
	     (when (and attp old (paint-p old))
		   (do-detach old)))
	    ((tile-p val) 
	     (setf (window val) self)
	     (when attp
		   (setf (xlib:gcontext-stipple gc) (res val)
			 (xlib:gcontext-foreground gc) (pixel (foreground val))
			 (xlib:gcontext-background gc) (pixel (background val))
			 (xlib:gcontext-fill-style gc) :opaque-stippled))
	     (setf (slot-value self 'foreground) val) 
	     (when (and attp old (paint-p old))
		   (do-detach old)))
	    (t
	     (error "window.setf.foreground: invalid paint \`~s\`." val)))
      (setf (slot-value self 'foreground) val)))

(defmethod (setf inverted-foreground) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self))
    (setf (slot-value self 'inverted-foreground) value))
   (t
    (error "window.setf-inverted-foreground: invalid color ~s" value))))

(defmethod (setf dimmed-foreground) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self))
    (setf (slot-value self 'dimmed-foreground) value))
   (t
    (error "window.setf-dimmed-foreground: invalid color ~s" value))))

(defmethod (setf inverted) (val (self window) &aux sv)
  (setq sv (slot-value self 'inverted))
  (unless (or (and val sv) (and (null val) (null sv)))
	  (psetf (background self) (inverted-background self)
		 (inverted-background self) (background self))
	  (psetf (foreground self) (inverted-foreground self)
		 (inverted-foreground self) (foreground self)))
  (setf (slot-value self 'inverted) val))

(defmethod (setf dimmed) (val (self window) &aux sv)
  (setq sv (slot-value self 'dimmed))
  (unless (or (and val sv) (and (null val) (null sv)))
	  (psetf (foreground self) (dimmed-foreground self)
		 (dimmed-foreground self) (foreground self)))
  (setf (slot-value self 'dimmed) val)
  (when (and (attached-p self) (black-and-white-display-p (display self))
	     (xlib:gcontext-p (setq sv (slot-value self 'gc-res))))
	(setf (xlib:gcontext-function sv) 
	      (if val 8 2))))

;;;
;;;	Label accessors
;;;

(defmethod (setf label-type) (val (self window) &aux oldval)
  (unless (eq val (setq oldval (slot-value self 'label-type)))
	  (setf (slot-value self 'label-type) val)
	  (when (exposed-p self)
		(label-clear oldval self))
	  (label-init val self)
	  (when (exposed-p self)
		(label-repaint val self)
		(flush-display (display self)))))

(defmethod (setf label) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-x) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label-x)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-x) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-y) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label-y)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-y) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod label-position ((self window))
  (list (slot-value self 'label-x) (slot-value self 'label-y)))

(defmethod (setf label-position) (val (self window) &aux ltype oldval)
  (unless (equal val (setq oldval (list (slot-value self 'label-x) 
					(slot-value self 'label-y))))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-x) (car val)
		(slot-value self 'label-y) (cadr val))
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-attributes) (val (self window))
  (setf (slot-value self 'label-attributes) val)
  (label-notify-change (slot-value self 'label-type) self))

(defmethod label-font ((self window))
  (getf (slot-value self 'label-attributes) :font))

(defmethod (setf label-font) (val (self window) &aux sval)
  (setq sval (slot-value self 'label-attributes))
  (setf (getf sval :font) val)
  (setf (slot-value self 'label-attributes) sval)
  (label-notify-change (slot-value self 'label-type) self))

;;;
;;;	Border accessors
;;;

(defmethod (setf border-type) (val (self window) &aux oldval)
  (unless (eq val (setq oldval (slot-value self 'border-type)))
	  (setf (slot-value self 'border-type) val)
	  (when (attached-p self)
		(when (exposed-p self)
		      (border-clear oldval self))
		(border-init val self)
		(when (exposed-p self)
		      (border-repaint val self)
		      (flush-display (display self))))))

(defmethod gray ((self window))
  (eq (border-type self) :frame))

(defmethod (setf gray) (val (self window))
  (setf (border-type self)
	(if val :frame :box)))

(defmethod (setf border-width) (val (self window) 
				    &aux oldval sup lbw tbw rbw bbw btp)
  (setq oldval (slot-value self 'border-width)
	sup (parent self))
  (unless (eq val oldval) 
	  (unless (or (integerp val) (= (length val) 4))
		  (error "window.setf.border-width: invalid width \'~s\'" val))
	  (setf (slot-value self 'border-width) val)
	  (when (exposed-p self)
		(if (consp val)
		    (setq lbw (first val)
			  tbw (second val)
			  rbw (third val)
			  bbw (fourth val))
		    (setq lbw val
			  tbw val
			  rbw val
			  bbw val))
		(setq btp (border-type self))
		(border-clear btp self)
		(border-resize btp self)
		(unless (opaque-window-p self)
			(do-repaint self))
		(border-repaint btp self))))

#|(export '(attach-when-possible
	  background
	  base-height
	  base-size
	  base-width
	  border-type
	  dimmed
	  dimmed-background
	  dimmed-foreground
	  foreground
	  geom-spec
	  gray
	  height
	  height-increment
	  increment-size
	  inverted
	  inverted-background
	  inverted-foreground
	  location
	  name
	  region
	  resize-hint
	  size
	  state
	  status
	  parent
	  width
	  width-height-ratio
	  width-increment
	  window
	  x-offset
	  y-offset)
  (find-package 'pt))|#
