;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/event.cl,v $
;;; $Revision: 1.5 $
;;; $Date: 90/07/30 17:38:27 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; Set the the to FAST
;;;
;; (proclaim '(optimize (speed 3) (safety 0)))

;;;
;;; event descriptor macros
;;;

(defvar *descriptor-table* nil)

(defun default-descriptor (type args)
  (declare (ignore args)
	   (symbol type))
  (list type nil nil))

(defun exposure-descriptor (type args)
  (declare (ignore args type))
  (list :exposure nil nil))

(defun motion-descriptor (type args)
  (declare (type list args)
	   (ignore type))
  (list :pointer-motion nil (getf args :state)))

(defun key-descriptor (type args)
  (declare (type list args)
	   (type symbol type))
  (let ((state (getf args :state))) 
       (declare (type integer state))
       (list type state 
	     (or 
	      (xlib:keycode->character (getf args :display) (getf args :code) 0)
	      (getf args :code)))))

(defvar *button-event-time* 0)
(defvar *button-double-time* 0)
(defun button-press-descriptor (type args &aux newtime)
  (declare (type list args)
	   (type symbol type))
  (setq newtime (getf args :time))
  (if (< (- newtime *button-double-time*) 300) 
      (setq type :triple-click 
	    *button-double-time* 0
	    *button-event-time* 0)
      (progn 
       (if (< (- newtime *button-event-time*) 300)
	   (progn 
	    (setq type :double-click 
		 *button-double-time* *button-event-time*)
	    (setq *button-event-time* 0)))
       (setq *button-event-time* newtime)))
  (list type (getf args :state) (getf args :code)))

(defun button-release-descriptor (type args)
  (declare (type list args)
	   (type symbol type))
  (list type (getf args :state) (getf args :code)))

(defun focus-descriptor (type args)
  (declare (type list args)
	   (type symbol type))
  (list type nil (getf args :mode)))

(defun crossing-descriptor (type args)
  (declare (type list args)
	   (type symbol type))
  (list type (getf args :state) (getf args :kind)))

(defun message-descriptor (type args)
  (declare (type list args)
	   (type symbol type))
  (list type (getf args :data) (getf args :type)))

(eval-when (eval load)
  (setq *descriptor-table* (make-array 35))
  ;; exposure
  (setf (svref *descriptor-table* 12) #'default-descriptor
  ;; map-notify
	(svref *descriptor-table* 19) #'default-descriptor
  ;; unmap-notify
	(svref *descriptor-table* 18) #'default-descriptor
  ;; visibility-notify
	(svref *descriptor-table* 15) #'default-descriptor
  ;; reparent-notify
	(svref *descriptor-table* 21) #'default-descriptor
  ;; configure-notify
	(svref *descriptor-table* 22) #'default-descriptor
  ;; create-notify
	(svref *descriptor-table* 16) #'default-descriptor
  ;; destroy-notify
	(svref *descriptor-table* 17) #'default-descriptor
  ;; exposure
	(svref *descriptor-table* 12) #'exposure-descriptor
  ;; button-motion
	(svref *descriptor-table* 6) #'motion-descriptor
  ;; key-press, key-release
	(svref *descriptor-table* 2) #'key-descriptor
	(svref *descriptor-table* 3) #'key-descriptor
  ;; button-press, button-release
	(svref *descriptor-table* 4) #'button-press-descriptor
	(svref *descriptor-table* 5) #'button-release-descriptor
  ;; focus-in, focus-out
	(svref *descriptor-table* 9) #'focus-descriptor
	(svref *descriptor-table* 10) #'focus-descriptor
  ;; enter-notify, leave-notify
	(svref *descriptor-table* 7) #'crossing-descriptor
	(svref *descriptor-table* 8) #'crossing-descriptor
  ;; client-message
	(svref *descriptor-table* 33) #'message-descriptor))

(defun descriptor (args &aux func)
  ;; declare arguments
  (declare (type list args))
  (if (setq func (svref *descriptor-table* (getf args :event-code))) 
      (funcall func (getf args :event-key) args) 
      (let ((type (getf args :event-key))) 
	   (list type nil nil)
	   (warn "no descriptor defined for event-type ~S" type))))

;;;
;;; event dispatcher for CLX events
;;;

(defvar *window-cache* nil)
(defvar *window-cache-res* nil)

(defun dispatch-event (&rest args &key display event-window
			     &allow-other-keys &aux window)
  (setq window *window-cache*)
  (unless (eq event-window *window-cache-res*)
	  (setq window (quick-find-window event-window display))
	  (unless window (return-from dispatch-event))
	  (setq *window-cache* window
		*window-cache-res* (res window)))
  
  ;;  (setq event args)
  ;;  (format t "event: ~S~%" event)
  ;;  (format t "event-type: ~S~%" event-key)
  
  ;; build the descriptor for this event
  (let* ((descriptor (descriptor args))
	 (mapping nil)
	 (rv nil))
	(setq mapping (lookup-event-mapping window descriptor))
	(cond
	 ;; test if mapping is a function
	 ((function-p mapping)
	  ;; call the function with window and args
	  (setq rv (apply mapping (cons window args))))
	 ;; test for no mapping
	 ((null mapping)
	  ;; Check for double-click
	  (if (eq (car descriptor) :double-click)
	      (if (function-p 
		   (setq mapping 
			 (lookup-event-mapping 
			  window 
			  (rplaca descriptor :button-press))))
		  (apply mapping (cons window args))
		  (warn "null mapping for event ~s on window = ~s~%" 
			(rplaca descriptor :double-click) window))
	      ;; Warn user 
	      (warn "null mapping for event ~s on window = ~s~%" 
		    descriptor window)))))
  nil)

(defun dispatch-one-event (&rest args &key display event-window
				 &allow-other-keys &aux window)
  (setq window *window-cache*)
  (unless (eq event-window *window-cache-res*)
	  (setq window (quick-find-window event-window display))
	  (unless window (return-from dispatch-one-event))
	  (setq *window-cache* window
		*window-cache-res* (res window)))
  
  ;;  (setq event args)
  ;;  (format t "event: ~S~%" event)
  
  ;; build the descriptor for this event
  (let* ((descriptor (descriptor args))
	 (mapping nil)
	 (rv nil))
	(setq mapping (lookup-event-mapping window descriptor))
	(cond
	 ;; test if mapping is a function
	 ((function-p mapping)
	  ;; call the function with window and args
	  (setq rv (apply mapping (cons window args))))
	 ;; test for no mapping
	 ((null mapping)
	  ;; Check for double-click
	  (if (eq (car descriptor) :double-click)
	      (if (function-p 
		   (setq mapping 
			 (lookup-event-mapping 
			  window 
			  (rplaca descriptor :button-press))))
		  (apply mapping (cons window args))
		  (warn "null mapping for event ~s on window = ~s~%" 
			(rplaca descriptor :double-click) window))
	      ;; Warn user 
	      (warn "null mapping for event ~s on window = ~s~%" 
		    descriptor window)))))
  t)

(defun dispatch-event-special (windows mask discard-p &rest args 
				       &key display event-window event-key 
				       &allow-other-keys &aux window)
  (unless (or (eq mask t) (member event-key mask))
	  (return-from dispatch-event-special))
  (setq window *window-cache*)
  (unless (eq event-window *window-cache-res*)
	  (setq window (quick-find-window event-window display))
	  (unless window (return-from dispatch-event-special))
	  (setq *window-cache* window
		*window-cache-res* (res window)))
  
  (unless (or (eq windows t) (if (consp windows) 
				 (member window windows) 
				 (eq window windows)))
	  (return-from dispatch-event-special))
  (when discard-p (return-from dispatch-event-special t))
  
  ;; build the descriptor for this event
  (let* ((descriptor (descriptor args))
	 (mapping nil)
	 (rv nil))
	(setq mapping (lookup-event-mapping window descriptor))
	(cond
	 ;; test if mapping is a function
	 ((function-p mapping)
	  ;; call the function with window and args
	  (setq rv (apply mapping (cons window args))))
	 ;; test for no mapping
	 ((null mapping)
	  ;; Check for double-click
	  (if (eq (car descriptor) :double-click)
	      (if (function-p 
		   (setq mapping 
			 (lookup-event-mapping 
			  window 
			  (rplaca descriptor :button-press))))
		  (apply mapping (cons window args))
		  (warn "null mapping for event ~s on window = ~s~%" 
			(rplaca descriptor :double-click) window))
	      ;; Warn user 
	      (warn "null mapping for event ~s on window = ~s~%" 
		    descriptor window)))))
  nil)

;;;
;;;	Processes next/current event
;;;
(defun event-dispatch-current (&optional (display (current-display)))
  (unless (display-p display)
	  (error "event-dispatch-current: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-dispatch-current: cannot use closed display \`~s\`." display))
  (xlib:process-event (res display) :handler #'dispatch-one-event
		       :discard-p t :peek-p nil :force-output-p t :timeout 0))

;;;
;;;	Processes all events and "hangs" for new ones
;;;
(defun event-loop (&key (display (current-display)) 
			(handler #'dispatch-event)
			(hang t))
  (unless (display-p display)
	  (error "event-loop: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-loop: cannot use closed display \`~s\`." display))
  (xlib:process-event (res display) :handler handler
		       :discard-p t :peek-p nil :force-output-p t
		       :timeout (if hang nil 0)))

;;;
;;;	Processes all current events specified by windows, mask, and count.
;;;	Can be aborted by returning :abort from a handler.
;;;
(defun event-sync (&key (display (current-display))
			(handler #'dispatch-event-special handlerp)
			(windows t)
			(mask t)
			(count :all)
			(discard-p nil)
			(discard-after-process nil)
			(hang nil)
			&aux ret-val)
  ;;	Test display
  (unless (display-p display)
	  (error "event-sync: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-sync: cannot use closed display \`~s\`." display))
  ;;	Test handler
  (unless (functionp handler)
	  (error "event-sync: illegal handler ~s." handler))
  ;;	Test windows
  (unless (or (and (atom windows) (or (eq windows t) (window-p windows)))
	      (consp windows))
	  (error "event-sync: illegal windows ~s." windows))
  ;;	Test mask
  (unless (or (eq mask t) (consp mask))
	  (error "event-sync: illegal mask ~s." mask))
  (unless (eq mask t)
	  (setq mask (mapcar #'convert-event-type mask)))
  ;;	Test count
  (unless (or (eq count t) (eq count :all) (integerp count))
	  (error "event-sync: illegal count ~s." count))
  
  (if hang
      (xlib:process-event (res display)
			  :handler 
			  (if handlerp 
			      handler
			      #'(lambda (&rest args)
					(setq ret-val
					      (apply handler
						     (append 
						      (list windows 
							    mask
							    discard-p)
						      args)))
					(if (eq ret-val :abort) t nil)))
			  :discard-p discard-after-process
			  :peek-p nil 
			  :force-output-p t :timeout nil)
      (if (integerp count)
	  (do ((i 0 (1+ i))
	       (ev t (xlib:process-event 
		      (res display) 
		      :handler 
		      (if handlerp 
			  handler
			  #'(lambda (&rest args)
				    (setq ret-val
					  (apply handler
						 (append 
						  (list windows mask discard-p)
						  args)))
				    (if (eq ret-val :abort) t nil)))
		      :discard-p discard-after-process
		      :peek-p nil 
		      :force-output-p t
		      :timeout 0)))
	      ((or (null ev) (eq ev :abort) (> i count) (eq ret-val :abort))))
	  (do ((ev t (xlib:process-event 
		      (res display) 
		      :handler 
		      (if handlerp
			  handler
			  #'(lambda (&rest args)
				    (setq ret-val
					  (apply handler
						 (append 
						  (list windows mask discard-p)
						  args)))))
		      :discard-p discard-after-process 
		      :peek-p nil 
		      :force-output-p t :timeout 0)))
	      ((or (null ev) (eq ev :abort) (eq ret-val :abort))
	       (if (null ev) nil :abort))))))

;;;
;;;	Discards all events in queue
;;;

(defun discard-handler (&rest args)
  (declare (ignore args))
  t)

(defun event-discard (&optional (display (current-display)) &aux n)
  (unless (display-p display)
	  (error "event-discard: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-discard: cannot use closed display \`~s\`." display))
  (when (numberp (setq n (event-count display)))
	(setq display (res display))
	(dotimes (i n)
		 (xlib:discard-current-event display)))
  (xlib:discard-current-event display))

;;;
;;;	Returns number of events on queue
;;;
(defun event-count (&optional (display (current-display)))
  (unless (display-p display)
	  (error "event-count: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-count: cannot use closed display \`~s\`." display))
  (setq display (res display))
  (xlib:display-finish-output display)
  (let ((count (xlib:event-listen display)))
       (if count
	   count
	   0)))

(defun event-print (&key (display (current-display)) 
			(hang nil) &aux window)
  (unless (display-p display)
	  (error "event-print: illegal display \`~s\`." display))
  (unless (attached-p display)
	  (error "event-print: cannot use closed display \`~s\`." display))
  (xlib:process-event 
   (res display) 
   :discard-p nil 
   :peek-p t 
   :force-output-p nil 
   :timeout (if hang nil 0) 
   :handler 
   #'(lambda (&key event-window event-key display
		   &allow-other-keys) 
	     (setq window *window-cache*) 
	     (unless (eq event-window *window-cache-res*) 
		     (setq window (quick-find-window event-window display)) 
		     (if window 
			 (setq *window-cache* window
			       *window-cache-res* (res window))))
	     
	     (format t "event: ~S on ~S~%" event-key window)
	     nil)))
