(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;	Event-handler table contains list of handler funcs for each widget-class
(defvar *event-handler-table* nil)
(eval-when (compile load eval)
  (when (or (not (boundp '*event-handler-table*)) (null *event-handler-table*))
	(setq *event-handler-table* 
	      (make-hash-table :size 40 :rehash-size 10))))

;;	Current resource-database
(defvar *resource-database* nil)

;;	system-level defaults path
(defvar *picasso-defaults-path* "~picasso/lib/picasso-defaults")

;;	user-level defaults path
(defvar *user-defaults-path* "~/.picasso-defaults")

;;
;;	Registers event-handler with optional default mapping.
;;

(defmacro defhandler (name arglist &rest body 
			   &aux class-name func-name mapping doc) 
  (if (atom (car arglist))
      (error "bad first-argument-form:  ~S for event-handler:  ~S" 
	     (car `,arglist) `,name))
  (setq class-name `,(cadar arglist))
  (setq func-name
	`,(read-from-string 
	   (concatenate 'string (string class-name) "-" (string name))))
  (setq arglist `,(cons (caar arglist) (cdr arglist)))
  (setq mapping (cadr (member '&default arglist)))
  (when mapping
	(setq mapping
	      (if (and (consp mapping) (listp (car mapping)))
		  (mapcar #'parse-event-spec mapping)
		  (list (parse-event-spec mapping))))
	(if (member nil mapping)
	    (warn "illegal default-event-mapping:  ~S for:  ~S" 
		  (cdr (member '&default arglist))
		  class-name)))
  (if (stringp (car body))
      (setq doc (car body)
	    body (cdr body))
      (setq doc ""))

  ;;  Define function handler 
  (if (evenp (length arglist)) 
      (remf arglist '&default)
      (remf (cdr arglist) '&default))

  ;;  Register handler
  `(progn
    (let ((hdata (gethash ',class-name *event-handler-table*))
	  (temp nil))
	  (if (setq temp (member ',name hdata :test #'ev-hand-test))
	      (setf (cadar temp) ',mapping) 
	      (setf (gethash ',class-name *event-handler-table*) 
		    (push ',(list name mapping) hdata))))
    (defun ,func-name ,arglist ,doc ,@body)))

;;;
;;;	Maps events into default handlers
;;;

(defmacro defevents (class-name &rest specs)
  `(let ((mapping nil)
	 (hdata nil)
	 (temp nil)
	 (func nil))
	(dolist (sp ',specs)
		(setq mapping (parse-event-spec (car sp)))
		(if (null mapping)
		    (warn "illegal default-event-mapping:  ~S for:  ~S" 
			  sp ',class-name)
		    (progn
		     (setq func (cadr sp))
		     (setq hdata (gethash ',class-name *event-handler-table*)) 
		     (if (setq temp (member mapping hdata :test #'ev-map-test))
			 (setf (caar temp) func) 
			 (setf (gethash ',class-name *event-handler-table*) 
			       (push (list func (list mapping)) hdata))))))))
  
(defun ev-hand-test (a b)
  (eql a (car b)))

(defun ev-map-test (a b)
  (equal a (cadr b)))

;;	Parses event-mapping spec for defhandler
(defun parse-event-spec (spec)
  (when spec 
	(if (atom spec)
	    (setq spec (list spec)))
	(if (oddp (length spec)) 
	    (if (or (member :state spec) (member :detail spec))
		(setq spec 
		      (list (car spec) 
			    (getf (cdr spec) :state)
			    (getf (cdr spec) :detail)))
		(let ((temp (make-list 3)))
		     (setf (first temp) (first spec)
			   (second temp) (convert-state (second spec))
			   (third temp) (convert-detail (first spec)
							(third spec)))
		     (setq spec temp)))
	    (if (or (member :state spec) (member :detail spec))
		(setq spec nil)
		(let ((temp (make-list 3)))
		     (setf (first temp) (first spec)
			   (second temp) (convert-state (second spec))
			   (third temp) (convert-detail (first spec) 
							(third spec)))
		     (setq spec temp))))
	spec))

;;
;;	Creates class-event-map and registers all relevent events for class-of
;;	self and all inherited classes.  This process works from the top of
;;	the class inheritence tree to the bottom.  Hence, all local mappings
;;	will override inherited ones.
;;	NOTE:  if one event is mapped to two different handlers, it is 
;;	undetermined which one will get called.
;;
(defun make-class-event-map (self &aux cpl map class-name func)

  ;;  Compute class-precedence-list from self
  (setq cpl (cddr (reverse (mapcar #'class-name 
				   (pcl::class-precedence-list 
				    (class-of self))))))

  ;;  Make hash-table if necessary
  (setq map (class-event-map self))
  (if (or (null map) (not (hash-table-p map)))
      ;;  create new map
      (setf (slot-value self 'class-event-map)
	    (setq map (make-hash-table :test #'equal))))

  ;;  Fill map in with entries starting at the top of the tree
  (dolist (class cpl)
	  (setq class-name (string class))
	  (dolist (mappings (gethash class *event-handler-table*))
		  (setq func (car mappings))
		  (setq func
			(if (atom func)
			    (symbol-function (read-from-string 
					      (concatenate 
					       'string class-name 
					       "-"
					       (string func))))
			    (symbol-function (read-from-string 
					      (concatenate 
					       'string (string (car func))
					       "-"
					       (string (cadr func)))))))
;;		  (format t "func: ~s~%" func)
		  (dolist (mapping (cadr mappings))
;;			  (format t "mapping: ~s~%" func)
			  (setq mapping (list (convert-event-type 
					       (first mapping))
					      (if (second mapping)
						  (apply 
						   #'xlib:make-state-mask 
						   (convert-state 
						    (second mapping)))
						  0)
					      (convert-detail (first mapping) 
							      (third mapping))))
			  (if (function-p func)
			      (setf (find-entry map mapping) func)))))

  ;;  Return class-event-map
  map)

;;;
;;;	Clear event-mappings for specified widget(s).
;;;	If widgets is null, clear all.
;;;
(defun clean-event-mapping (&optional widgets)
  (cond ((null widgets)
	 (clrhash *event-handler-table*))
	((atom widgets)
	 (remhash widgets *event-handler-table*))
	(t
	 (dolist (w widgets)
	 (remhash w *event-handler-table*)))))

;;;
;;;	Load event defaults in from file(s)
;;;

(defun load-event-maps (&key (erase-old t) &aux pic-loaded user-loaded)
  (if erase-old
      (setq *resource-database* nil))
  (when (probe-file *picasso-defaults-path*)
	(if (null *resource-database*)
	    (setq *resource-database* (xlib:make-resource-database)))
	(unwind-protect
	 (progn
	  (xlib:read-resources *resource-database* 
			       *picasso-defaults-path*)
	  (setq pic-loaded t))
	 (if (not pic-loaded)
	     (setq *resource-database* nil))))
  (when (probe-file *user-defaults-path*)
	(if (null *resource-database*)
	    (setq *resource-database* (xlib:make-resource-database)))
	(unwind-protect
	 (progn
	  (xlib:read-resources *resource-database* 
			       *user-defaults-path*)
	  (setq user-loaded t))
	 (if (not user-loaded)
	     (if pic-loaded 
		 (xlib:read-resources *resource-database* 
				      *picasso-defaults-path*)
		 (setq *resource-database* nil)))))
  (when (or pic-loaded user-loaded) 
	(xlib:map-resource *resource-database* #'register-loaded-event)
	t))

(defun register-loaded-event (event func &aux class-name hdata mapping temp)
  (when (not (eq (length event) 4))
	(return-from register-loaded-event))
  (setq func (read-from-string func nil))
  (setq class-name (read-from-string (car event)))
  (setq hdata (gethash class-name *event-handler-table*))
  (setq mapping (list (convert-event-type 
		       (excl::make-keyword 
			(parse-from-database (second event))))
		      (parse-from-database (third event)) 
		      (parse-from-database2 (fourth event))))
  (setq mapping (parse-event-spec mapping))
  (when mapping
	(if (setq temp (member func hdata :test #'ev-hand-test)) 
	    (push mapping (cadar temp))
	    (setf (gethash class-name *event-handler-table*) 
		  (push (list func (list mapping)) hdata)))))

(defun parse-from-database (str)
  (if (string= str "*")
      nil
      (read-from-string str nil)))

(defun parse-from-database2 (str &aux s)
  (if (string= str "*")
      nil
      (if (setq s (character str))
	  s
	  (read-from-string str))))
