(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;	Event-handler table contains list of handler funcs for each widget-class
(defvar *event-handler-table* nil)
(eval-when (compile load eval)
  (when (or (not (boundp '*event-handler-table*)) (null *event-handler-table*))
	(setq *event-handler-table* 
	      (make-hash-table :size 40 :rehash-size 10))))

;;	Current resource-database
(defvar *resource-database* nil)

;;	system-level defaults path
(defvar *picasso-defaults-path* "~picasso/lib/picasso-defaults")

;;	user-level defaults path
(defvar *user-defaults-path* "~/.picasso-defaults")

;;
;;	DEFHANDLER NOW IS LOCATED IN macros/window-macros.cl
;;

;;
;;	DEFEVENTS NOW IS LOCATED IN macros/window-macros.cl
;;

(defun ev-hand-test (a b)
  (eql a (car b)))

(defun ev-map-test (a b)
  (equal a (cadr b)))

;;
;;      Registers event-handler with optional default mapping.
;;

(defmacro defhandler (name arglist &rest body
                           &aux class-name func-name mapping doc)
  (if (atom (car arglist))
      (error "bad first-argument-form:  ~S for event-handler:  ~S"
             (car `,arglist) `,name))
  (setq class-name `,(cadar arglist))
  (setq func-name
        `,(read-from-string
           (concatenate 'string (string class-name) "-" (string name))))
  (setq arglist `,(cons (caar arglist) (cdr arglist)))
  (setq mapping (cadr (member '&default arglist)))
  (when mapping
        (setq mapping
              (if (and (consp mapping) (listp (car mapping)))
                  (mapcar #'parse-event-spec mapping)
                  (list (parse-event-spec mapping))))
        (if (member nil mapping)
            (warn "illegal default-event-mapping:  ~S for:  ~S"
                  (cdr (member '&default arglist))
                  class-name)))
  (if (stringp (car body))
      (setq doc (car body)
            body (cdr body))
      (setq doc ""))

  ;;  Define function handler
  (if (evenp (length arglist))
      (remf arglist '&default)
      (remf (cdr arglist) '&default))

  ;;  Register handler
  `(progn
    (let ((hdata (gethash ',class-name *event-handler-table*))
          (temp nil))
          (if (setq temp (member ',name hdata :test #'ev-hand-test))
              (setf (cadar temp) ',mapping)
              (setf (gethash ',class-name *event-handler-table*)
                    (nconc hdata ',(list (list name mapping))))))
    (defun ,func-name ,arglist ,doc (block ,name (let nil ,@body)))))

;;;
;;;     Maps events into default handlers
;;;

(defmacro defevents (class-name &rest specs)
  `(let ((mapping nil)
         (hdata nil)
         (temp nil)
         (func nil))
        (dolist (sp ',specs)
                (setq mapping (parse-event-spec (car sp)))
                (if (null mapping)
                    (warn "illegal default-event-mapping:  ~S for:  ~S"
                          sp ',class-name)
                    (progn
                     (setq func (cadr sp))
                     (setq hdata (gethash ',class-name *event-handler-table*))
                     (if (setq temp (member mapping hdata :test #'ev-map-test))
                         (setf (caar temp) func)
                         (setf (gethash ',class-name *event-handler-table*)
			       (nconc hdata 
				      (list (list func (list mapping)))))))))))

;;      Parses event-mapping spec for defhandler
(defun parse-event-spec (spec)
  (when spec
        (if (atom spec)
            (setq spec (list spec)))
        (if (oddp (length spec))
            (if (or (member :state spec) (member :detail spec))
                (setq spec
                      (list (car spec)
                            (getf (cdr spec) :state)
                            (getf (cdr spec) :detail)))
                (let ((temp (make-list 3)))
                     (setf (first temp) (first spec)
                           (second temp) (convert-state (second spec))
                           (third temp) (convert-detail (first spec)
                                                        (third spec)))
                     (setq spec temp)))
            (if (or (member :state spec) (member :detail spec))
                (setq spec nil)
                (let ((temp (make-list 3)))
                     (setf (first temp) (first spec)
                           (second temp) (convert-state (second spec))
                           (third temp) (convert-detail (first spec)
                                                        (third spec)))
                     (setq spec temp))))
        spec))

(defun convert-state-key (key)
  (setq key (excl::make-keyword key))
  (case key
        (:left-button :button-1)
        (:middle-button :button-2)
        (:right-button :button-3)
        (:meta :mod-1)
        (t key)))

(defun convert-state (state)
  (unless (listp state) (setq state (list state)))
  (mapcar #'convert-state-key state))

(defun convert-detail (event-type detail)
  (case event-type
	((:pointer-motion :button-motion :button-1-motion :button-2-motion
			  :button-3-motion :button-4-motion
			  :button-5-motion)
	 (case detail 
	       ((:left-button :button-1) (xlib:make-state-mask :button-1))
	       ((:middle-button :button-2) (xlib:make-state-mask :button-2))
	       ((:right-button :button-3) (xlib:make-state-mask :button-3))
	       ((:button-3 :button-4 :button-5) 
		(xlib:make-state-mask detail))
	       (t nil)))
	(t
	 (typecase detail
		   (keyword 
		    (case detail 
			  ((:left-button :button-1) 1) 
			  ((:middle-button :button-2) 2) 
			  ((:right-button :button-3) 3)
			  (t detail)))
		   (character detail) 
		   (string
		    (let ((char (character detail)))
			 (if char char detail)))
		   (null 0)
		   (t detail)))))


