;;;
;;; Copyright (c) 1982 Regents of the University of California
;;; Copyright (c) 1987 Regents of the University of California
;;;
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 

(in-package 'sling)

;;;
;;; These symbols are all functions.
;;;
(export '(          
	  db-createdb db-destroydb db-sysmod
	  db-connect db-disconnect
	  db-close db-commit db-copy
	  db-create-index db-create-integrity db-create-procedure
	  db-create-table db-create-view db-declare-cursor
	  db-delete db-describe
	  db-drop db-drop-integrity db-drop-permit db-drop-procedure
	  db-execute db-execute-procedure db-fetch db-grant 
	  db-inquire-ingres db-insert db-modify db-open db-prepare
	  db-relocate db-rollback db-save db-savepoint db-select db-update
	  db-whenever db-mapc-select db-mapcar-select db-mapcan-select
	  db-set-autocommit
	  db-set-journaling db-set-nojournaling
	  db-set-result-structure
	  db-set-lockmode
	  end-command
	  db-endquery db-errorno db-errortext db-messagetext
	  db-messagenumber db-rowcount db-transaction
	  ))
;;;
;;; These symbols are exported so that internal routines may compare
;;; for 'eq'uality.  Thus, any target lists or qualifications sent to
;;; cling should have been read (or looked up) while using the cling package.
;;;
(export '(
	  + - * / **
	  = != > >= < <=
	  and or
	  ))

;;;
;;; User visible routines:
;;;

;;;
;;;  Unix type commands:
;;;
(defun db-createdb (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "createdb" dbname flags))))
	  
(defun db-destroydb (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "destroydb" dbname flags))))

(defun db-sysmod (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "sysmod" dbname flags))))

;;;
;;; Pass the strings to myIIsqConnect.
;;; Also record the current database name for those who may wish to know
;;; (or remember).
;;; 
;;;
(defun db-connect (dbname &key (flags (make-string 0)))
  (record-time
   (when current-db-name (error "A database is already open"))
   (sqlca-init)
   (ii-sqConnect (string-downcase (string dbname)) (string flags))
   (setq current-db-name (string dbname))
   t))

;;;
;;; Major action here is to call IIexit.
;;;
(defun db-disconnect ()
  (record-time
   (setq current-db-name nil)
   (sqlca-init)
   (ii-sqDisconnect)
  t))

;;;;
;;;; The following functions perform basic sql operations.
;;;; Alphabetical order.
;;;;

;;;
;;;; CLOSE
;;;
(defun db-close (cursor)
  (record-time
   (state-check)
   (sqlca-init)
   (apply 'ii-csClose (cursor-id cursor))
   (end-command)))

;;;
;;;; COMMIT
;;;
(defun db-commit ()
  (record-time
   (state-check)
   (sqlca-init)
   (ii-xact 3)
   (end-command)))

;;;
;;;; COPY
;;;
;;; Format is a list structure, not a string.
;;;
(defun db-copy (relation format &key into from
			 (terminate-on-error nil toep) error-count
			 (rollback nil rbp)
			 log &aux others)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "copy table ")
   (write-out-name relation)
   (write-out-=-list format)
   (write-out (if into " into " " from "))
   (write-out-file-name (string (if into into from)))
   (when (OR toep error-count rbp log)
	 (write-out " with "))
   (when toep
	 (setf others t)
	 (write-out "on_error = ")
	 (write-out (if terminate-on-error "terminate" "continue")))
   (when error-count
	 (when others (write-out ", "))
	 (setf others error-count)
	 (write-out "error_count = ")
	 (write-out-item error-count))
   (when rbp
	 (when others (write-out ", "))
	 (setf others t)
	 (write-out "rollback = ")
	 (write-out (if rollback "enabled" "disabled")))
   (when log
	 (when others (write-out ", "))
	 (write-out "log = ")
	 (write-out-file-name log))
   (end-command)))

;;;
;;;; CREATE INDEX
;;;
(defun db-create-index (indexname tablename columns &key
				  structure key fillfactor minpages
				  maxpages leaffill nonleaffill maxindexfill
				  location &aux others) 
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "create index ")
   (write-out-name indexname)
   (write-out " on ")
   (write-out-name tablename)
   (write-out-list columns 'write-out-corr-element)
   (when (OR structure key fillfactor minpages maxpages leaffill nonleaffill
	     maxindexfill location) 
	 (write-out " with "))
   (when structure
	 (setf others structure)
	 (write-out " fillfactor=")
	 (write-out-item fillfactor))
   (when key
	 (when others (write-out ","))
	 (setf others key)
	 (write-out " key=")
	 (write-out-target-list key))
   (when fillfactor
	 (when others (write-out ","))
	 (setf others fillfactor)
	 (write-out " fillfactor=")
	 (write-out-item fillfactor))
   (when minpages
	 (when others (write-out ","))
	 (setf others minpages)
	 (write-out " minpages=")
	 (write-out-item minpages))
   (when maxpages
	 (when others (write-out ","))
	 (setf others maxpages)
	 (write-out " maxpages=")
	 (write-out-item maxpages))
   (when leaffill
	 (when others (write-out ","))
	 (setf others leaffill)
	 (write-out " leaffill=")
	 (write-out-item leaffill))
   (when nonleaffill
	 (when others (write-out ","))
	 (setf others nonleaffill)
	 (write-out " nonleaffill=")
	 (write-out-item nonleaffill))
   (when maxindexfill
	 (when others (write-out ","))
	 (setf others nonleaffill)
	 (write-out " maxindexfill=")
	 (write-out-item maxindexfill))
   (when location
	 (when others (write-out ","))
	 (write-out " location=")
	 (write-out-location location))
   (end-command)))

;;;
;;;; CREATE INTEGRITY
;;;
(defun db-create-integrity (name condition)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "create integrity on ")
   (write-out-corr-element name)
   (write-out " is")
   (write-out-qual condition " ")
   (end-command)))

;;;
;;;; CREATE PROCEDURE
;;;
(defun db-create-procedure (name params &rest stmts)
  (record-time
   (state-check)
   (sqlca-init)
   (ii-LQpriProcInit 0 (string name))
   (write-out "create procedure ")
   (write-out-name name)
   (ii-putctrl 0) (when show-query (terpri))
   (when params (write-out-list params 'write-out-column-format))
   (write-out " as")
   (ii-putctrl 0) (when show-query (terpri))
   (write-out "begin")
   (ii-putctrl 0) (when show-query (terpri))
   (mapc #'(lambda (x)
		   (nested-command (apply (car x) (rest x)))
		   (write-out ";")
		   (ii-putctrl 0)
		   (when show-query (terpri))) stmts)
   (write-out "end")
   (ii-putctrl 0) (when show-query (terpri))
   (end-command)))

;;;
;;;; CREATE TABLE
;;;
(defun db-create-table (relation format &key as
				 location (journaling nil journp)
				 (duplicates nil dupp) &aux others)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "create table ")
   (write-out-location relation)
   (write-out-list format 'write-out-column-format)
   (when as
	 (write-out " as ")
	 (nested-command (apply (car as) (rest as))))
   (when (OR location journp dupp)
	 (write-out " with "))
   (when location
	 (setf others location)
	 (write-out "location = ")
	 (write-out-location location))
   (when journp
	 (when others (write-out ", "))
	 (setf others t)
	 (write-out (if journaling "journaling" "nojournaling")))
   (when dupp
	 (when others (write-out ", "))
	 (setf others t)
	 (write-out (if duplicates "duplicates" "noduplicates")))
   (end-command)))

;;;
;;;; CREATE VIEW
;;;
(defun db-create-view (name columns subselect &key check-option)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "create view ")
   (write-out-name name)
   (unless (null columns) (write-out-target-list columns))
   (write-out " as ")
   (nested-command (apply (car subselect) (rest subselect)))
   (when check-option (write-out " with check option"))
   (end-command)))

;;;
;;;; DECLARE CURSOR
;;;
(defun db-declare-cursor (name query &key
			       deferred-update direct-update
			       &aux for-update)
  (unless (atom query)
	  (rplaca query (apply 'compute-targetlist query)))
  ;; check here for bad declarations e.g. update on a join
  (when deferred-update
	(setq for-update (list 'deferred deferred-update)))
  (when direct-update
	(setq for-update (list 'direct direct-update)))
  (record-cursor-name name query (update-type for-update))
  nil)

;;;
;;;; DELETE
;;;
(defun db-delete (relation &key where)
  (record-time
   (state-check)
   (sqlca-init)
   (if (equal (car where) 'current-of)
       ;; might check here to see if relation is the same as cursor
       (apply 'ii-csDelete relation (cursor-id (second where)))
       (progn
	(write-out "delete from ")
	(write-out-corr-element relation)
	(write-out-qual where)
	(end-command)))))

;;;
;;;; DESCRIBE
;;;
(defun db-describe (stmtname &key using-names (desc sqlda))
  (record-time
   (state-check)
   (setf (sqlda-sqln desc) sqlda-num-vars)
   (sqlca-init)
   (ii-sqDescribe 0 (string stmtname) desc (if using-names 1 0))
   (end-command)))

;;;
;;;; DROP {TABLE | INDEX | VIEW}
;;;
(defun db-drop (&rest names)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "drop")
   (write-out-sequence names 'write-out-name)
   (end-command)))

;;;
;;;; DROP INTEGRITY
;;;
(defun db-drop-integrity (relation integrities)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "drop integrity on ")
   (write-out-name relation)
   (write-out " ")
   (if (eq integrities 'all)
       (write-out "all")
       (write-out-sequence (listify integrities) 'write-out-number))
   (end-command)))

;;;
;;;; DROP PERMIT
;;;
(defun db-drop-permit (what permits &key procedure)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "drop permit on ")
   (when procedure (write-out "procedure "))
   (write-out-name what)
   (write-out " ")
   (if (eq permits 'all)
       (write-out "all")
       (write-out-sequence (listify permits) 'write-out-number))
   (end-command)))

;;;
;;;; DROP PROCEDURE
;;;
(defun db-drop-procedure (name)
  (record-time
   (state-check)
   (sqlca-init)
   (ii-LQpriProcInit 1 (string name))
   (write-out "drop procedure ")
   (write-out-name name)
   (end-command)))

;;;
;;;; EXECUTE
;;;
(defun db-execute (name &key values descriptor)
  (record-time
   (state-check)
   (sqlca-init)
   (ii-sqExStmt (string name) (if descriptor 1 0))
   (if descriptor
       (ii-sqDaIn 0 descriptor)
       (send-values-to-ingres values))
   (end-command)))

;;;
;;;; EXECUTE PROCEDURE
;;;
(defun db-execute-procedure (name &rest args)
  (record-time
   (state-check)
   (sqlca-init)
   (ii-LQpriProcInit 2 (string name))
   (send-values-to-ingres args :procedure t)
   (do ()
       ((eql (ii-LQprsProcStatus) 0))
       (dispatch-whenever (sqlstatus)))
   (ii-eqiqi "iiret")))

;;;
;;;; FETCH
;;;
(defun db-fetch (cursor &key using &aux result)
  (record-time
   (state-check)
   (sqlca-init)
   (when (/= (apply 'ii-csRetrieve (cursor-id cursor)) 0)
	 (if using
	     (progn
	      (ii-csDaGet 0 using)
	      (setf result nil))
	     (setf result (fetch-tuple (determine-target-types
					(caar (lookup-cursor-name cursor))))))
	 (ii-csEretrieve)
	 (if (sqlstatus)
	     (dispatch-whenever (sqlstatus))
	     result))))

;;;
;;;; GRANT
;;;
(defun db-grant (which what whom &key procedure)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "grant ")
   (write-out-sequence
    (listify which)
    #'(lambda (x) (if (atom x)
		      (write-out-name x)
		      (progn
		       (write-out-name (car x))
		       (write-out-list (rest x) 'write-out-column)))))
   (write-out " on ")
   (when procedure (write-out "procedure "))
   (write-out-sequence (listify what) 'write-out-name)
   (write-out " to ")
   (write-out-sequence (listify whom) 'write-out-name)
   (end-command)))

;;;
;;;; INQUIRE INGRES
;;;
(defun db-inquire-ingres (&key errortext errorno rowcount messagetext
			       messagenumber endquery transaction
			       &aux (result nil))
  (record-time
   (when transaction
	 (push (cons "transaction" (do-inquire 'transaction)) result))
   (when endquery
	 (push (cons "endquery" (do-inquire 'endquery)) result))
   (when messagenumber
	 (push (cons "messagenumber" (do-inquire 'messagenumber)) result))
   (when messagetext
	 (push (cons "messagetext" (do-inquire 'messagetext)) result))
   (when rowcount
	 (push (cons "rowcount" (do-inquire 'rowcount)) result))
   (when errorno
	 (push (cons "errorno" (do-inquire 'errorno)) result))
   (when errortext
	 (push (cons "errortext" (do-inquire 'errortext)) result))
   result))

;; individual objects from inquire-ingres
(defun db-endquery ()
  (record-time
   (do-inquire 'endquery))
(defun db-errorno ()
  (record-time
   (do-inquire 'errorno))
(defun db-errortext ()
  (record-time
   (do-inquire 'errortext))
(defun db-messagenumber ()
  (record-time
   (do-inquire 'messagenumber))
(defun db-messagetext ()
  (record-time
   (do-inquire 'messagetext))
(defun db-rowcount ()
  (record-time
   (do-inquire 'rowcount))
(defun db-transaction ()
  (record-time
   (do-inquire 'transaction)))

;;;
;;;; INSERT
;;;
;;; Takes a relname, not a variable name.
;;;
(defun db-insert (relname targetlist vallist)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "insert into ")
   (write-out-name relname)
   (unless (null targetlist) (write-out-target-list targetlist))
   (if (eq (car (listify vallist)) 'db-select)
       (progn
	(write-out " ")
	(nested-command (apply 'db-select (rest vallist)))
	)
       (progn
	(write-out " values ")
	(write-out-list vallist 'write-out-expression)))
   (end-command)))

;;;
;;;; MODIFY
;;;
(defun db-modify (relation structure &key on unique fillfactor minpages
		   maxpages leaffill nonleaffill maxindexfill &aux others)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "modify ")
   (write-out-name relation)
   (write-out " to ")
   (write-out-name structure)
   (when unique (write-out " unique "))
   (when on
	 (write-out " on ")
	 (write-out-sort-list on))
   (when (OR fillfactor minpages maxpages leaffill nonleaffill maxindexfill) 
	 (write-out " with"))
   (when fillfactor
	 (setf others fillfactor)
	 (write-out " fillfactor=")
	 (write-out-item fillfactor))
   (when minpages
	 (when others (write-out ","))
	 (setf others minpages)
	 (write-out " minpages=")
	 (write-out-item minpages))
   (when maxpages
	 (when others (write-out ","))
	 (setf others maxpages)
	 (write-out " maxpages=")
	 (write-out-item maxpages))
   (when leaffill
	 (when others (write-out ","))
	 (setf others leaffill)
	 (write-out " leaffill=")
	 (write-out-item leaffill))
   (when nonleaffill
	 (when others (write-out ","))
	 (setf others nonleaffill)
	 (write-out " nonleaffill=")
	 (write-out-item nonleaffill))
   (when maxindexfill
	 (when others (write-out ","))
	 (write-out " maxindexfill=")
	 (write-out-item maxindexfill))
   (end-command)))


;;;
;;;; OPEN
;;;
;;;
(defun db-open (cursor-name &key readonly values descriptor
			&aux (select-update (lookup-cursor-name cursor-name))
			(cursor-id (cursor-id cursor-name))
			(fix (make-array 1 :element-type 'fixnum))
			(flo (make-array 1 :element-type 'double-float)))
  (when (null select-update)
	(error "No cursor named ~s has been declared.~%" cursor-name))
  (when (equal (cadr select-update) 'deferred)
	(if *deferred-update*
	    (error "Cannot open cursor ~s: ~
		   cursor ~s is already open for deferred update"
		   cursor-name *deferred-update*)
	    (setq *deferred-update* cursor-name)))
  (record-time
   (state-check)
   (sqlca-init)
   (apply 'ii-csOpen cursor-id)
   (if (atom (car select-update))
       ;; statement name
       (progn
	(write-out-name (car select-update))
	(when (or descriptor values)
	      (ii-sqExStmt 0 1))
	(when values
	      (send-values-to-ingres values))
	(when descriptor
	      (ii-sqDaIn 0 descriptor)))
       ;; select
       (progn
	(nested-command (apply 'db-select (car select-update)))
	(when readonly (write-out " for readonly"))))
   (apply 'ii-csQuery cursor-id)
   (dispatch-whenever (sqlstatus))))


;;;
;;;; PREPARE
;;;
(defun db-prepare (stmtname str &key (into 0))
  (record-time
   (state-check)
   (sqlca-init)
   (ii-sqPrepare 0 (string stmtname) into 0 str)
   (end-command)))

;;;
;;;; RELOCATE
;;;
(defun db-relocate (table location)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "relocate ")
   (write-out-name table)
   (write-out " to ")
   (write-out-location location)
   (end-command)))

;;;
;;;; ROLLBACK
;;;
(defun db-rollback (&key to)
  (record-time
   (state-check)
   (sqlca-init)
   (if to
       (progn
	(write-out "rollback to ")
	(write-out-name to))
       (ii-xact 2))
   (end-command)))

;;;
;;;; SAVE
;;;
(defun db-save (relation month day year)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "save ")
   (write-out-name relation)
   (write-out " until ")
   (write-out-item month)
   (write-out " ")
   (write-out-item day)
   (write-out " ")
   (write-out-item year)
   (end-command)))

;;;
;;;; SAVEPOINT
;;;
(defun db-savepoint (name)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "savepoint ")
   (write-out-item name)
   (end-command)))

;;;
;;;; SELECT
;;;
;;;
(defun db-select (targetlist &key from
			     distinct where group-by having union order-by
			     for-update direct &aux result)
  (record-time
   (setq targetlist (listify targetlist))
   (setq from (listify from))
   (unless (nested-command-mode)
	   (setq targetlist (compute-targetlist targetlist from)))
   (state-check)
   (sqlca-init)
   (send-query targetlist from distinct where group-by having union order-by
	       for-update direct)
   (unless (nested-command-mode)
	   (setq result (retrieve-result-from-ingres targetlist))
	   (if (sqlstatus)
	       (dispatch-whenever (sqlstatus))
	       result))))

;;;
;;;; UPDATE
;;;
(defun db-update (tablename columns &key where)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "update ")
   (write-out-name tablename)
   (write-out " set ")
   (write-out-target-sequence columns)
   (if where
       (if (equal (car where) 'current-of)
	   (apply 'ii-csERplace (cursor-id (second where)))
	   (write-out-qual where)))
   (end-command)))

;;;
;;;; WHENEVER
;;;
(defun db-whenever (&key (sqlwarning nil warnp) (sqlerror nil errp)
			 (not-found nil nfp) (sqlmessage nil messp))
  (when warnp
	(setq *warn-handler* sqlwarning))
  (when errp	
	(setq *error-handler* sqlerror))
  (when nfp
	(setq *not-found-handler* not-found))
  (when messp
	(setq *mesg-handler* sqlmessage))
  nil)

;;;
;;;; DB-MAPC-SELECT
;;;
;;; The select is executed, and each tuple returned is made into a list.
;;; The function is applied to each such list.  If the function returns
;;; nil, then the retrieve loop is terminated.
;;; No results are kept.
;;;
(defun db-mapc-select (function targetlist &key from
				distinct where group-by having union order-by
				for-update direct &aux result)
  (record-time
   (setq targetlist (listify targetlist))
   (setq from (listify from))
   (unless (nested-command-mode)
	   (setq targetlist (compute-targetlist targetlist from)))
   (state-check)
   (sqlca-init)
   (send-query targetlist from distinct where group-by having union order-by
	       for-update direct)
   (mapc-retrieve function targetlist)
   (dispatch-whenever (sqlstatus))))

;;;
;;;; DB-MAPCAR-SELECT
;;;
;;; Like db-mapc-select, except that the return value of the function
;;; is collected in a list.  There is no way to break the retrieve loop.
;;;
(defun db-mapcar-select (function targetlist &key from
				  distinct where group-by having union order-by
				  for-update direct &aux result)
  (record-time
   (setq targetlist (listify targetlist))
   (setq from (listify from))
   (unless (nested-command-mode)
	   (setq targetlist (compute-targetlist targetlist from)))
   (state-check)
   (sqlca-init)
   (send-query targetlist from distinct where group-by having union order-by
	       for-update direct)
   (prog1
    (mapcar-retrieve function targetlist)
    (dispatch-whenever (sqlstatus)))))

;;;
;;;; DB-MAPCAN-SELECT
;;;
;;; Like db-mapc-select, except that the return values of the function
;;; are nconc'd together.  There is no way to break the retrieve loop.
;;;
(defun db-mapcan-select (function targetlist &key from
				  distinct where group-by having union order-by
				  for-update direct &aux result)
  (record-time
   (setq targetlist (listify targetlist))
   (setq from (listify from))
   (unless (nested-command-mode)
	   (setq targetlist (compute-targetlist targetlist from)))
   (state-check)
   (sqlca-init)
   (send-query targetlist from distinct where group-by having union order-by
	       for-update direct)
   (prog1
    (mapcan-retrieve function targetlist)
    (dispatch-whenever (sqlstatus)))))

;;;
;;;; SET commands
;;;

;;;
;;;; SET AUTOCOMMIT
;;;
(defun db-set-autocommit (state)
  (record-time
   (state-check)
   (write-out "set autocommit ")
   (write-out (if state "on" "off"))
   (end-command)))

;;;
;;;; SET JOURNALING
;;;
(defun db-set-journaling (&key table)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "set journaling")
   (when table
	 (write-out " on ")
	 (write-out-item table))
   (end-command)))

;;;
;;;; SET NOJOURNALING
;;;
(defun db-set-nojournaling (&key table)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "set nojournaling")
   (when table
	 (write-out " on ")
	 (write-out-name table))
   (end-command)))

;;;
;;;; SET RESULT STRUCTURE
;;;
(defun db-set-result-structure (result-structure)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "set result_structure")
   (write-out-string (string result-structure))
   (end-command)))

;;;
;;;; SET LOCKMODE
;;;
(defun db-set-lockmode (&key on level readlock maxlocks timeout
				&aux other-settings)
  (record-time
   (state-check)
   (sqlca-init)
   (write-out "set lockmode ")
   (when on 
	 (write-out "on ") 
	 (write-out-item on))
   (when (NOT on) (write-out "session"))
   (write-out " where")
   (when level
	 (setf other-settings level)
	 (write-out " level=")
	 (write-out-item level))
   (when readlock
	 (if other-settings (write-out ",") (setf other-settings readlock))
	 (write-out " readlock=")
	 (write-out-item readlock))
   (when maxlocks
	 (if other-settings (write-out ",") (setf other-settings maxlock))
	 (write-out " maxlocks=")
	 (write-out-item maxlocks))
   (when timeout
	 (if other-settings (write-out ","))
	 (write-out " timeout=")
	 (write-out-item timeout))
   (end-command)))

