;;;
;;; PDS (Picasso defsystem) Utility
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/system.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 19:02:24 $
;;;

(in-package 'pds :use '(lisp))

;;;
;;; System structure definition
;;;
(defstruct (system (:print-function print-system))
  (name "")
  (home-pathname (pathname (excl:current-directory)) :type pathname)
  (src-pathname (pathname "") :type pathname)
  (bin-pathname (pathname "") :type pathname)
  (init-code nil)
  (init-code-loaded nil)
  (module-list nil :type list))

;;;
;;; System print function
;;;
(defun print-system (sys str level)
  (declare (ignore level))
  (format str "#<System ~A>" (system-name sys)))

;;;
;;; Find system
;;;
(defun find-system-named (name)
  "return the system structure for the specified system"
  (let ((entry (assoc name *all-systems*)))
       (if (null entry)
	   (error "no system description named ~a loaded." name))
       (cdr entry)))

;;;
;;; Parse-system function
;;;
(defun parse-system (name)
  (let ((system (find-system-named name))
	(curr-dir (excl:current-directory)))
       (output "; Parsing system ~s~%" name)
       (excl:chdir (system-home-pathname system))
       ;; Reset globals used
       (setq *class-file-map* (make-hash-table))
       (setq *setf-file-map* (make-hash-table))
       (setq *macro-file-map* (make-hash-table))
       (setq *file-macro-usage-map* (make-hash-table))
       (setq *file-setf-usage-map* (make-hash-table))
       (setq *file-class-usage-map* (make-hash-table))
       (init-word-table)
       (init-file-cache)
       ;; Execute the init-code
       (when (and (not (system-init-code-loaded system))
		  (system-init-code system))
	     (eval (system-init-code system)))
       ;; parse modules
       (dolist (module (system-module-list system))
	       (output "; Parsing file ~s~%" (make-module-pathname module))
	       (parse-module module))
       ;; Resolve dependencies
       (dolist (module (system-module-list system))
	       (resolve-module-dependencies module))
       (excl:chdir curr-dir)
       (output "; Done parsing system ~s~%" name))
  nil)

;;;
;;; Load-system function
;;;
(defun load-system (name)
  "load the system with the specified name"
  (let ((system (find-system-named name))
	(curr-dir (excl:current-directory)))
       (output "; Loading system ~s~%" name)
       (excl:chdir (system-home-pathname system))
       (init-file-cache)
       ;; Execute the init-code
       (when (and (not (system-init-code-loaded system))
		  (system-init-code system))
	     (eval (system-init-code system)))
       (dolist (module (system-module-list system))
	       (load-module module))
       (excl:chdir curr-dir)
       (output "; Done loading system ~s~%" name))
  ;; return nil
  nil)

;;;
;;; Compile-system function
;;;
(defun compile-system (name)
  (let ((system (find-system-named name))
	(curr-dir (excl:current-directory)))
       (output "; Compiling system ~s~%" name)
       (excl:chdir (system-home-pathname system))
       (init-file-cache)
       ;; Execute the init-code
       (when (and (not (system-init-code-loaded system))
		  (system-init-code system))
	     (eval (system-init-code system)))
       ;; Clear modules compile-in-process flag
       (dolist (module (system-module-list system))
	       (setf (module-compile-in-process module) nil))
       (dolist (module (system-module-list system))
	       (compile-module module))
       (excl:chdir curr-dir)
       (output "; Done compiling system ~s~%" name)
       nil))

;;;
;;; show a system
;;;
(defun show-system (name)
  "show a system definition in detail"
  (let ((sys (find-system-named name)))
       (output "system: ~s~%" name)
       (output "home pathname: ~s~%" (system-home-pathname sys))
       (output "source pathname: ~s~%" (system-src-pathname sys))
       (output "binary pathname: ~s~%" (system-bin-pathname sys))
       (output "init-code: ~s~%" (system-init-code sys))
       (output "init-code-loaded: ~s~%" (system-init-code-loaded sys))
       (output "~%Modules:~%")
       ; (dolist (module (system-module-list sys))
	       ; (show-module module)
	       ; (output "~%"))
       ))
