;;;
;;; PDS (Picasso defsystem) Utility
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/parse.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 19:02:16 $
;;;

(in-package 'pds :use '(lisp))

(defvar reserved-word-list
  '(car cdr cons append list nconc assoc subst length
	concatenate setf setq set and or not atom null
	numberp symbolb listp endp consp typep t nil eq
	eql equal member zerop minusp plusp evenp oddp
	cond if when unless case otherwise do dolist dotimes
	loop prog return return-from prog prog1 progn mapc
	mapcan mapcar map catch throw quote eval apply
	funcall macroexpand multiple-value-bind
	values let* let print princ pprint ppi terpri format defun defmacro
	defsetf defstruct lambda function symbol-function export
	import provide require in-package use-package mod rem make-array
	vector aref declare proclaim type-of deftype call-next-method
	class-of defclass defgeneric-options defgeneric-options-setf defmethod
	defmethod-setf get-method get-setf-generic-function remove-method
	slot-value with-slots remhash + - 1+ 1- do* warn round push pop
	floor pushnew))

(defvar *reserved-word-table* (make-hash-table))

(defun init-word-table ()
  (dolist (rw reserved-word-list)
	  (setf (gethash rw *reserved-word-table*) t)))

(defun reserved-word-p (sym)
  (gethash sym *reserved-word-table*))

;;;
;;; Scan a body of code for setf forms. These are s-exp of the form:
;;;	(setf f1 v1 f2 v2 ...)
;;; Return a list of all such forms in the body of code.
;;;
(defun walk-code-for-setfs (body)
  (let ((ret nil))
       (if (eq 'setf (car body))
	   (do ((setf-form (cdr body) (cddr setf-form)))
	       ((null setf-form))
	       (if (consp (car setf-form))
		   (pushnew (car (car setf-form)) ret)))
	   (if (consp body)
	       (do* ((iter body (cdr iter))
		     (param (if (consp iter) (car iter) nil)
			    (if (consp iter) (car iter) nil)))
		    ((or (not (consp iter)) (null iter)))
		    (if (consp param)
			(setq ret 
			      (union ret (walk-code-for-setfs param)))))))
       ret))

;;;
;;; Scan a body of code for identifiers in a function or macro call context.
;;; Ignores reserved words.
;;; Return a list of all such identifiers in the body of code.
;;;
(defun walk-code-for-macros (body)
  (let ((ret nil))
       (if (and (symbolp (car body)) (not (reserved-word-p (car body))))
	   (setq ret (list (car body))))
       (if (consp body)
	   (do* ((iter body (cdr iter))
		 (param (if (consp iter) (car iter) nil)
			(if (consp iter) (car iter) nil)))
		((or (not (consp iter)) (null iter)))
		(if (consp param)
		    (setq ret 
			  (union ret (walk-code-for-macros param))))))
       ret))

;;;
;;; Function to parse a specialized-lambda-list.
;;;
(defun parse-spec-lambda-list (mod sll)
  "Walk a specialized-lambda-list and register any classes used therein"
  (let ((classes-used nil))
       (block walker
	      (dolist (var sll)
		      (if (member var lambda-list-keywords)
			  (return-from walker))
		      (if (and (consp var) (symbolp (cadr var)))
			  (pushnew (cadr var) classes-used))))
       (register-class-usage mod classes-used)))

;;;
;;; Function to parse a body of code.
;;;
(defun parse-body (mod body)
  (let ((macros-used nil)
	(setfs-used nil))
       ;; Skip declarations and documentation in in body.
       (skip-decl-doc body)
       ;; Grovel around the body looking for function or macro calls
       (dolist (form body)
	       (when (consp form)
		     (setq setfs-used 
			   (union setfs-used (walk-code-for-setfs form)))
		     (setq macros-used 
			   (union macros-used (walk-code-for-macros form)))))
       (register-setf-usage mod setfs-used)
       (register-macro-usage mod macros-used)))

;;;
;;; Parse something of the form:
;;;	(defun name lambda-list {declaration|doc-string}* {form}* )
;;;
(defun parse-defun (mod in-list)
  (parse-body mod (cdddr in-list)))

;;;
;;; Parse something of the form:
;;;  (defmethod name {qualifier}* spec-lambda-list {decl|doc}* {form}* )
;;;			-or-
;;;  (defmethod (setf name) {qualifier}* spec-lambda-list {decl|doc}* {form}* )
;;;
;;; Where spec-lambda-list ::= ( {var|(var param-spec)}* [ll-keyword ...] )
;;;
(defun parse-defmethod (mod in-list)
  ;; See if this is a setf method definition...
  (let ((name (cadr in-list)))
       (if (and (consp name) (eq 'setf (first name)))
	   (register-setf-definition mod (second name))))

  ;; Skip ahead in input to the spec-lambda-list
  (setq in-list (cddr in-list))
  (skip-method-qualifiers in-list)
  ;; (car in-list) is now spec-lambda-list
  (let ((sll (car in-list))
	(body (cdr in-list)))
       ;; First process spec-lambda-list
       (parse-spec-lambda-list mod sll)
       ;; Now process body
       (parse-body mod body)))

;;;
;;; Parse something of the form:
;;; (define-setf-method access-fn lambda-list {decl|doc}* {form}* )
;;;
(defun parse-define-setf-method (mod in-list)
  (register-setf-definition mod (second in-list))
  (parse-body mod (cdddr in-list)))

;;;
;;; Parse something of the form:
;;; (defsetf access-fn update-fn)
;;;
(defun parse-defsetf (mod in-list)
  (register-setf-definition mod (second in-list)))

;;;
;;; Parse something of the form:
;;;     (defmethod-setf name spec-lambda-list spec-lambda-list {decl|doc}*
;;;			{form}* )
;;;
(defun parse-defmethod-setf (mod in-list)
  (register-setf-definition mod (second in-list))
  (setq in-list (cddr in-list))
  (let ((sll-1 (first in-list))
	(sll-2 (second in-list))
	(body (cddr in-list)))
       ;; First process spec-lambda-lists
       (parse-spec-lambda-list mod sll-1)
       (parse-spec-lambda-list mod sll-2)
       ;; Now process body
       (parse-body mod body)))

;;;
;;; Parse something of the form:
;;;	(defmacro name lambda-list {declaration|doc-string}* {form}* )
;;;
(defun parse-defmacro (mod in-list)
  (let ((name (second in-list)))
       (register-macro-definition mod name)))

;;;
;;; Parse something of the form:
;;;	(defclass name ({superclass}*) ({slot-spec}*) ...)
;;; 
;;; Where slot-spec ::= slot-name | (slot-name slot-options)
;;;
(defun parse-defclass (mod in-list)
  (let ((name (second in-list))
	(supers (third in-list))
	(slot-specs (fourth in-list))
	(setf-names nil)
	(type-names nil))
       (register-class-definition mod name)
       (register-class-usage mod supers)
       ;; Run down the slot specs, looking for :writer or :accessor
       ;; slot-options -- these are setf names.
       ;; Also, look for initforms that have macros or setfs imbedded.
       (dolist (slot-spec slot-specs)
	       (when (consp slot-spec)
		     (do ((slot-option (cdr slot-spec) (cddr slot-option)))
			 ((null slot-option))
			 (if (member (car slot-option) '(:accessor :writer))
			     (pushnew (cadr slot-option) setf-names))
			 (if (eq (car slot-option) :type)
			     (pushnew (cadr slot-option) type-names))
			 ;; Check for initform with macro imbedded.
			 (when (and (eq (car slot-option) :initform)
				  (consp (cadr slot-option)))
			       (register-setf-usage mod 
			         (walk-code-for-setfs (cadr slot-option)))
			       (register-macro-usage mod 
			         (walk-code-for-macros (cadr slot-option)))))))
       ;; All the setf functions in setf-names are defined in this file...
       (dolist (setf setf-names)
	       (register-setf-definition mod setf))))

;;;
;;; Parse something of the form:
;;;	(defstruct {name | (name options)} [doc-string] 
;;;		   {slot | (slot slot-options)}* )
;;;
;;; The interesting "options" are (:conc-name prefix)
;;;
(defun parse-defstruct (mod in-list)
  (let ((name (second in-list))
	(slot-specs (cddr in-list))
	(conc-name "")
	(setf-names nil))
       (if (listp name)
	   (progn
	    (register-class-definition mod (car name))
	    (setq conc-name (strcat (string (car name)) "-")))
	   (progn
	    (register-class-definition mod name)
	    (setq conc-name (strcat (string name) "-"))))
       ;; Look for :conc-name of :include in options list
       (when (listp name)
	     (dolist (option (cdr name))
		     (if (and (listp option) (eq (car option) :include))
			 (register-class-usage mod (cdr option)))
		     (if (and (listp option) (eq (car option) :conc-name))
			 (setq conc-name (string (cadr option))))))
       ;; Skip doc-string
       (if (stringp (car slot-specs))
	   (setf slot-specs (cdr slot-specs)))
       ;; Run down the slot specs, grabbing for setf names
       (dolist (slot-spec slot-specs)
	       (if (consp slot-spec)
		   (setq slot-spec (car slot-spec)))
	       (push (read-from-string (strcat conc-name (string slot-spec)))
		     setf-names))
       ;; All the setf functions in setf-names are defined in this file...
       (dolist (setf setf-names)
	       (register-setf-definition mod setf))))

;;;
;;; Parse a module
;;;
(defun parse-module (module)
  (let* ((src-pathname (make-source-pathname module))
	 (in-file (open src-pathname :direction :input)))
	(do* ((in-list (read in-file nil nil t) (read in-file nil nil t)))
	     ((null in-list))
	     (cond ((eql (first in-list) 'defun)
		    (parse-defun module in-list))
		   ((eql (first in-list) 'define-setf-method)
		    (parse-define-setf-method module in-list))
		   ((eql (first in-list) 'defsetf)
		    (parse-defsetf module in-list))
		   ((eql (first in-list) 'defstruct)
		    (parse-defstruct module in-list))
		   ((eql (first in-list) 'defmethod)
		    (parse-defmethod module in-list))
		   ((eql (first in-list) 'defhandler)
		    (parse-defmethod module in-list))
		   ((eql (first in-list) 'defclass)
		    (parse-defclass module in-list))
		   ((eql (first in-list) 'defmethod-setf)
		    (parse-defmethod-setf module in-list))
		   ((eql (first in-list) 'defmacro)
		    (parse-defmacro module in-list))
		   (t
		    (parse-body module in-list))))
	(close in-file)))

(defun show-macros-used-in (mod &aux file)
  (let ((name (module-name mod)))
       (dolist (macro (gethash mod *file-macro-usage-map*))
	       (setq file (gethash macro *macro-file-map*))
	       (if file
		   (format t "Macro ~s used in ~s, defined in ~s~%" 
			   macro name file)))))

(defun show-classes-used-in (mod &aux file)
  (let ((name (module-name mod)))
       (dolist (class (gethash mod *file-class-usage-map*))
	       (setq file (gethash class *class-file-map*))
	       (if file
		   (format t "Class ~s used in ~s, defined in ~s~%" 
			   class name file)))))

(defun show-setfs-used-in (mod &aux file)
  (let ((name (module-name mod)))
       (dolist (setf (gethash mod *file-setf-usage-map*))
	       (setq file (gethash setf *setf-file-map*))
	       (if file
		   (format t "Setf name ~s used in ~s, defined in ~s~%" 
			   setf name file)))))

;;;
;;; Resolve a modules dependencies
;;;
(defun resolve-module-dependencies (module)
  ;; allocate locals
  (let* ((fmu (gethash module *file-macro-usage-map*))
	 (fsu (gethash module *file-setf-usage-map*))
	 (fcu (gethash module *file-class-usage-map*))
	 (macro-defs nil)
	 (setf-defs nil)
	 (class-defs nil)
	 (recomp nil)
	 (la nil)
	 (lbc nil))
	(dolist (macro fmu)
		(let ((defmod (gethash macro *macro-file-map*)))
		     (if (and defmod (not (equal defmod module)))
			 (pushnew defmod macro-defs))))
	(dolist (class fcu)
		(let ((defmod (gethash class *class-file-map*)))
		     (if (and defmod (not (equal defmod module)))
			 (pushnew defmod class-defs))))
	(dolist (setf-name fsu)
		(let ((defmods (gethash setf-name *setf-file-map*)))
		     (if (and defmods (not (member module defmods)))
			 (pushnew defmods setf-defs :test #'equalp))))
	(setf recomp (union (module-recompile-on module) macro-defs)
	      la (union (module-load-after module) 
			(append class-defs setf-defs) :test #'equalp)
	      lbc (union (module-load-before-compile module)
			 (union (append class-defs setf-defs) macro-defs) 
			 :test #'equalp))
	(setf (module-recompile-on module) recomp
	      (module-load-after module) la
	      (module-load-before-compile module) lbc)))
