;;;
;;; PDS (Picasso defsystem) Utility
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/load.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 19:02:14 $
;;;

(in-package 'pds :use '(lisp))

;;;
;;; Load the load environment for a module
;;;
(defun load-load-environment (module &aux done)
  ;; Go through all files in the load-after list and
  ;; try to load them loading compiled modules if at all possible.
  ;; If we can't load the load environment, then we've got a
  ;; circularity in the dependency list.
  (dolist (lbc-mod (module-load-after module))
	  (setq done nil)
	  (if (listp lbc-mod)
	      ;; module is a list, load any in the list.
	      (progn
	       ;; Check to see if any of the need modules are already loaded
	       (dolist (mod lbc-mod)
		       (setq done (or done (module-loaded mod))))
	       ;; If no hit there, try to compile and load the module.
	       (dolist (mod lbc-mod)
		       (when (not done)
			     (if (compile-module mod :warn nil)
				 (setq done (load-module mod)))))
	       ;; If no hit there (couldn't compile because of circularity),
	       ;; try to just load the module.
	       (dolist (mod lbc-mod)
		       (if (not done)
			   (setq done (load-module mod)))))
	      ;; load single module
	      (unless (setq done (module-loaded lbc-mod))
		      (compile-module lbc-mod :warn nil)
		      (setq done (load-module lbc-mod))))
	  ;; Error if load failed...
	  (if (not done)
	      (circular module))))

;;;
;;; Load a module if needed
;;;
(defun load-module (module)
  (when *debug* 
	(dotimes (i *indent*) (format t "  "))
        (format t "(load-module ~s)~%" module)
	(force-output *standard-output*)
	)
  (incf *indent*)
  (let ((pathname (make-source/binary-pathname module)))
       (when (< (module-load-date module) (get-file-write-date pathname))
	     (setf (module-load-in-process module) t)
	     (load-load-environment module)
	     (setf (module-load-in-process module) nil)
	     (if *just-testing*
		 (progn
		  (format t "Loading file ~s~%" (pathname-name pathname))
		  (force-output *standard-output*))
		 (load pathname))
	     ;; set the module loaded flag and timestamp
	     (setf (module-loaded module) t
		   (module-load-date module) (get-file-write-date pathname))))
  (decf *indent*)
  (module-load-date module))

