;;;
;;; PDS (Picasso defsystem) Utility
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/file.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 19:02:12 $
;;;

(in-package 'pds :use '(lisp))

(defvar *write-dates* nil)

(defun init-file-cache ()
  (setq *write-dates* (make-hash-table :test #'equal)))

(defun get-file-write-date (pathname)
  (or (gethash pathname *write-dates*)
      (setf (gethash pathname *write-dates*)
	    (file-write-date pathname))))

(defun fake-file-write (pathname)
  (format t "Compiling file ~s~%" (pathname-name pathname))
  (setf (gethash pathname *write-dates*) (get-universal-time)))

;;;
;;; module path utilities
;;;
(defun make-module-pathname (module &aux mpath)
  "return the module pathname for the module"
  (setq mpath (module-pathname module))
  (if (not (pathnamep mpath))
      (let ((spath (system-src-pathname (module-system module)))
	    (subdir (module-sub-directory module)))
	   ;; validate pathname for module
	   (cond ((null mpath)
		  (setq mpath (make-pathname
			       :directory (pathname-directory 
					   (strcat spath subdir "/"))
			       :name (string (module-name module)))))
		 ((and (stringp mpath) (not (pathnamep mpath)))
		  (setq mpath (pathname mpath)))
		 ((pathnamep mpath))
		 (t
		  (error "invalid pathname ~s" mpath)))
	   (setf (module-pathname module) mpath)))
  ;; return mpath
  mpath)

(defun make-source-pathname (module &aux spath)
  "return the source pathname"
  (setq spath (module-source-pathname module))
  (unless (pathnamep spath)
	  (setq spath
		(let ((mpath (make-module-pathname module)))
		     (make-pathname
		      :directory (pathname-directory mpath)
		      :name (pathname-name mpath)
		      :type (car *suffixes*))))
	  (setf (module-source-pathname module) spath))
  spath)

(defun make-binary-pathname (module &aux bpath)
  "return the source pathname"
  (setq bpath (module-binary-pathname module))
  (unless (pathnamep bpath)
	  (setq bpath
		(let ((bpath (system-bin-pathname (module-system module)))
		      (subdir (module-sub-directory module))
		      (mpath (make-module-pathname module)))
		     (make-pathname
		      :directory (strcat bpath subdir "/")
		      :name (pathname-name mpath)
		      :type (cdr *suffixes*))))
	  (setf (module-binary-pathname module) bpath))
  bpath)

(defun make-source/binary-pathname (module)
  "return the pathname to the newest of source/binary of a module"
  (let ((sname (make-source-pathname module))
	(bname (make-binary-pathname module)))
       ;; get the source and binary timestamps
       (let ((sdate (get-file-write-date sname))
	     (bdate (get-file-write-date bname)))
	    ;; return name of file to load
	    (if (and sdate bdate)
		;; both exist - test if source is newer than binary
		(if (> sdate bdate)
		    sname
		    bname)
		;; return name of existing file
		(if bdate bname
		    (if sdate sname
			;; no source or binary
			(error "can't find any file for module ~a" 
			       (module-name module))))))))
