;;;
;;; PDS (Picasso defsystem) Utility
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/compile.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/22 19:01:57 $
;;;

(in-package 'pds :use '(lisp))

(defun module-needs-compiling (module)
  (let* ((recomp-on (module-recompile-on module))
	 (s-pathname (make-source-pathname module))
	 (b-pathname (make-binary-pathname module))
	 (s-date (get-file-write-date s-pathname))
	 (b-date (get-file-write-date b-pathname)))
	;; Make sure source exists.
	(if (null s-date)
	    (error "source file ~a not found." s-pathname))
	;; Needs to be recompiled if source is younger than binary
	(if (or (null b-date) (< b-date s-date))
	    (return-from module-needs-compiling t))
	;; Needs to be recompiled if any of the recomp-on files are younger...
	(dolist (mod recomp-on)
		(if (< b-date 
		       (get-file-write-date (make-source-pathname mod)))
		    (return-from module-needs-compiling t))))
  nil)

(defun load-compile-environment (module &aux done)
  ;; Go through all files in the load-before-compile list and
  ;; try to load them loading compiled modules if at all possible.
  ;; If we can't load the compile environment, then we've got a
  ;; circularity in the dependency list.
  (dolist (lbc-mod (module-load-before-compile module))
	  (setq done nil)
	  (if (listp lbc-mod)
	      ;; module is a list, load any in the list.
	      (progn
	       ;; Check to see if any of the need modules are already loaded
	       (dolist (mod lbc-mod)
		       (setq done (or done (module-loaded mod))))
	       ;; If no hit there, try to compile and load the module.
	       (dolist (mod lbc-mod)
		       (when (not done)
			     (if (compile-module mod :warn nil)
				 (setq done (load-module mod)))))
	       ;; If no hit there (couldn't compile because of circularity),
	       ;; try to just load the module.
	       (dolist (mod lbc-mod)
		       (if (not done)
			   (setq done (load-module mod)))))
	      ;; load single module
	      (unless (setq done (module-loaded lbc-mod))
		      (compile-module lbc-mod :warn nil)
		      (setq done (load-module lbc-mod))))
	  ;; Error if load failed...
	  (if (not done)
	      (circular module))))


;;;
;;; Compile a module if needed
;;;
(defun compile-module (module &key (warn t))
  (when (module-compile-in-process module)
	(if warn (warn "module ~a is already compiling" module))
	(return-from compile-module nil))
  (if (not (module-needs-compiling module))
      (return-from compile-module))
  (when *debug* 
	(dotimes (i *indent*) (format t "  "))
	(format t "Compiling module ~s~%" module)
	(force-output *standard-output*))
  (incf *indent*)
  (let ((system (module-system module))
	(s-pathname (make-source-pathname module))
	(b-pathname (make-binary-pathname module)))
       (setf (module-compile-in-process module) t)
       (load-compile-environment module)
       (setf (module-compile-in-process module) nil)
       (if (not *just-testing*)
	   (compile-file s-pathname :output-file b-pathname))
       (fake-file-write b-pathname))
  (decf *indent*)
  t)
