#-cling (in-package 'pt :use '(pcl lisp))
#+cling (in-package 'pt :use '(pcl lisp cling))

;;;
;;; Define the class hierarchy for the facility database schema.
;;;

(defclass junct-kind (fac-object)
  ((bitmaps :initform nil :type t :accessor bitmaps)
   (id :initform 0 :type integer :accessor id)
   (tuple :initform nil :type list :accessor tuple)))

(defun junct-kind-p (x) (typep x 'junct-kind))

(defclass junction (fac-object)
  ((id :initform 0 :type integer :accessor id)
   (util-id :initform 0 :type integer :accessor util-id)
   (kind :initform nil :type string :accessor kind)
   (kind-id :initform 0 :type integer :accessor kind-id)
   (vertex-id :initform nil :type symbol :accessor vertex-id)
   (tuple :initform nil :type list :accessor tuple)))

(defun junction-p (x) (typep x 'junction))

(defclass eq-junction (junction)
  ((equip-id :initform 0 :type integer :accessor equip-id)))

(defun eq-junction-p (x) (typep x 'eq-junction))

;;; =============================================================
;;;
;;; Convert a eq-junct tuple to a CLOS eq-junction object.
;;;
(defun make-eq-junct-from-tuple (tuple)
  ;; Tuples are of form (id uid x1 y1 kind eq-id)
  (let* ((id (first tuple))
	 (uid (second tuple))
	 (x (nth 2 tuple))
	 (y (nth 3 tuple))
	 (kind-id (nth 4 tuple))
	 (eq-id (nth 5 tuple))
	 (v (new-sgg-vertex x y))
	 (sym (new-sgg-symbol kind-id v))
	 rv)
	(setq rv (make-instance 'eq-junction
				:id id
				:util-id uid
				:tuple tuple
				:vertex-id v
				:kind-id kind-id
				:equip-id eq-id))
	(new-sgg-object :id rv
			:line-ids nil
			:annot-ids nil
			:symbol-ids `(,sym)
			:selectable t
			:visible t
			:color "white"
			:line-width 0)
	rv))

;;;
;;; Read all the equipment junctions from ~picasso/lib/db/eq-junct
;;; Return a list of the objects read.
;;;
#-cling
(defun read-db-eq-junct (fac-id)
  "Read the equipment junctions from the database for this facility and build the equipment junction objects."
  (let* ((in-file (open "~picasso/lib/db/eq-junct" :direction :input))
	 (rv (mapcar #'make-eq-junct-from-tuple (read in-file nil nil t))))
	(close in-file)
       ;; Set the fac-id.  Also need to copy in the geometric data.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

#+cling
(defun read-db-eq-junct (fac-id)
  "Read the equipment junctions from the database for this facility and build the equipment junction objects."
  (let ((rv (db-mapl-retrieve
	     #'make-eq-junct-from-tuple
	     '((:integer junction.id)
	       (:integer junction.utility_id)
	       (:float junction.x)
	       (:float junction.y)
	       (:integer junction.kind)
	       (:integer equip_junct.e_id))
	     :where '(= equip_junct.j_id junction.id))))
       ;; Set the fac-id.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

;;;
;;; Save all the equipment junctions to the file ~picasso/lib/db/eq-junct
;;;
(defun save-db-eq-junct (ejlist)
  (let ((tuple-list (mapcar #'tuple ejlist))
	(out-file (open "~picasso/lib/db/eq-junct" :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

;;; =============================================================
;;;
;;; Convert a junction tuple to a CLOS junction object.
;;;
(defun make-junct-from-tuple (tuple)
  ;; Tuples are of form (id uid x1 y1 kind)
  (let* ((id (first tuple))
	 (uid (second tuple))
	 (x (nth 2 tuple))
	 (y (nth 3 tuple))
	 (kind-id (nth 4 tuple))
	 (v (new-sgg-vertex x y))
	 (sym (new-sgg-symbol kind-id v))
	 rv)
	(setq rv (make-instance 'junction
				:id id
				:util-id uid
				:vertex-id v
				:tuple tuple
				:kind-id kind-id))
	(new-sgg-object :id rv
			:line-ids nil
			:annot-ids nil
			:symbol-ids `(,sym)
			:selectable t
			:visible t
			:color "white"
			:line-width 0)
	rv))

;;;
;;; Read all the junctions from the database -- faked, for now.
;;; Return a list of the objects read.
;;;
#+cling
(defun read-db-junct (fac-id)
  "Read the junctions from the database for this facility and build the junction objects."
  (let ((rv (db-mapl-retrieve
	     #'make-junct-from-tuple
	     '((:integer junction.id)
	       (:integer junction.utility_id)
	       (:float junction.x)
	       (:float junction.y)
	       (:integer junction.kind)))))

       ;; Set the fac-id.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

#-cling
(defun read-db-junct (fac-id)
  "Read the junctions from the database for this facility and build the junction objects."
  (let* ((in-file (open "~picasso/lib/db/junct" :direction :input))
	 (rv (mapcar #'make-junct-from-tuple (read in-file nil nil t))))
	(close in-file)
       ;; Set the fac-id.  Also need to copy in the geometric data.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

;;;
;;; Save all the junctions to the file ~picasso/lib/db/junct
;;;
(defun save-db-junct (jlist)
  (let ((tuple-list (mapcar #'tuple jlist))
	(out-file (open "~picasso/lib/db/junct" :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

;;; =============================================================
(defun find-junct-kind-width (name)
  (float 
   (cond ((string= name "sprinkler") 1)
	 ((string= name "shut-off") 1)
	 ((string= name "end cap") 1)
	 ((string= name "junction") 1)
	 ((string= name "bulk head") 1)
	 ((string= name "imaginary") 1))))

(defun find-junct-kind-height (name)
  (float 
   (cond ((string= name "sprinkler") 1)
	 ((string= name "shut-off") 1)
	 ((string= name "end cap") 1)
	 ((string= name "junction") 1)
	 ((string= name "bulk head") 1)
	 ((string= name "imaginary") 1))))

(defvar *sprinkler-bitmaps*
  (list (make-image :file "sprinkler-big.bitmap")
	(make-image :file "sprinkler-med.bitmap")
	(make-image :file "sprinkler-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defvar *shut-off-bitmaps*
  (list (make-image :file "shut-off-big.bitmap")
	(make-image :file "shut-off-med.bitmap")
	(make-image :file "shut-off-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defvar *end-cap-bitmaps*
  (list (make-image :file "end-cap-big.bitmap")
	(make-image :file "end-cap-med.bitmap")
	(make-image :file "end-cap-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defvar *junction-bitmaps*
  (list (make-image :file "junction-big.bitmap")
	(make-image :file "junction-med.bitmap")
	(make-image :file "junction-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defvar *bulk-head-bitmaps*
  (list (make-image :file "bulk-head-big.bitmap")
	(make-image :file "bulk-head-med.bitmap")
	(make-image :file "bulk-head-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defvar *imaginary-bitmaps*
  (list (make-image :file "imaginary-big.bitmap")
	(make-image :file "imaginary-med.bitmap")
	(make-image :file "imaginary-small.bitmap")
	(make-image :file "tiny.bitmap")))

(defun find-junct-kind-bitmaps (name)
  (cond ((string= name "sprinkler") *sprinkler-bitmaps*)
	((string= name "shut-off") *shut-off-bitmaps*)
	((string= name "end cap") *end-cap-bitmaps*)
	((string= name "junction") *junction-bitmaps*)
	((string= name "bulk head") *bulk-head-bitmaps*)
	((string= name "imaginary") *imaginary-bitmaps*)))

;;;
;;; Convert a junction kind tuple to a CLOS junction kind object.
;;;
(defun make-junct-kind-from-tuple (tuple)
  ;; Tuples are of form (id name)
  (let* ((id (first tuple))
	 (rv nil)
	 (name (trim-str (second tuple)))
	 (sw (find-junct-kind-width name))
	 (sh (find-junct-kind-height name)))
	(new-sgg-symbol-type :id id :width sw :height sh
			     :bitmaps (find-junct-kind-bitmaps name))
	(setq rv (make-instance 'junct-kind :tuple tuple :id id :name name))
	rv))

;;;
;;; Read all the junction kinds from the database -- faked, for now.
;;; Return a list of the objects read.
;;;
#+cling
(defun read-db-junct-kind (fac-id)
  "Read the junct-kind objects from the database for this facility and build the CLOS junct-kind objects."
  (let ((rv (db-mapl-retrieve #'make-junct-kind-from-tuple
			      '((:integer junct_kind.code)
				(:string junct_kind.name)))))
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))


#-cling
(defun read-db-junct-kind (fac-id)
  "Read the junctions from the database for this facility and build the junction objects."
  (let* ((in-file (open "~picasso/lib/db/junct-kind" :direction :input))
	 (rv (mapcar #'make-junct-kind-from-tuple (read in-file nil nil t))))
	(close in-file)
	(dolist (obj rv) (setf (fac-id obj) fac-id))
	rv))

(defun save-db-junct-kind (jk-list)
  (let ((tuple-list (mapcar #'tuple jk-list))
	(out-file (open "~picasso/lib/db/junct-kind" :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

;;; =============================================================
;;;
;;; Resolve all the graphical objects
;;;
(defun resolve-juncts (junct-list kind-tab)
  (dolist (j junct-list)
	  (let* ((kind-id (kind-id j))
		 (kind (gethash kind-id kind-tab))
		 (name "Bogus"))
		(if kind
		    (setf name (name kind))
		    (warn "Junction kind ~s not found.~%" kind-id))
		(setf (kind j) name))))

(defun delete-dup-junct (junct-list eq-junct-list)
  (delete-if #'(lambda (x) (member x eq-junct-list :key #'id))
	     junct-list))
