;;;
;;; Copyright (c) 1982 Regents of the University of California
;;; Copyright (c) 1987 Regents of the University of California
;;;
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 

(in-package 'cling)

(defvar *binary-ops* '(+ - * / ** > < >= <= = != and or))

;;;
;;; Utility functions:
;;;

;;;
;;;; state-check
;;;
;;; This function checks the state of cling before a quel command takes
;;; place.  An error is returned if
;;; there is a is no database open.  If a retrieve is in progress, it is
;;; killed to prevent errors.
;;;

(defun state-check ()
   (when retrieve-in-progress
         (ii-break)
	 (setf retrieve-in-progress NIL))
   (when (Not current-db-name) (error "No database has been opened")))

;;;
;;;; LP & RP
;;;
;;; These keep the code cleaner.  Parentheses in strings confuse the editor.
;;;
(defun write-out-lp ()
  (write-out "(" ;)
	     ))

(defun write-out-rp ()
  (write-out ;(
	       ")"))

;;;
;;;; STRING
;;;
;;; Given a string'able argument, put it out with quotes around it.
;;; The user should take care that the string is suitable for ingres.
;;; Thus, the following convertions must be made from C print style to
;;; lisp print style.
;;; Each backslash not escaping a quote must be doubled.  Each backslash
;;; escaping a quote must be tripled.  Of course, you don't have to build
;;; a string using the read syntax -- you can just construct one.
;;; Examples:
;;;  "hello" -> "hello"
;;;  "hello\n" -> "hello\\n"
;;;  "\"hello\"" -> "\\\"hello\\\""
;;;  "\\\"hello\"\n" -> "\\\\\\\"hello\\\"\\n"
;;;
(defun write-out-string (str)
  (write-out "\"")
  (write-out (string str))
  (write-out "\""))

;;;
;;;; NUMBER
;;;
;;; Handle doubles specially by changing 'd' to 'e'
;;;
(defun write-out-number (value)
  (if (typep value 'double-float)
      (write-out (nsubstitute #\e #\d (prin1-to-string value)
			      :test 'char-equal))
      (write-out (prin1-to-string value))))

;;;
;;;; FILE-NAME
;;;
;;; Used by the copy command.  Takes a string as input.
;;; Try to be helpful by prepending the current directory to the name
;;; if the name isn't a full pathname [since ingres will bitch and moan
;;; if it doesn't get a full pathname].
;;;
(defun write-out-file-name (name)
  (write-out-string (full-path-name name)))

;;;
;;;; NAME
;;;
;;; This takes a symbol or a string and prints it out, without quotes,
;;; to a maximum of 12 characters.  This keeps ingres from complaining
;;; about names being too long.  Names include relations, attribute,
;;; variable, indexes, and locations.
;;; If the item could be a column name it should be sent to write-out-column.
;;;
(defun write-out-name (name &aux (s (string name)))
  (if (> (length s) max-ingres-name-length)
      (write-out (subseq s 0 max-ingres-name-length))
      (write-out s)))

;;;
;;;; COLUMN
;;;
;;; The argument must be either a symbol, string, or a dotted pair.
;;; If there is no dot, then the argument is sent to write-out-name.
;;; Otherwise, each half is printed, up to 12 characters, with a dot
;;; separating them.
;;; ### Should use a subroutine that breaks the argument up into strings.
;;;
(defun write-out-column (name &aux dot-posn)
  (if (consp name)	; structural version
      (progn
       (write-out-name (first name))
       (write-out ".")
       (write-out-name (rest name)))
      (progn
       (setf name (string name))	; textual version
       (if (setf dot-posn (position #\. name))	; there's a dot
	   (progn
	    (write-out (subseq name 0 (min dot-posn max-ingres-name-length)))
	    (write-out (subseq name dot-posn
			       (+ dot-posn (min (- (length name) dot-posn)
						(1+ max-ingres-name-length))))))
	   (write-out-name name)))))

;;;
;;;; LOCATION
;;;
;;; The argument is structured the same as a column, ie, either a
;;; symbol, string, or dotted pair.
;;; However, in printing the result, a colon is used to separate the names.
;;;
(defun write-out-location (location &aux dot-posn)
  (if (consp location)	; structural version
      (progn
       (write-out-name (first location))
       (write-out ":")
       (write-out-name (second location)))
      (progn
       (setf location (string location))
       (if (setf dot-posn (position #\. location))
	   (progn
	    (write-out (subseq location 0
			       (min dot-posn max-ingres-name-length)))
	    (write-out ":")
	    (write-out (subseq location (1+ dot-posn)
			       (+ dot-posn (min (- (length location)
						   dot-posn)
						(1+ max-ingres-name-length))))))
	   (write-out-name location)))))

;;;
;;;; SEQUENCE
;;;
;;; Use the function argument to print a comma separated version of the list.
;;;
(defun write-out-sequence (list function)
  (do ((l list (rest l))
       )
      ((null l)
       )
      (funcall function (first l))
      (if (rest l) (write-out ","))))

;;;
;;;; LIST
;;;
;;; Surround the comma separated list by parentheses.
;;; Same arguments as to write-out-sequence.
;;;
(defun write-out-list (list function)
  (write-out-lp)
  (write-out-sequence list function)
  (write-out-rp))

;;;
;;;; ITEM
;;;
;;; This differs from an expression in that strings are printed without
;;; quotes.  Lists aren't supported either.  Used for savepoints,
;;; permit elements, format and other '=' lists, or anything that isn't
;;; an expression and doesn't need quotes around strings.
;;;
(defun write-out-item (item)
  (typecase
   item
   (number (write-out-number item))
   (atom (write-out-name item))
   (t (error "A list is not valid here: ~s" item))))

;;;
;;;; EXPRESSION
;;;
;;; Given an expression or a qualification, print it out.
;;; The atomic elements are strings and numbers and column names, which
;;; are either symbols or dotted pairs.
;;; Otherwise, we have a function, which is either unary, infix, or
;;; functional.  Functions may be aggregate, in which case :by and :where
;;; introduce new material - each takes one argument.
;;; Function names are treated as items, and thus could be strings.
;;; All other strings get quotes around them.
;;; Binary operators must be symbols, however.
;;;
(defun write-out-expression (expression)
  (typecase
   expression
   (string (write-out-string expression))
   (number (write-out-number expression))
   (atom (write-out-column expression))
   (list 
    (if (not (listp (rest expression)))
	(write-out-column expression)	; is a dotted pair column name
	(case (length expression)
	      ;; put parenthses around something
	      (1 (write-out-lp)
		 (write-out-expression (first expression))
		 (write-out-rp))
	      ;; function notation, one argument, works for 'not', too.
	      (2 (if (eq (first expression) '-)  ; check for unary minus
		     (progn
		      (write-out-lp)
		      (write-out "-")
		      (write-out-expression (second expression))
		      (write-out-rp))
		     (progn
		      (write-out-item (first expression))
		      (write-out-lp)
		      (write-out-expression (second expression))
		      (write-out-rp))))
	      ;; function, two args, either infix or functional
	      (3 (if (member (first expression) *binary-ops*)
		     (progn
		      (write-out-lp)
		      (write-out-expression (second expression))
		      (write-out " ")
		      (write-out-item (first expression))
		      (write-out " ")
		      (write-out-expression (third expression))
		      (write-out-rp))
		     (progn
		      (write-out-item (first expression))
		      (write-out-list (rest expression)
				      'write-out-expression))))
	      ;; function, lots of args, check for aggregate function
	      (t (cond
		  ((member (first expression) *binary-ops*)
		   (write-out-lp)
		   (do ((operator (first expression))
			(l (rest expression) (rest l))
			)
		       ((null l)
			)
		       (write-out-expression (first l))
		       (when (rest l)
			     (write-out " ") ; needed for 'and' and 'or'
			     (write-out-expression operator)
			     (write-out " "))
		       )
		   (write-out-rp))
		  ((or (member :by expression)
		       (member :where expression))
		   (write-out-aggregate expression))
		  (t
		   (write-out-item (first expresssion))
		   (write-out-list (rest expression)
				   'write-out-expression)))))))))


;;;
;;;; AGGREGATE
;;;
;;; This method of printing a function call is invoked when the
;;; :by or :where keywords are detected.
;;; The call looks like '(function argument [:by var|(vars)][:where qual])
;;;
(defun write-out-aggregate (aggregate)
  (write-out-name (pop aggregate))
  (write-out-lp)
  (write-out-expression (pop aggregate))
  (when (eq :by (first aggregate))
	(write-out " by ")
	(write-out-sequence (listify (second aggregate)) 'write-out-column)
	(setf aggregate (nthcdr 2 aggregate)))
  (when (eq :where (first aggregate))
	(write-out-qual (second aggregate)))
  (write-out-rp))

;;;
;;;; TARGET-LIST
;;;
;;; A target list is a list of expressions, with a few exceptions at the
;;; top level.  An element may be a magic function (:'type') that takes
;;; one argument that is a top-level element.
;;; Otherwise, a top level element must be a column name or the '=' function.
;;; 
(defun write-out-target-list (list)
  (if list
      (write-out-list list 'write-out-target-element)
      (error "Target list cannot be nil")))

(defun write-out-target-element (element)
  (cond ((consp element)
	 (cond ((not (listp (rest element)))
		(write-out-column element))
	       ((member (first element) '(:integer :float :string))
		(write-out-target-element (second element)))
	       ((eq (first element) '=)
		(write-out-name (second element))
		(write-out "=")
		(write-out-expression (third element)))
	       (t
		(error "Invalid target-list element: ~s" element))))
	((atom element)
	 (write-out-column element))
	(t
	 (error "Invalid targaet-list element: ~s" element))))

;;;
;;;; QUAL
;;;
;;; The second argument is used for integrities, where "where" is not used.
;;; The qualification is generally optional.
;;;
(defun write-out-qual (qual &optional (intro " where "))
  (unless (null qual)
	  (write-out intro)
	  (write-out-expression qual)))


;;;
;;;; SORT LIST ONLY
;;;
;;; Print out sort list with no sort prepended
;;;
(defun write-out-sort-list (sort)
  (unless (null sort) (write-out-sequence (listify sort) 'write-out-sort-item)))

;;;
;;;; SORT
;;;
;;; Print out the sort clause to a retrieve statement or the 'on' clause
;;; of a modify statement, if one was given.
;;; The names are assumed to be consistent with one of the column headings.
;;; The argument is  name | ({name | (name :sortorder)}+) .
;;; The sortorder must be an item whose fist letter is 'a' or 'd'.
;;;
(defun write-out-sort (sort &optional (intro " sort "))
  (unless (null sort)
	  (write-out intro)
	  (write-out-sequence (listify sort) 'write-out-sort-item)))

(defun write-out-sort-item (item)
  (if (consp item)
      (progn
       (write-out-name (first item))
       (write-out ":")
       (write-out (subseq (string (second item)) 0 1)))
      (write-out-name item)))

;;;
;;;; =-LIST
;;;
;;; Each element is a list of length two.
;;; The two items are printed separated by a '='.
;;; Used by create, format, and modify.
;;; Target lists are handled separately, thus both elements
;;; are printed as items, not expressions.
;;;
(defun write-out-=-list (list)
  (write-out-list list 'write-out-=-element))

(defun write-out-=-element (element)
  (write-out-item (first element))
  (write-out "=")
  (write-out-item (second element)))

;
; ************************
; Ingres (Equel) interface
; ************************
;

;;;
;;;; RECORD-RANGE-VAR
;;;
;;; Assuming we're paying attention to declarations made by the user,
;;; this allows us to look at a range variable and know which table
;;; it is indexing.
;;;
(defun record-range-var (var relation)
  )

;;;
;;;; DETERMINE-TARGET-TYPES
;;;
;;; Given a target list, return a list of equal length containing the
;;; elements :string, :integer, and :float.
;;;
(defun determine-target-types (target-list)
  (mapcar #'(lambda (element)
		    (if (listp element)
			(cond ((eq (first element) :integer)
			       :integer)
			      ((eq (first element) :float)
			       :float)
			      (t
			       :string))
			:string))
	  target-list
	  ))

;;;
;;;; RETRIEVE-RESULT-FROM-INGRES
;;;
;;; This is used by db-retrieve to return results in list form
;;; Use a tail pointer to efficiently construct the list.
;;;
(defun retrieve-result-from-ingres (target-list &aux tail-ptr result)
  (setf result (cons nil nil))	; dummy cons to make things simpler
  (setf tail-ptr result)	; the last cons appended to the list
  (map-retrieve #'(lambda (x &aux (c (cons x nil)))
			  (setf (rest tail-ptr) c)
			  (setf tail-ptr c))	; non-nil return value
		target-list)
  (rest result))

    
;;;
;;;; MAPL-RETRIEVE
;;;
;;; This is used by db-mapl-retrieve to return results in list form.
;;; MAPL-RETRIEVE applies a function to every element in the list.
;;; Use a tail pointer to efficiently construct the list.
;;;
(defun mapl-retrieve (function target-list &aux tail-ptr result)
  (setf result (cons nil nil))	; dummy cons to make things simpler
  (setf tail-ptr result)	; the last cons appended to the list
  (map-retrieve-nobreak #'(lambda (x &aux (c (cons (funcall function x) nil)))
				     (setf (rest tail-ptr) c)
				     (setf tail-ptr c))	; non-nil return value
		target-list)
  (setq retrieve-in-progress NIL)
  (rest result))

;;;
;;;; MAPN-RETRIEVE
;;;
;;; This is used by db-mapl-retrieve to return results in a nconc'ed list.
;;; MAPN-RETRIEVE applies a function to every element in the list, and
;;; the results are nconc'ed together.
;;;
(defun mapn-retrieve (function target-list &aux result)
  (setf result '(nil))
  (map-retrieve-nobreak #'(lambda (x &aux (c (nconc 
					    result (funcall function x))))
				     (setf result c))		
			target-list)
  (setq retrieve-in-progress NIL)
  (rest result))

;;;
;;;; MAP-RETRIEVE
;;;
;;; Once the query has been written out, apply the given function
;;; to a list containing each returned tuple.
;;; The target list is passed so that the number of returned domains and
;;; their types can be determined.
;;; If the function wants to kill the retrieve, it should return nil.
;;;
(defun map-retrieve (function target-list
			      &aux (target-types
				    (determine-target-types target-list)))
  (ii-retinit 0 0)
  (when (eql 0 (ii-errtest))
	(do ((tuple-count 0 (1+ tuple-count))
	     (cur-tuple nil)
	     )
	    ((eql 0 (ii-nextget))
	     (ii-flush 0 0)
	     )
	    (if (setf cur-tuple (retrieve-tuple target-types))
		(unless (funcall function cur-tuple)
			(ii-break)
			(return))))	; break loop
	)
  (setq retrieve-in-progress NIL))

(defvar *retrieve-string-buffer* (make-string 2001))

;;;
;;;; MAP-RETRIEVE-NOBREAK
;;;
;;; This function takes a function and a target list and
;;; applies to it to every tuple of a query.  Unlike map-retrieve, it
;;; does not kill the retrieve when the function returns nil.
;;; The query must first be written out before this function is used.

(defun map-retrieve-nobreak (function target-list
				      &aux (target-types
				      (determine-target-types target-list)))
  (ii-retinit 0 0)
  (when (eql 0 (ii-errtest))
	(do ((tuple-count 0 (1+ tuple-count))
	     (cur-tuple nil)
	     )
	    ((eql 0 (ii-nextget))
	     (ii-flush 0 0)
	     )
	    (setf cur-tuple (retrieve-tuple target-types))
	    (funcall function cur-tuple)))
  (setq retrieve-in-progress NIL))


;;;
;;;; RETRIEVE-TUPLE
;;;
;;; Given that the retrieve loop has been initialized, snarf
;;; the required results from the backend.  Result is a list of the
;;; same length as the target-types argument.
;;; Any errors cause the return value to be nil.
;;;
(defun retrieve-tuple (target-types &aux result)
  (setf result
	(mapcar
	 #'(lambda (item-type &aux str-len)
		   (case item-type
			 (:float (ii-retf))
			 (:integer (ii-reti))
			 (:string
			  (setf str-len (ii-rets *retrieve-string-buffer*))
			  (subseq *retrieve-string-buffer* 0 str-len))
			 (t
			  (error "Unknown target type: ~s" item-type))))
	 target-types))
  (if (eql 0 (ii-errtest))
      result))

;;;
;;;; SEND-QUERY
;;;
(defun send-query (targetlist unique qual sort)
  (write-out "retrieve ")
  (if unique (write-out "unique "))
  (write-out-target-list targetlist)
  (write-out-qual qual)
  (write-out-sort sort)
  (setq retrieve-in-progress T))
