;;;
;;; Copyright (c) 1982 Regents of the University of California
;;; Copyright (c) 1987 Regents of the University of California
;;;
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 

(in-package 'cling)

;;;
;;; These symbols are all functions.
;;;
(export '(          
	  db-createdb db-destroydb db-sysmod
	  db-open db-close
	  db-create db-delete db-copy db-relocate
	  db-help db-destroy 
	  db-permit db-integrity
	  db-destroy-permit db-destroy-integrity
	  db-print
	  db-index db-modify db-save
	  db-begin-xact db-end-xact db-abort db-savepoint
	  db-range db-view
	  db-append db-replace db-retrieve
	  db-map-retrieve db-mapl-retrieve db-mapn-retrieve
	  db-set-journaling db-set-nojournaling
	  db-set-joinop db-set-nojoinop
	  db-set-ret-into
	  db-set-deadlock db-set-nodeadlock
	  db-set-lockmode
	  db-set-qbufsize db-set-query-size
	  db-set-cache-local
	  end-command
	  ))
;;;
;;; These symbols are exported so that internal routines may compare
;;; for 'eq'uality.  Thus, any target lists or qualifications sent to
;;; cling should have been read (or looked up) while using the cling package.
;;;
(export '(
	  + - * / **
	  = != > >= < <=
	  and or
	  ))

;;;
;;; User visible routines:
;;;

;;;
;;;  Unix type commands:
;;;
(defun db-createdb (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "createdb" dbname flags))))
	  
(defun db-destroydb (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "destroydb" dbname flags))))

(defun db-sysmod (dbname &key (flags (make-string 0)))
  (record-time
   (eql 0 (system-level-db-command "sysmod" dbname flags))))

;;;
;;; Pass the strings to myIIingopen.
;;; Also record the current database name for those who may wish to know
;;; (or remember).
;;;
(defun db-open (dbname &key (flags (make-string 0)))
  (record-time
   (when current-db-name (error "A database is already open"))
   (ii-ingopen (string dbname) (string flags))
   (setq current-db-name (string dbname))
   t))

;;;
;;; Major action here is to call IIexit.
;;;
(defun db-close ()
  (record-time
   (setq current-db-name nil)
   (ii-exit)
  t))

;;;;
;;;; The following functions perform basic quel operations.
;;;; Alphabetical order.
;;;;

;;;
;;;; ABORT
;;;
(defun db-abort (&key to)
  (record-time
   (state-check)
   (write-out "abort")
   (when to
	 (write-out " to ")
	 (write-out-item to))
   (end-command)))

;;;
;;;; APPEND
;;;
;;; Takes a relname, not a variable name.
;;;
(defun db-append (relname targetlist &key where)
  (record-time
   (state-check)
   (write-out "append ")
   (write-out-name relname)
   (write-out-target-list targetlist)
   (write-out-qual where)
   (end-command)))

;;;
;;;; BEGIN TRANSACTION
;;;
(defun db-begin-xact ()
  (record-time
   (state-check)
   (write-out "begin transaction")
   (end-command)))

;;;
;;;; COPY
;;;
;;; Format is a list structure, not a string.
;;;
(defun db-copy (relation format &key into from)
  (record-time
   (state-check)
   (write-out "copy ")
   (write-out-name relation)
   (write-out-=-list format)
   (write-out (if into " into " " from "))
   (write-out-file-name (string (if into into from)))
   (end-command)))

;;;
;;;; CREATE
;;;
(defun db-create (relation format &key journaling)
  (record-time
   (state-check)
   (write-out "create ")
   (write-out-location relation)
   (write-out-=-list format)
   (if journaling (write-out "with journaling"))
   (end-command)))

;;;
;;;; DELETE
;;;
(defun db-delete (relation &key where)
  (record-time
   (state-check)
   (write-out "delete ")
   (write-out-name relation)
   (write-out-qual where)
   (end-command)))

;;;
;;;; DESTROY
;;;
(defun db-destroy (&rest relations)
  (record-time
   (state-check)
   (write-out "destroy ")
   (write-out-sequence relations 'write-out-name)
   (end-command)))

;;;
;;;; DESTROY-PERMIT
;;;
;;; The permits argument must be an integer, a list of integers,
;;; or else the keyword :all.
;;;
(defun db-destroy-permit (relation permits)
  (record-time
   (state-check)
   (write-out "destroy permit ")
   (write-out-name relation)
   (write-out " ")
   (if (eq permits 'all)
       (write-out "all")
       (write-out-sequence (listify permits) 'write-out-number))
   (end-command)))

;;;
;;;; DESTROY-INTEGRITY
;;;
;;; Similar to db-destroy-permit
;;;
(defun db-destroy-integrity (relation integrities)
  (record-time
   (state-check)
   (write-out "destroy integrity ")
   (write-out-name relation)
   (write-out " ")
   (if (eq integrities 'all)
       (write-out "all")
       (write-out-sequence (listify integrities) 'write-out-number))
   (end-command)))

;;;
;;;; END TRANSACTION
;;;
(defun db-end-xact ()
  (record-time
   (state-check)
   (write-out "end transaction")
   (end-command)))

;;;
;;;; HELP
;;;
;;; db-help [all | tablename+]
;;; db-help :permit | :view | :integrity tablename+
;;;
(defun db-help (&rest items)
  (record-time
   (state-check)
   (cond ((null items)
	  (write-out "help"))
	 (t
	  (write-out "help ")
	  (case (first items)
		(:permit (write-out "permit "))
		(:integrity (write-out "integrity "))
		(:view (write-out "view "))
		(t
		 (setf items (cons nil items))))	; make dummy first elt
	  (write-out-sequence (rest items) 'write-out-name)))
   (end-command)))

;;;
;;;; INDEX
;;;
;;; db-index rel index (colname+)
;;;
(defun db-index (relation index domains)
  (record-time
   (state-check)
   (write-out "index on ")
   (write-out-name relation)
   (write-out "=")
   (write-out-location index)
   (write-out-list domains 'write-out-name)
   (end-command)))

;;;
;;;; INTEGRITY
;;;
;;; db-integrity var (qual)
;;;
(defun db-integrity (var qual)
  (record-time
   (state-check)
   (write-out "define integrity ")
   (write-out-name var)
   (write-out-qual qual " ")	; suppress "where" with second argument
   (end-command)))

;;;
;;;; MODIFY
;;;
(defun db-modify (relation structure &key on unique fillfactor minpages
			   maxpages indexfill maxindexfill)
  (record-time
   (state-check)
   (write-out "modify ")
   (write-out-name relation)
   (write-out " to ")
   (write-out-name structure)
   (when unique (write-out " unique "))
   (when on
	 (write-out " on ")
	 (write-out-sort-list on))
   (when (OR fillfactor minpages maxpages indexfill maxindexfill) 
	 (write-out " where"))
   (when fillfactor
	 (setf others fillfactor)
	 (write-out " fillfactor=")
	 (write-out-item fillfactor))
   (when minpages
	 (when others (write-out ","))
	 (setf others minpages)
	 (write-out " minpages=")
	 (write-out-item minpages))
   (when maxpages
	 (when others (write-out ","))
	 (setf others maxpages)
	 (write-out " maxpages=")
	 (write-out-item maxpages))
   (when indexfill
	 (when others (write-out ","))
	 (setf others indexfill)
	 (write-out " indexfill=")
	 (write-out-item indexfill))
   (when maxindexfill
	 (when others (write-out ","))
	 (write-out " maxindexfill=")
	 (write-out-item maxindexfill))
   (end-command)))


;;;
;;;; PERMIT
;;;
;;; db-permit oplist var user [:cols (columnlist)]
;;;	[:at term] [:fromtime (time1 time2)] [:on (day1 day2)] :where qual
;;;
;;; Most args are printed as items, not expressions, so strings can be used
;;; for hh:mm formatted times.
;;; NOTE: the user comes before the columns, unlike quel syntax.
;;; User may also be specified by 'all.
;;;
(defun db-permit (oplist var user &key cols at from on where)
  (record-time
   (state-check)
   (write-out "define permit ")
   (write-out-sequence oplist 'write-out-name)
   (write-out " on ")
   (write-out-name var)
   (when  cols (write-out-list cols 'write-out-name))
   (write-out " to ")
   (write-out-name user)
   (when at
	 (write-out " at ")
	 (write-out-item at))
   (when from
	 (write-out " from ")
	 (write-out-item (first from))
	 (write-out " to ")
	 (write-out-item (second from)))
   (when on
	 (write-out " on ")
	 (write-out-item (first on))
	 (write-out " to ")
	 (write-out-item (second on)))
   (when where (write-out-qual where))))

;;;
;;;; PRINT
;;;
(defun db-print (&rest relations)
  (record-time
   (state-check)
   (write-out "print ")
   (write-out-sequence relations 'write-out-name)
   (end-command)))

;;;
;;;; RANGE
;;;
;;; This function also gloms the range variable to an internal db
;;; so that field types can be inferred, when necessary.
;;;
(defun db-range (variable relation)
  (record-time
   (state-check)
   (record-range-var variable relation)
   (write-out "range of ")
   (write-out-name variable)
   (write-out "=")
   (write-out-name relation)
   (end-command)))
;;;
;;;; RELOCATE
;;;
(defun db-relocate (table location)
  (record-time
   (state-check)
   (write-out "relocate ")
   (write-out-name table)
   (write-out " to ")
   (write-out-name location)
   (end-command)))

;;;
;;;; REPLACE
;;;
(defun db-replace (variable targetlist &key where)
  (record-time
   (state-check)
   (write-out "replace ")
   (write-out-name variable)
   (write-out-target-list targetlist)
   (write-out-qual where)
   (end-command)))

;;;
;;;; RETRIEVE
;;;
;;; If the :into argument isn't given, then you get a list, each
;;; element of which is a list representing a tuple.  The element of
;;; a tuple list are the returned values in the order dictated by the
;;; target list.  You gave the target list, so you should know what's
;;; being handed back to you.
;;;
(defun db-retrieve (targetlist &key where unique sort into) 
  (record-time
   (state-check)
   (cond (into
	  (write-out "retrieve ")
	  (write-out-location into)
	  (write-out-target-list targetlist)
	  (write-out-qual where)
	  (write-out-sort sort)
	  (end-command))
	 (t
	  (send-query targetlist unique where sort)
	  (retrieve-result-from-ingres targetlist)))))

;;;
;;;; DB-MAP-RETRIEVE
;;;
;;; The retrieve is executed, and each tuple returned is made into a list.
;;; The function is applied to each such list.  If the function returns
;;; nil, then the retrieve loop is terminated.
;;; No results are kept.
;;;
(defun db-map-retrieve (function targetlist &key where sort unique)
  (record-time
   (state-check)
   (send-query targetlist unique where sort)
   (map-retrieve function targetlist)
   nil))

;;;
;;;; DB-MAPL-RETRIEVE
;;;
;;; Like db-map-retrieve, except that the return value of the function
;;; is collected in a list.  There is no way to break the retrieve loop.
;;;
(defun db-mapl-retrieve (function targetlist &key where sort unique)
  (record-time
   (state-check)
   (send-query targetlist unique where sort)
   (mapl-retrieve function targetlist)))

;;;
;;;; DB-MAPN-RETRIEVE
;;;
;;; Like db-map-retrieve, except that the return values of the function
;;; are nconc'd together.  There is no way to break the retrieve loop.
;;;
(defun db-mapn-retrieve (function targetlist &key where sort unique)
  (record-time
   (state-check)
   (send-query targetlist unique where sort)
   (mapn-retrieve function targetlist)))

;;;
;;;; SAVE
;;;
(defun db-save (relation month day year)
  (record-time
   (state-check)
   (write-out "save ")
   (write-out-name relation)
   (write-out " until ")
   (write-out-item month)
   (write-out " ")
   (write-out-item day)
   (write-out " ")
   (write-out-item year)
   (end-command)))

;;;
;;;; SAVEPOINT
;;;
(defun db-savepoint (name)
  (record-time
   (state-check)
   (write-out "savepoint ")
   (write-out-item name)
   (end-command)))

;;;
;;;; SET commands
;;;

;;;
;;;; SET JOURNALING
;;;
(defun db-set-journaling (&key on)
  (record-time
   (state-check)
   (write-out "set journaling")
   (when on
     (write-out " on ")
     (write-out-item on))
   (end-command)))

;;;
;;;; SET NOJOURNALING
;;;
(defun db-set-nojournaling (&key on)
  (record-time
   (state-check)
   (write-out "set nojournaling")
   (when on
     (write-out " on ")
     (write-out-name on))
   (end-command)))

;;;
;;;; SET JOINOP
;;;
(defun db-set-joinop ()
  (record-time
   (state-check)
   (write-out "set joinop")
   (end-command)))

;;;
;;;; SET NOJOINOP
;;;
(defun db-set-nojoinop ()
  (record-time
   (state-check)
   (write-out "set nojoinop")
   (end-command)))

;;;
;;;; SET RESULTSTRUCTURE
;;;
(defun db-set-ret-into (result-structure)
  (record-time
   (state-check)
   (write-out "set ret_into")
   (write-out-string result-structure)
   (end-command)))

;;;
;;;; SET DEADLOCK
;;;
(defun db-set-deadlock ()
  (record-time
   (state-check)
   (write-out "set deadlock")
   (end-command)))
   
;;;
;;;; SET NODEADLOCK
;;;
(defun db-set-nodeadlock ()
  (record-time
   (state-check)
   (write-out "set nodeadlock")
   (end-command)))

;;;
;;;; SET LOCKMODE SESSION
;;;
(defun db-set-lockmode (&key on level readlock maxlocks timeout
				&aux other-settings)
  (record-time
   (state-check)
   (write-out "set lockmode ")
   (when on 
       (write-out "on ") 
       (write-out-item on))
   (when (NOT on) (write-out "session"))
   (write-out " where")
   (when level
	 (setf other-settings level)
	 (write-out " level=")
	 (write-out-item level))
   (when readlock
	 (if other-settings (write-out ",") (setf other-settings readlock))
	 (write-out " readlock=")
	 (write-out-item readlock))
   (when maxlocks
	 (if other-settings (write-out ",") (setf other-settings maxlock))
	 (write-out " maxlocks=")
	 (write-out-item maxlocks))
   (when timeout
	 (if other-settings (write-out ","))
	 (write-out " timeout=")
	 (write-out-item timeout))
   (end-command)))

;;;
;;;; SET QBUFSIZE
;;;
(defun db-set-qbufsize (newsize)
  (record-time
   (state-check)
   (write-out "set qbufsize ")
   (write-out-number newsize)
   (end-command)))

;;;
;;;; SET QUERYSIZE
;;;
(defun db-set-query-size (newsize)
  (record-time
   (state-check)
   (write-out "set query_size ")
   (write-out-number newsize)
   (end-command)))

;;;
;;;; SET CACHELOCAL
;;;
(defun db-set-cache-local (&key size readahead)
  (record-time
   (state-check)
   (write-out "set cache ")
   (write-out "local where")
   (when size
	 (write-out " size=")
	 (write-out-item size))
   (when readahead
	 (if size (write-out ","))
	 (write-out " readahead=")
	 (write-out-item readahead))
   (end-command)))

;;;
;;;; VIEW
;;;
(defun db-view (name target-list &key where)
  (record-time
   (state-check)
   (write-out "define view ")
   (write-out-item name)
   (write-out-target-list target-list)
   (write-out-qual where)
   (end-command)))
