;;; ---------------------------------------------------------------------------
;;;	Bitmap-Gadget
;;; ---------------------------------------------------------------------------

(defclass bitmap-gadget (gadget) ;; bitmap-gadget inherits from gadget class
  ((dimensions			;; list of <bits-wide> <bits-high>
    :type list
    :initform nil
    :initarg :dimensions	;; (user-specifiable)
    :reader dimensions) 
   (bit-size			;; size of each square on the screen
    :type integer
    :initform 1)
   (gc-dotted			;; graphic-context for drawing grid
    :type vector
    :initform nil
    :reader gc-dotted)
   (gc-spec			;; graphic-context specifications
    :initform '((gc-res "default")
		(gc-dotted (:paint "gray50"))))))

(defmacro pos-intp (val)
  `(and (integerp ,val) (plusp ,val)))

(defmethod (setf value) (val (self bitmap-gadget))
  (if (not (arrayp val))
      (warn "bitmap-gadget.setf.value:  bad array:  ~S~%" val)
      (progn 
       (setf (slot-value self 'value) val
	     (slot-value self 'dimensions) (array-dimensions val))
       (resize-window-handler self)
       (repaint self))))

(defmethod (setf dimensions) (val (self bitmap-gadget)) 
  (if (not (and (pos-intp (car val)) (pos-intp (cadr val))))
      (warn "bitmap-gadget.setf.dimensions:  invalid dims:  ~S~%" val)
      (setf (value self)
	    (make-array val :initial-element 0 :element-type 'bit))))

(defmethod new-instance ((self bitmap-gadget) 
			 &key 
			 value 
			 dimensions
			 &allow-other-keys)
  (call-next-method)
  (if value
      (setf (value self) value)
      (if dimensions
	  (setf (dimensions self) dimensions))))

(defmethod do-repaint ((self bitmap-gadget) &aux dims bit-size bit-array gc)
  (setq dims (dimensions self)
	bit-size (slot-value self 'bit-size)
	bit-array (value self)
	gc (gc-res self))
  (when bit-array
	(dotimes (x (car dims))
		 (dotimes (y (cadr dims))
			  (draw-bit self bit-array bit-size gc x y)))
	(draw-grid self)))

(defmethod resize-window-handler ((self bitmap-gadget) &aux dims)
  (if (setq dims (dimensions self))
      (setf (slot-value self 'bit-size)
	    (min (truncate (width self) (car dims))
		 (truncate (height self) (cadr dims))))))

(defun draw-bit (self bit-array bit-size gc i j &aux w x y)
  (setq x (1+ (* i bit-size)) 
	y (1+ (* j bit-size))
	w (- bit-size 1))
  (if (zerop (aref bit-array i j))
      (clear-region self x y w w)
      (xlib:draw-rectangle (res self) gc x y w w t)))

(defun draw-grid (self &aux gc res dims bit-size w h)
  (setq res (res self)
	gc (gc-dotted self)
	dims (dimensions self)
	bit-size (slot-value self 'bit-size))
  (setq w (* bit-size (car dims))
	h (* bit-size (cadr dims)))
  (do ((x bit-size (+ x bit-size)))
      ((> x w))
      (xlib:draw-line res gc x 0 x h))
  (do ((y bit-size (+ y bit-size)))
      ((> y h))
      (xlib:draw-line res gc 0 y w y)))

;;; ---------------------------------------------------------------------------
;;;	Bitmap-Editor (widget)
;;; ---------------------------------------------------------------------------

(defclass bitmap-editor (widget bitmap-gadget)
  ((event-mask :initform '(:exposure :button-press))))

(defun make-bitmap-editor (&rest args)
  (apply #'make-instance 'bitmap-editor :allow-other-keys t args))

(defhandler toggle-bit ((self bitmap-editor) &key x y &allow-other-keys 
			&aux bit-size bit-array dims
			&default :button-press)
  (setq bit-array (value self)
	bit-size (slot-value self 'bit-size)
	dims (dimensions self))
  (setq x (truncate x bit-size)
	y (truncate y bit-size))
  (when (and (<= x (car dims)) (<= y (cadr dims)))
	(setf (aref bit-array x y) (- (lognot (- (aref bit-array x y)))))
	(draw-bit self bit-array bit-size (gc-res self) x y)))
