#ifndef lint
static char sccsid[] = "@(#)tif_print.c	1.22 5/15/90";
#endif

/*
 * Copyright (c) 1988, 1990 by Sam Leffler.
 * All rights reserved.
 *
 * This file is provided for unrestricted use provided that this
 * legend is included on all tape media and as a part of the
 * software program in whole or part.  Users may copy, modify or
 * distribute this file at will.
 */

/*
 * TIFF Library.
 *
 * Directory Printing Support
 */
#include <stdio.h>
#include "file_formats/tiffio.h"

#define	FIELD(tif,f)	TIFFFieldSet(tif, CAT(FIELD_,f))

static char *ResponseUnitNames[] = {
	"#0",
	"10ths",
	"100ths",
	"1,000ths",
	"10,000ths",
	"100,000ths",
};
static	float ResponseUnit[] = { 1., .1, .01, .001, .0001, .00001 };
#define	MAXRESPONSEUNIT \
    (sizeof (ResponseUnitNames) / sizeof (ResponseUnitNames[0]))

/*
 * Print the contents of the current directory
 * to the specified stdio file stream.
 */
void
TIFFPrintDirectory(tif, fd, showstrips, showresponsecurve, showcolormap)
	TIFF *tif;
	FILE *fd;
	int showstrips, showresponsecurve, showcolormap;
{
	register TIFFDirectory *td;
	char *sep;
	int i;
	long n;
	float unit;

	fprintf(fd, "TIFF Directory at offset 0x%x\n", tif->tif_diroff);
	td = &tif->tif_dir;
	if (((tif)->tif_dir.td_fieldsset[FIELD_SUBFILETYPE/32] &
	    (1L<<(FIELD_SUBFILETYPE&0x1f)))) {
		fprintf(fd, "  Subfile Type:");
		sep = " ";
		if (td->td_subfiletype & FILETYPE_REDUCEDIMAGE) {
			fprintf(fd, "%sreduced-resolution image", sep);
			sep = "/";
		}
		if (td->td_subfiletype & FILETYPE_PAGE) {
			fprintf(fd, "%smulti-page document", sep);
			sep = "/";
		}
		if (td->td_subfiletype & FILETYPE_MASK) {
			fprintf(fd, "%stransparency mask", sep);
			sep = "/";
		}
		fprintf(fd, " (%u = 0x%x)\n",
		    td->td_subfiletype, td->td_subfiletype);
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_IMAGEDIMENSIONS/32] &
	    (1L<<(FIELD_IMAGEDIMENSIONS&0x1f))))
		fprintf(fd, "  Image Width: %u Image Length: %u\n",
		    td->td_imagewidth, td->td_imagelength);
	if (((tif)->tif_dir.td_fieldsset[FIELD_RESOLUTION/32] &
	    (1L<<(FIELD_RESOLUTION&0x1f)))) {
		fprintf(fd, "  Resolution: %g, %g",
		    td->td_xresolution, td->td_yresolution);
		if (((tif)->tif_dir.td_fieldsset[FIELD_RESOLUTIONUNIT/32] &
		    (1L<<(FIELD_RESOLUTIONUNIT&0x1f)))) {
			switch (td->td_resolutionunit) {
			case RESUNIT_NONE:
				fprintf(fd, " (unitless)");
				break;
			case RESUNIT_INCH:
				fprintf(fd, " pixels/inch");
				break;
			case RESUNIT_CENTIMETER:
				fprintf(fd, " pixels/cm");
				break;
			default:
				fprintf(fd, " (unit %u = 0x%x)",
				    td->td_resolutionunit,
				    td->td_resolutionunit);
				break;
			}
		}
		fprintf(fd, "\n");
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_POSITION/32] &
	    (1L<<(FIELD_POSITION&0x1f))))
		fprintf(fd, "  Position: %g, %g\n",
		    td->td_xposition, td->td_yposition);
	if (((tif)->tif_dir.td_fieldsset[FIELD_BITSPERSAMPLE/32] &
	    (1L<<(FIELD_BITSPERSAMPLE&0x1f))))
		fprintf(fd, "  Bits/Sample: %u\n", td->td_bitspersample);
	if (((tif)->tif_dir.td_fieldsset[FIELD_COMPRESSION/32] &
	    (1L<<(FIELD_COMPRESSION&0x1f)))) {
		fprintf(fd, "  Compression Scheme: ");
		switch (td->td_compression) {
		case COMPRESSION_NONE:
			fprintf(fd, "none\n");
			break;
		case COMPRESSION_CCITTRLE:
			fprintf(fd, "CCITT modified Huffman encoding\n");
			break;
		case COMPRESSION_CCITTFAX3:
			fprintf(fd, "CCITT Group 3 facsimile encoding\n");
			break;
		case COMPRESSION_CCITTFAX4:
			fprintf(fd, "CCITT Group 4 facsimile encoding\n");
			break;
		case COMPRESSION_CCITTRLEW:
			fprintf(fd, "CCITT modified Huffman encoding %s\n",
			    "w/ word alignment");
			break;
		case COMPRESSION_PACKBITS:
			fprintf(fd, "Macintosh PackBits encoding\n");
			break;
		case COMPRESSION_THUNDERSCAN:
			fprintf(fd, "ThunderScan 4-bit encoding\n");
			break;
		case COMPRESSION_LZW:
			fprintf(fd, "Lempel-Ziv & Welch encoding\n");
			break;
		case COMPRESSION_PICIO:
			fprintf(fd, "Pixar picio encoding\n");
			break;
		case COMPRESSION_NEXT:
			fprintf(fd, "NeXT 2-bit encoding\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_compression, td->td_compression);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_PHOTOMETRIC/32] &
	    (1L<<(FIELD_PHOTOMETRIC&0x1f)))) {
		fprintf(fd, "  Photometric Interpretation: ");
		switch (td->td_photometric) {
		case PHOTOMETRIC_MINISWHITE:
			fprintf(fd, "min-is-white\n");
			break;
		case PHOTOMETRIC_MINISBLACK:
			fprintf(fd, "min-is-black\n");
			break;
		case PHOTOMETRIC_RGB:
			fprintf(fd, "RGB color\n");
			break;
		case PHOTOMETRIC_PALETTE:
			fprintf(fd, "palette color (RGB from colormap)\n");
			break;
		case PHOTOMETRIC_MASK:
			fprintf(fd, "transparency mask\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_photometric, td->td_photometric);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_MATTEING/32] &
	    (1L<<(FIELD_MATTEING&0x1f))))
		fprintf(fd, "  Matteing: %s\n", td->td_matteing ?
		    "pre-multiplied with alpha channel" : "none");
	if (((tif)->tif_dir.td_fieldsset[FIELD_THRESHHOLDING/32] &
	    (1L<<(FIELD_THRESHHOLDING&0x1f)))) {
		fprintf(fd, "  Thresholding: ");
		switch (td->td_threshholding) {
		case THRESHHOLD_BILEVEL:
			fprintf(fd, "bilevel art scan\n");
			break;
		case THRESHHOLD_HALFTONE:
			fprintf(fd, "halftone or dithered scan\n");
			break;
		case THRESHHOLD_ERRORDIFFUSE:
			fprintf(fd, "error diffused\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_threshholding, td->td_threshholding);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_FILLORDER/32] &
	    (1L<<(FIELD_FILLORDER&0x1f)))) {
		fprintf(fd, "  FillOrder: ");
		switch (td->td_fillorder) {
		case FILLORDER_MSB2LSB:
			fprintf(fd, "msb-to-lsb\n");
			break;
		case FILLORDER_LSB2MSB:
			fprintf(fd, "lsb-to-msb\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_fillorder, td->td_fillorder);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_PREDICTOR/32] &
	    (1L<<(FIELD_PREDICTOR&0x1f)))) {
		fprintf(fd, "  Predictor: ");
		switch (td->td_predictor) {
		case 1:
			fprintf(fd, "none\n");
			break;
		case 2:
			fprintf(fd, "horizontal differencing\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_predictor, td->td_predictor);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_ARTIST/32] &
	    (1L<<(FIELD_ARTIST&0x1f))))
		fprintf(fd, "  Artist: \"%s\"\n", td->td_artist);
	if (((tif)->tif_dir.td_fieldsset[FIELD_DATETIME/32] &
	    (1L<<(FIELD_DATETIME&0x1f))))
		fprintf(fd, "  Date & Time: \"%s\"\n", td->td_datetime);
	if (((tif)->tif_dir.td_fieldsset[FIELD_HOSTCOMPUTER/32] &
	    (1L<<(FIELD_HOSTCOMPUTER&0x1f))))
		fprintf(fd, "  Host Computer: \"%s\"\n", td->td_hostcomputer);
	if (((tif)->tif_dir.td_fieldsset[FIELD_SOFTWARE/32] &
	    (1L<<(FIELD_SOFTWARE&0x1f))))
		fprintf(fd, "  Software: \"%s\"\n", td->td_software);
	if (((tif)->tif_dir.td_fieldsset[FIELD_DOCUMENTNAME/32] &
	    (1L<<(FIELD_DOCUMENTNAME&0x1f))))
		fprintf(fd, "  Document Name: \"%s\"\n", td->td_documentname);
	if (((tif)->tif_dir.td_fieldsset[FIELD_IMAGEDESCRIPTION/32] &
	    (1L<<(FIELD_IMAGEDESCRIPTION&0x1f))))
		fprintf(fd, "  Image Description: \"%s\"\n",
		    td->td_imagedescription);
	if (((tif)->tif_dir.td_fieldsset[FIELD_MAKE/32] &
	    (1L<<(FIELD_MAKE&0x1f))))
		fprintf(fd, "  Make: \"%s\"\n", td->td_make);
	if (((tif)->tif_dir.td_fieldsset[FIELD_MODEL/32] &
	    (1L<<(FIELD_MODEL&0x1f))))
		fprintf(fd, "  Model: \"%s\"\n", td->td_model);
	if (((tif)->tif_dir.td_fieldsset[FIELD_ORIENTATION/32] &
	    (1L<<(FIELD_ORIENTATION&0x1f)))) {
		fprintf(fd, "  Orientation: ");
		switch (td->td_orientation) {
		case ORIENTATION_TOPLEFT:
			fprintf(fd, "row 0 top, col 0 lhs\n");
			break;
		case ORIENTATION_TOPRIGHT:
			fprintf(fd, "row 0 top, col 0 rhs\n");
			break;
		case ORIENTATION_BOTRIGHT:
			fprintf(fd, "row 0 bottom, col 0 rhs\n");
			break;
		case ORIENTATION_BOTLEFT:
			fprintf(fd, "row 0 bottom, col 0 lhs\n");
			break;
		case ORIENTATION_LEFTTOP:
			fprintf(fd, "row 0 lhs, col 0 top\n");
			break;
		case ORIENTATION_RIGHTTOP:
			fprintf(fd, "row 0 rhs, col 0 top\n");
			break;
		case ORIENTATION_RIGHTBOT:
			fprintf(fd, "row 0 rhs, col 0 bottom\n");
			break;
		case ORIENTATION_LEFTBOT:
			fprintf(fd, "row 0 lhs, col 0 bottom\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_orientation, td->td_orientation);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_SAMPLESPERPIXEL/32] &
	    (1L<<(FIELD_SAMPLESPERPIXEL&0x1f))))
		fprintf(fd, "  Samples/Pixel: %u\n", td->td_samplesperpixel);
	if (((tif)->tif_dir.td_fieldsset[FIELD_ROWSPERSTRIP/32] &
	    (1L<<(FIELD_ROWSPERSTRIP&0x1f)))) {
		fprintf(fd, "  Rows/Strip: ");
		if (td->td_rowsperstrip == 0xffffffffL)
			fprintf(fd, "(infinite)\n");
		else
			fprintf(fd, "%u\n", td->td_rowsperstrip);
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_MINSAMPLEVALUE/32] &
	    (1L<<(FIELD_MINSAMPLEVALUE&0x1f))))
		fprintf(fd, "  Min Sample Value: %u\n", td->td_minsamplevalue);
	if (((tif)->tif_dir.td_fieldsset[FIELD_MAXSAMPLEVALUE/32] &
	    (1L<<(FIELD_MAXSAMPLEVALUE&0x1f))))
		fprintf(fd, "  Max Sample Value: %u\n", td->td_maxsamplevalue);
	if (((tif)->tif_dir.td_fieldsset[FIELD_PLANARCONFIG/32] &
	    (1L<<(FIELD_PLANARCONFIG&0x1f)))) {
		fprintf(fd, "  Planar Configuration: ");
		switch (td->td_planarconfig) {
		case PLANARCONFIG_CONTIG:
			fprintf(fd, "single image plane\n");
			break;
		case PLANARCONFIG_SEPARATE:
			fprintf(fd, "separate image planes\n");
			break;
		default:
			fprintf(fd, "%u (0x%x)\n",
			    td->td_planarconfig, td->td_planarconfig);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_PAGENAME/32] &
	    (1L<<(FIELD_PAGENAME&0x1f))))
		fprintf(fd, "  Page Name: \"%s\"\n", td->td_pagename);
	if (((tif)->tif_dir.td_fieldsset[FIELD_GRAYRESPONSEUNIT/32] &
	    (1L<<(FIELD_GRAYRESPONSEUNIT&0x1f)))) {
		fprintf(fd, "  Gray Response Unit: ");
		if (td->td_grayresponseunit < MAXRESPONSEUNIT)
			fprintf(fd, "%s\n",
			    ResponseUnitNames[td->td_grayresponseunit]);
		else
			fprintf(fd, "%u (0x%x)\n",
			    td->td_grayresponseunit, td->td_grayresponseunit);
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_GRAYRESPONSECURVE/32] &
	    (1L<<(FIELD_GRAYRESPONSECURVE&0x1f)))) {
		fprintf(fd, "  Gray Response Curve: ");
		if (showresponsecurve) {
			fprintf(fd, "\n");
			unit = ResponseUnit[td->td_grayresponseunit];
			n = 1L<<td->td_bitspersample;
			for (i = 0; i < n; i++)
				fprintf(fd, "    %2d: %g (%u)\n",
				    i,
				    td->td_grayresponsecurve[i] * unit,
				    td->td_grayresponsecurve[i]);
		} else
			fprintf(fd, "(present)\n");
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_GROUP3OPTIONS/32] &
	    (1L<<(FIELD_GROUP3OPTIONS&0x1f)))) {
		fprintf(fd, "  Group 3 Options:");
		sep = " ";
		if (td->td_group3options & GROUP3OPT_2DENCODING)
			fprintf(fd, "%s2-d encoding", sep), sep = "+";
		if (td->td_group3options & GROUP3OPT_FILLBITS)
			fprintf(fd, "%sEOL padding", sep), sep = "+";
		if (td->td_group3options & GROUP3OPT_UNCOMPRESSED)
			fprintf(fd, "%sno compression", sep), sep = "+";
		fprintf(fd, " (%u = 0x%x)\n",
		    td->td_group3options, td->td_group3options);
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_CLEANFAXDATA/32] &
	    (1L<<(FIELD_CLEANFAXDATA&0x1f)))) {
		fprintf(fd, "  Fax Data: ");
		switch (td->td_cleanfaxdata) {
		case CLEANFAXDATA_CLEAN:
			fprintf(fd, "clean\n");
			break;
		case CLEANFAXDATA_REGENERATED:
			fprintf(fd, "receiver regenerated\n");
			break;
		case CLEANFAXDATA_UNCLEAN:
			fprintf(fd, "uncorrected errors\n");
			break;
		default:
			fprintf(fd, "(%u = 0x%x)\n",
			    td->td_cleanfaxdata, td->td_cleanfaxdata);
			break;
		}
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_BADFAXLINES/32] &
	    (1L<<(FIELD_BADFAXLINES&0x1f))))
		fprintf(fd, "  Bad Fax Lines: %u\n", td->td_badfaxlines);
	if (((tif)->tif_dir.td_fieldsset[FIELD_BADFAXRUN/32] &
	    (1L<<(FIELD_BADFAXRUN&0x1f))))
		fprintf(fd, "  Consecutive Bad Fax Lines: %u\n",
		    td->td_badfaxrun);
	if (((tif)->tif_dir.td_fieldsset[FIELD_GROUP4OPTIONS/32] &
	    (1L<<(FIELD_GROUP4OPTIONS&0x1f))))
		fprintf(fd, "  Group 4 Options: 0x%x\n", td->td_group4options);
	if (((tif)->tif_dir.td_fieldsset[FIELD_PAGENUMBER/32] &
	    (1L<<(FIELD_PAGENUMBER&0x1f))))
		fprintf(fd, "  Page Number: %u-%u\n",
		    td->td_pagenumber[0], td->td_pagenumber[1]);
	if (((tif)->tif_dir.td_fieldsset[FIELD_COLORRESPONSEUNIT/32] &
	    (1L<<(FIELD_COLORRESPONSEUNIT&0x1f)))) {
		fprintf(fd, "  Color Response Unit: ");
		if (td->td_colorresponseunit < MAXRESPONSEUNIT)
			fprintf(fd, "%s\n",
			    ResponseUnitNames[td->td_colorresponseunit]);
		else
			fprintf(fd, "%u (0x%x)\n",
			    td->td_colorresponseunit, td->td_colorresponseunit);
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_COLORMAP/32] &
	    (1L<<(FIELD_COLORMAP&0x1f)))) {
		fprintf(fd, "  Color Map: ");
		if (showcolormap) {
			fprintf(fd, "\n");
			n = 1L<<td->td_bitspersample;
			for (i = 0; i < n; i++)
				fprintf(fd, "   %5d: %5u %5u %5u\n",
				    i,
				    td->td_redcolormap[i],
				    td->td_greencolormap[i],
				    td->td_bluecolormap[i]);
		} else
			fprintf(fd, "(present)\n");
	}
	if (((tif)->tif_dir.td_fieldsset[FIELD_COLORRESPONSECURVE/32] &
	    (1L<<(FIELD_COLORRESPONSECURVE&0x1f)))) {
		fprintf(fd, "  Color Response Curve: ");
		if (showresponsecurve) {
			fprintf(fd, "\n");
			unit = ResponseUnit[td->td_colorresponseunit];
			n = 1L<<td->td_bitspersample;
			for (i = 0; i < n; i++)
				fprintf(fd, "    %2d: %6.4f %6.4f %6.4f\n",
				    i,
				    td->td_redresponsecurve[i] * unit,
				    td->td_greenresponsecurve[i] * unit,
				    td->td_blueresponsecurve[i] * unit);
		} else
			fprintf(fd, "(present)\n");
	}
	if (showstrips && ((tif)->tif_dir.td_fieldsset[FIELD_STRIPOFFSETS/32] &
	    (1L<<(FIELD_STRIPOFFSETS&0x1f)))) {
		fprintf(fd, "  %u Strips:\n", td->td_nstrips);
		for (i = 0; i < td->td_nstrips; i++)
			fprintf(fd, "    %3d: [%8u, %8u]\n",
			    i, td->td_stripoffset[i], td->td_stripbytecount[i]);
	}
}
