/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/*************************************************************
*
*    file:      grapher.c
*
*    contents:  Functions for control of interactive 
*               graphics display.
*/


#include "include.h"
#ifndef TRUE
#define  TRUE 1
#define  FALSE 0
#endif

static REAL dang = M_PI/30;  /* rotation increment */
static REAL zoomfactor  = 1.2;  /* scale factor */
static REAL low[MAXCOORD],high[MAXCOORD];  /* extreme coordiates */
static REAL mid[MAXCOORD];          /* midpoint of extremes */

/* matrices */
/* image display motion done via homogeneous coordinates */
static REAL **spinl, **tipup;  /* rotation increment matrices */
static REAL **spinr, **tipdown;  /* rotation increment matrices */
static REAL **clockwise, **counterclock;  /* rotation increment matrices */
static REAL **transleft,**transright; /* translation increment matrices */
static REAL **transup,**transdown; /* translation increment matrices */
static REAL **zoom, **shrink;  /* scaling matrices */


static int showflag;

void update_display() /* called whenever something changes */
{
  if (go_display_flag )
#ifdef NeXTapp
     NeXT_display();
#else
     display();
#endif
   if ( OOGL_flag ) UpdateOOGL();
}

void torshow()
{
  char line[100]; /* for reading user commands */

  if ( web.torus_flag && !web.torus_clip_flag && !web.torus_body_flag )
   {
     char response[100];
     prompt("Display raw cell, connected bodies or clipped cell? (0,1,2): ",
	response);
     switch ( response[0] )
       {
         case '0' : web.torus_body_flag = 0; 
                    web.torus_clip_flag = 0; 
                    break;
         case '1' : web.torus_body_flag = 1; 
                    web.torus_clip_flag = 0; 
                    break;
         case '2' : web.torus_body_flag = 0; 
                    web.torus_clip_flag = 1; 
                    break;
       }
   }

  else if ( web.symmetry_flag && !web.torus_clip_flag && !web.torus_body_flag )
   {
     char response[100];
     prompt("Display raw cell or connected bodies? (0,1): ",response);
     switch ( response[0] )
       {
         case '0' : web.torus_body_flag = 0; 
                    web.torus_clip_flag = 0; 
                    break;
         case '1' : web.torus_body_flag = 1; 
                    web.torus_clip_flag = 0; 
                    break;
       }
   }


  /* main loop */
  showflag = 1;
  innerflag = outerflag = 1;
  do
   { 
     breakflag = 0;
#ifdef NeXTapp
     if ( showflag ) NeXT_display();
#else
     if ( showflag ) display();
#endif
     showflag = 1; /* default to show next time around, unless option
                      below decides otherwise */
     prompt("Graph command: ",line);
    }
   while ( view_transform(line) );
}

int view_transform(string)
char *string;
{
  char *c;
  char response[100];

  for ( c = string ; *c ; c++ )
    { int reps = 1;  /* repetition count */
      double val;  /* for arbitrary rotations */
      int decflag = 0;  /* whether have real number for angle */
  
      val = atof(c);
      if ( val != 0.0 ) /* have digits */
        { 
          reps = atoi(c);
          while ( isdigit(*c) || (*c=='.') || (*c=='-'))
             { if ( *c == '.' ) decflag = 1;
               c++;
             }
         }
      if ( decflag ) reps = 1;
      else val = 6.0;  /* default angle */

      while ( reps-- > 0 )
       switch ( *c )
        {
         case 'u':
	  set_tipup(val*M_PI/180); 
          mat_mult(tipup,view,view,HOMDIM,HOMDIM,HOMDIM); break;  
         case 'd': 
	  set_tipdown(val*M_PI/180); 
          mat_mult(tipdown,view,view,HOMDIM,HOMDIM,HOMDIM); break;  
         case 'r': 
	  set_spinr(val*M_PI/180); 
          mat_mult(spinr,view,view,HOMDIM,HOMDIM,HOMDIM); break;  
         case 'l': 
	  set_spinl(val*M_PI/180); 
          mat_mult(spinl,view,view,HOMDIM,HOMDIM,HOMDIM); break;  
         case 'c': 
	  set_clockwise(val*M_PI/180); 
          mat_mult(clockwise,view,view,HOMDIM,HOMDIM,HOMDIM); 
          break;  
         case 'C': 
	  set_counterclockwise(val*M_PI/180); 
          mat_mult(counterclock,view,view,HOMDIM,HOMDIM,HOMDIM); 
          break;  
         case 'z': 
          mat_mult(zoom ,view,view,HOMDIM,HOMDIM,HOMDIM); break;  
         case 's': 
          mat_mult(shrink,view,view,HOMDIM,HOMDIM,HOMDIM); break;  

         /* MS-DOS arrow keys for translation */
         case 30: 
          mat_mult(transup,  view,view,HOMDIM,HOMDIM,HOMDIM);
          break;
         case 31: 
          mat_mult(transdown,view,view,HOMDIM,HOMDIM,HOMDIM);
          break;
         case 28: 
          mat_mult(transright,view,view,HOMDIM,HOMDIM,HOMDIM);
          break;
         case 29: 
          mat_mult(transleft,view,view,HOMDIM,HOMDIM,HOMDIM);
          break;

         case 0x1b : /* ANSI arrow keys for translation */
           if ( *(++c) != 0x5B )
             { sprintf(msg,"Unrecognized char: %02X  %c\n",*c,*c);
               outstring(msg);
               break;
             }
           switch ( *(++c) )
            {
               case 0x41: 
                mat_mult(transup,  view,view,HOMDIM,HOMDIM,HOMDIM);
                break;
               case 0x42: 
                mat_mult(transdown,view,view,HOMDIM,HOMDIM,HOMDIM);
                break;
               case 0x43: 
                mat_mult(transright,view,view,HOMDIM,HOMDIM,HOMDIM);
                break;
               case 0x44: 
                mat_mult(transleft,view,view,HOMDIM,HOMDIM,HOMDIM);
                break;
               default:
                   sprintf(msg,"Unrecognized char: %02X  %c\n",*c,*c); 
                   outstring(msg);
                   break; 
             }
           break;
         case 'R': resize(); reps = 0; break;
/*         case 'm': make_movie(); showflag = 0; break;
 */
         case 'x':
         case 'q': return 0;     
         case 't': 
              if ( !web.symmetry_flag ) break;
              prompt("Display raw cell, connected bodies or clipped cell? (0,1,2): ",response);
              switch ( response[0] )
                {
                  case '0' : web.torus_body_flag = 0; 
                             web.torus_clip_flag = 0; 
                             break;
                  case '1' : web.torus_body_flag = 1; 
                             web.torus_clip_flag = 0; 
                             break;
                  case '2' : web.torus_body_flag = 0; 
                             web.torus_clip_flag = 1; 
                             break;
                }
             reps = 0; break;
         case 'B': bdry_showflag = !bdry_showflag;
                   reps = 0; break;
         case 'v': ridge_color_flag = !ridge_color_flag; reps = 0; break;
         case 'w': no_wall_flag = !no_wall_flag; reps = 0; break;
         case 'b': box_flag = !box_flag; reps = 0; break;
	 case 'e': edgeshow_flag = !edgeshow_flag; break;
	 case 'E': triple_edgeshow_flag = !triple_edgeshow_flag; break;
         case '+': fillcolor++;
                   sprintf(msg,"fill %d\n",fillcolor);
                   outstring(msg); 
                   reps = 0; break;
         case '-': fillcolor--;
                   sprintf(msg,"fill %d\n",fillcolor);
                   outstring(msg); 
                   reps = 0; break;
         case 'H': web.hide_flag = !web.hide_flag; reps = 0; break;
         case '?': 
         case 'h':
                   graph_help();
                   showflag = 0;
                   reps = 0; break;
         default:  sprintf(msg,"Unrecognized letter: %04x  %c\n",*c,*c);
                   outstring(msg);
                   reps = 0; showflag = 0;
                   break;

        }
    }

  return 1;
}

void init_view()
{
  int i;

  /* set up before we know sdim */

  view = dmatrix(0,MAXCOORD,0,MAXCOORD);

  /* set up identity matrix */
  identity = dmatrix(0,MAXCOORD,0,MAXCOORD);
  for ( i = 0 ; i <= MAXCOORD ; i++ )
    identity[i][i] = 1.0;

  /* set rotation matrices */
  spinr = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(spinr,identity,MAXCOORD+1,MAXCOORD+1);
  spinr[0][0] = spinr[1][1] = cos(dang);
  spinr[0][1] = -(spinr[1][0] = sin(dang));

  spinl = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(spinl,identity,MAXCOORD+1,MAXCOORD+1);
  spinl[0][0] = spinl[1][1] = cos(dang);
  spinl[0][1] = -(spinl[1][0] = -sin(dang));

  tipup = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(tipup,identity,MAXCOORD+1,MAXCOORD+1);
  tipup[0][0] = tipup[2][2] = cos(dang);
  tipup[0][2] = -(tipup[2][0] = sin(dang));

  tipdown = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(tipdown,identity,MAXCOORD+1,MAXCOORD+1);
  tipdown[0][0] = tipdown[2][2] = cos(dang);
  tipdown[0][2] = -(tipdown[2][0] = -sin(dang));

  clockwise = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(clockwise,identity,MAXCOORD+1,MAXCOORD+1);
  clockwise[1][1] = clockwise[2][2] = cos(dang);
  clockwise[1][2] = -(clockwise[2][1] = -sin(dang));

  counterclock = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(counterclock,identity,MAXCOORD+1,MAXCOORD+1);
  counterclock[1][1] = counterclock[2][2] = cos(dang);
  counterclock[1][2] = -(counterclock[2][1] = sin(dang));

  /* set magnifying matrix */
  zoom = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(zoom,identity,MAXCOORD+1,MAXCOORD+1);
  zoom[0][0] = zoom[1][1] = zoom[2][2] = zoomfactor;

  /* set shrink matrix */
  shrink = dmatrix(0,MAXCOORD,0,MAXCOORD);
  matcopy(shrink,identity,MAXCOORD+1,MAXCOORD+1);
  shrink[0][0] = shrink[1][1] = shrink[2][2] = 1/zoomfactor;

  /* set translation matrices */
  transleft = dmatrix(0,MAXCOORD,0,MAXCOORD);
  transup = dmatrix(0,MAXCOORD,0,MAXCOORD);
  transright = dmatrix(0,MAXCOORD,0,MAXCOORD);  
  transdown = dmatrix(0,MAXCOORD,0,MAXCOORD);
}

void set_spinr(val)
double val;
{
  spinr[0][0] = spinr[1][1] = cos(val);
  spinr[0][1] = -(spinr[1][0] = sin(val));
}
void set_spinl(val)
double val;
{
  spinl[0][0] = spinl[1][1] = cos(val);
  spinl[0][1] = -(spinl[1][0] = -sin(val));
}
void set_tipup(val)
double val;
{
  tipup[0][0] = tipup[2][2] = cos(val);
  tipup[0][2] = -(tipup[2][0] = sin(val));
}
void set_tipdown(val)
double val;
{
  tipdown[0][0] = tipdown[2][2] = cos(val);
  tipdown[0][2] = -(tipdown[2][0] = -sin(val));
}
void set_clockwise(val)
double val;
{
  clockwise[1][1] = clockwise[2][2] = cos(val);
  clockwise[1][2] = -(clockwise[2][1] = -sin(val));
}
void set_counterclockwise(val)
double val;
{
  counterclock[1][1] = counterclock[2][2] = cos(val);
  counterclock[1][2] = -(counterclock[2][1] = sin(val));
}

void reset_view()
{
  matcopy(transleft,identity,HOMDIM,HOMDIM);
  matcopy(transdown,identity,HOMDIM,HOMDIM);
  matcopy(transright,identity,HOMDIM,HOMDIM);
  matcopy(transup,identity,HOMDIM,HOMDIM);
  
  if ( web.dimension >= 2 )
  {
    transright[1][HOMDIM-1] = .25;
    transleft[1][HOMDIM-1] = -.25;
    transup[2][HOMDIM-1] = .25;
    transdown[2][HOMDIM-1] = -.25;
  }
  else  /* show x-y plane */
  {
    transright[0][HOMDIM-1] = .25;
    transleft[0][HOMDIM-1] = -.25;
    transup[1][HOMDIM-1] = .25;
    transdown[1][HOMDIM-1] = -.25;
  }

  /* set total transformation matrix */
  matcopy(view,identity,HOMDIM,HOMDIM);
}

void resize()
{
  int i,j;
  vertex_id v_id;
  REAL *x;
  REAL size;


  /* if domain is torus, get torus fundamental cell in view */
  if ( web.torus_flag )
    {
      for ( i = 0 ; i < web.sdim ; i++ )  /* coordinate loop */
        {
          low[i] = high[i] = 0.0;
          for ( j = 0 ; j < web.sdim ; j++ ) /* axis loop */
            if ( web.torus_period[j][i] < 0.0 ) low[i] += web.torus_period[j][i];
            else high[i] += web.torus_period[j][i];
        }
     }
   else if ( web.symmetry_flag )
     { edge_id e_id;
       for ( i = 0 ; i < web.sdim ; i++ )  /* initialize */
         {
           low[i] = 1e10;
           high[i] = -1e10;
         }
       /* figure out how big window should be  */
       FOR_ALL_EDGES(e_id)
         { REAL s[MAXCOORD];
           get_edge_side(e_id,s);   /* unwrapped */
           x = get_coord(get_edge_tailv(e_id));
           for ( i = 0 ; i < web.sdim ; i++ )
             {
               if ( x[i] < low[i] ) low[i] = x[i];
               if ( x[i] > high[i] ) high[i] = x[i];
               if ( x[i]+s[i] < low[i] ) low[i] = x[i]+s[i];
               if ( x[i]+s[i] > high[i] ) high[i] = x[i]+s[i];
             }
         }
     }
   else
     {  
       for ( i = 0 ; i < web.sdim ; i++ )  /* initialize */
         {
           low[i] = 1e10;
           high[i] = -1e10;
         }
       /* figure out how big window should be  */
       
       FOR_ALL_VERTICES(v_id)
         { x = get_coord(v_id);
           for ( i = 0 ; i < web.sdim ; i++ )
             {
               if ( x[i] < low[i] ) low[i] = x[i];
               if ( x[i] > high[i] ) high[i] = x[i];
             }
         }
     }
   
  for ( i = 0 ; i < web.sdim ; i++ ) mid[i] = (low[i] + high[i])/2;
  if ( web.dimension == SOAPFILM )
   if ( (high[1] - low[1]) > (high[2] - low[2]) ) size = high[1] - low[1];
   else size = high[2] - low[2];
  else
   if ( (high[1] - low[1]) > (high[0] - low[0]) ) size = high[1] - low[1];
   else size = high[0] - low[0];
  
  /* transformation matrix will be set up to scale object into
     [-1,1]^3 cube */

  matcopy(view,identity,HOMDIM,HOMDIM);
  for ( i = 0 ; i < HOMDIM-1 ; i++ ) 
    { view[i][i] = 2/size;
      view[i][HOMDIM-1] = -mid[i]*2/size;
    }

  overall_size = size;  /* for anybody who wants to know how big */
}

