(*--------------------------------------------------------------------------*)
(*                 Tinv -- Inverse Central t Distribution                   *)
(*--------------------------------------------------------------------------*)

FUNCTION tinv( Alpha, Df: REAL ) : REAL;

(*--------------------------------------------------------------------------*)
(*                                                                          *)
(*       Function:  tinv                                                    *)
(*                                                                          *)
(*       Purpose:   Calculates central inverse t                            *)
(*                                                                          *)
(*       Calling Sequence:                                                  *)
(*                                                                          *)
(*            tval   := tinv( Alpha, Df : REAL ) : REAL;                    *)
(*                                                                          *)
(*                 Alpha  --- Probability value                             *)
(*                 Df     --- Degrees of freedom                            *)
(*                                                                          *)
(*                 tval   --- returned t-value corresponding to 'Alpha'.    *)
(*                                                                          *)
(*       Calls:                                                             *)
(*                                                                          *)
(*            BetaInv (Inverse Beta Distribution)                           *)
(*                                                                          *)
(*       Remarks:                                                           *)
(*                                                                          *)
(*            Any error results in '-1.0' as returned t value.              *)
(*                                                                          *)
(*--------------------------------------------------------------------------*)

CONST
   MaxIter = 100;
   Dprec   = 10;

VAR
   tin:   REAL;
   Iter:  INTEGER;
   Cprec: REAL;
   Ierr:  INTEGER;

BEGIN (* tinv *)

   tin   := -1.0;

   IF( Df > 0.0 ) THEN

      IF( ( Alpha >= 0.0 ) AND ( Alpha <= 1.0 ) ) THEN

         BEGIN

            tin := BetaInv( Alpha, 0.5, Df / 2.0, MaxIter, Dprec,
                            Iter, Cprec, Ierr );

            IF( ( tin >= 0.0 ) AND ( tin < 1.0 ) AND ( Ierr = 0 ) ) THEN
                  tin  := SQRT( tin * Df / ( 1.0 - tin ) );

         END;

END   (* tinv *);
