#include <stdio.h>

#if defined(__STDC__) || defined(__TURBOC__)
#include <stdlib.h>
#else
#include <sys/types.h>
#endif

#ifdef __TURBOC__
#include <io.h>
#else
#include <unistd.h>
#endif

#include <fcntl.h>
#include <sys/stat.h>
#include <string.h>
#ifdef __TURBOC__
#include <alloc.h>
#else
#include <malloc.h>
#endif

#ifndef O_BINARY
#define O_BINARY 0
#endif

#include "spconv.h"

struct sna_s sna;
struct vga_s vga;
struct z80_s z80;
struct prg_s prg;
struct ach_s ach;
struct kgb_s kgb;

unsigned char image[IMSIZE];
unsigned int z80_size;

int	intype;
int	outtype;

/* Path names are limited to 80 characters under MSDOS so this sould be */
/* enough... If you are using UNIX than you may have to make this array */
/* larger.                                                              */
char	my_directory[120];

/* this fd is used all over the place, defined only once here */
int fd;

main(int argc,char *argv[])
{
	char *p;
	struct stat status;
	char * fromstring;
	char * tostring;
	char * outfile;

	if(argc != 3)
	{
		fprintf(stderr,"SPCONV version 1.05 - %s\n\n",__DATE__);
		fprintf(stderr,"Usage: spconv <source> <target>\n\n");
		fprintf(stderr,"Source must be a valid .SNA, .SP, .Z80, .PRG, .ACH, .ZX or RAW file.\n");
		fprintf(stderr,"Target must be a .SNA, .SP, .Z80, .PRG, .ACH or .ZX file.\n\n");
		fprintf(stderr,"If the second parameter contains only a suffix, the prefix\n");
		fprintf(stderr,"of the input file will be used (i.e. 'spconv file.sna .z80')\n\n");
		fprintf(stderr,"Output .SP files are in the new format, .Z80 files are compressed.\n\n");
		fprintf(stderr,"If <source> and <target> are .SP files, convertion from old\n");
		fprintf(stderr,"to new format or from new to old format will be performed.\n");
		fprintf(stderr,"If <source> and <target> are of the same type an error message\n");
		fprintf(stderr,"will be generated (unless they are both .SP files)\n");
		fprintf(stderr,"\n\nPublic Domain, H. de Groot 1993\n\n");

		return 1;
	}

	if((strchr(argv[1],'*')!=NULL) || (strrchr(argv[1],'?')!=NULL) ||
	   (strchr(argv[2],'*')!=NULL) || (strrchr(argv[2],'?')!=NULL))
	{
		fprintf(stderr,"This program can't handle wildcards, sorry!\n");
		return 1;
	}

	strcpy(my_directory,argv[0]);
	if(strrchr(my_directory,'\\')!=NULL)
	{
		*strrchr(my_directory,'\\')='\0';
	}
	else if(strrchr(my_directory,'/')!=NULL)
	{
		*strrchr(my_directory,'/')='\0';
	}
	else if(strrchr(my_directory,':')!=NULL)
	{
		*strrchr(my_directory,':')='\0';
	}
	else
	{
		my_directory[0]='\\';
		my_directory[1]='\0';
	}

	if(stat(argv[1],&status)<0)
	{
		perror(argv[1]);
		return 1;
	}

	/* 
	 * recognize input type on filename:
	 *
	 *	.SNA	->	SNA file (used in JPP)
	 *	.SP	->	SP file (was VGASPEC)
	 *	.Z80	->	Z80 file
	 *	.PRG	->	PRG file
	 *	.ACH	->	ACH file (from archimedes emulator)
	 *	.ZX	->	KGB file (from KGB emulator)
	 *	other	->	if exact 48+header -> raw file
	 *	otherwise 	unknown
	 */

	intype=UNKNOWN;
	p=strrchr(argv[1],'.');
	if(p==NULL) p=argv[1]; /* not found, set at begin of string */
	if((strcmp(p,".SNA")==0) || (strcmp(p,".sna")==0))
	{
		fromstring=".SNA";
		intype=SNA;
	}
	if((strcmp(p,".Z80")==0) || (strcmp(p,".z80")==0)) 
	{
		fromstring=".Z80";
		intype=Z80;
	}
	if((strcmp(p,".SP")==0) || (strcmp(p,".sp")==0)) 
	{
		fromstring=".SP";
		intype=SP;
	}
	if((strcmp(p,".PRG")==0) || (strcmp(p,".prg")==0)) 
	{
		fromstring=".PRG";
		intype=PRG;
	}
	if((strcmp(p,".ACH")==0) || (strcmp(p,".ach")==0)) 
	{
		fromstring=".ACH";
		intype=ACH;
	}
	if((strcmp(p,".ZX")==0) || (strcmp(p,".zx")==0)) 
	{
		fromstring=".ZX";
		intype=KGB;
	}

	if(intype==UNKNOWN)
		if (status.st_size == (sizeof(h)+IMSIZE)) 
		{
			fromstring="RAW";
			intype=RAW;
		}

	/* 
	 * recognize output type on filename:
	 *
	 *	.SNA	->	SNA file
	 *	.SP	->	SP file (was VGASPEC)
	 *	.Z80	->	Z80 file
	 *	.PRG	->	PRG file
	 *	.ACH	->	ACH file (from archimedes emulator)
	 *	.ZX	->	KGB file (from KGB emulator)
	 *	otherwise 	unknown
	 */
	outtype=UNKNOWN;

	p=strrchr(argv[2],'.');
	if(p==NULL) p=argv[2]; /* not found, set at begin of string */
	if((strcmp(p,".SNA")==0) || (strcmp(p,".sna")==0)) 
	{
		tostring=".SNA";
		outtype=SNA;
	}
	if((strcmp(p,".Z80")==0) || (strcmp(p,".z80")==0)) 
	{
		tostring=".Z80";
		outtype=Z80;
	}
	if((strcmp(p,".SP")==0) || (strcmp(p,".sp")==0)) 
	{
		tostring=".SP";
		outtype=SP;
	}
	if((strcmp(p,".PRG")==0) || (strcmp(p,".prg")==0)) 
	{
		tostring=".PRG";
		outtype=PRG;
	}
	if((strcmp(p,".ACH")==0) || (strcmp(p,".ach")==0)) 
	{
		tostring=".ACH";
		outtype=ACH;
	}
	if((strcmp(p,".ZX")==0) || (strcmp(p,".zx")==0)) 
	{
		tostring=".ZX";
		outtype=KGB;
	}

	if(intype==UNKNOWN)
	{
		fprintf(stderr,"Unknown input file format. Must be a valid .SNA, .SP, .Z80, .PRG, .ACH or\n");
		fprintf(stderr,".ZX file, or a Raw file\n");
		return 3;
	}

	if(outtype==UNKNOWN)
	{
		fprintf(stderr,"Unknown output file format. Must be a .SNA, .SP, .Z80, .PRG, .ACH or .ZX file\n");
		return 4;
	}

	/* 
	   if argv[2] only contains the suffix then use the prefix of
	   argv[1];
	 */
	if(argv[2][0]=='.')
	{
		outfile=malloc(strlen(argv[1])+strlen(argv[2])+1);
		strcpy(outfile,argv[1]); /* copy prefix    */

		p=strrchr(outfile,'.');
		if(p!=NULL) *p='\0'; /* put end of string at position of '.' */

		strcat(outfile,argv[2]); /* append suffix  */
	}
	else
	{
		outfile=malloc(strlen(argv[2]));

		strcpy(outfile,argv[2]);
	}

	if(intype==outtype)
	{
		if(intype!=SP)
		{
		fprintf(stderr,"Input and output file format are the same. ");
		fprintf(stderr,"What you try to do\n");
		fprintf(stderr,"is handled much better by the MSDOS \"COPY\" ");
		fprintf(stderr,"command!\n");
		return 3;
		}
		else
		{
			if((status.st_size == (sizeof(vga)+IMSIZE)))
			{
				printf("Converting %s from new .SP format to old .SP format.\n",argv[1]);
				read_vgaspec(argv[1]);

				write_old_vgaspec(outfile);
				return 0;
			}
			else if((status.st_size == (sizeof(vga)+IMSIZE-6)))
			{
				read_vgaspec_header(argv[1]);
				if((vga.S=='S')&&(vga.P=='P'))
				{
fprintf(stderr,"Invalid input file format. This could be a new syle .SP file with\n");
fprintf(stderr,"an image of another length than 48Kb. This kind of .SP files cannot\n");
fprintf(stderr,"be converted. All other file formats (including the old .SP format)\n");
fprintf(stderr,"contain images of 48Kb length.\n");
					return 3;
				}

				printf("Converting %s from old .SP format to new .SP format.\n",argv[1]);
				read_old_vgaspec(argv[1]);
				vga.S='S';
				vga.P='P';
				vga.len_l=0x00;
				vga.len_h=0xC0;
				vga.start_l=0x00;
				vga.start_h=0x40;
				write_vgaspec(outfile);
				return 0;
			}
			else
			{
				read_vgaspec_header(argv[1]);
				if((vga.S=='S')&&(vga.P=='P'))
				{
fprintf(stderr,"Invalid input file format. This could be a new syle .SP file with\n");
fprintf(stderr,"an image of another length than 48Kb. This kind of .SP files cannot\n");
fprintf(stderr,"be converted. All other file formats (including the old .SP format)\n");
fprintf(stderr,"contain images of 48Kb length.\n");
					return 3;
				}
				else
				{
fprintf(stderr,"Unknown input file format. Must be a valid .SNA, .SP, .Z80 or .PRG file\n");
					return 3;
				}
			}
		}
	}

	printf("Converting %s from %s to %s\n",argv[1],fromstring,tostring);

	/*
	 * convert input_file to SNA
	 */
	if((intype==SNA) && (status.st_size == (sizeof(sna)+IMSIZE)))
	{
		read_sna(argv[1]);
	}
	else if ((intype==SP)&&((status.st_size == (sizeof(vga)+IMSIZE))))
	{
		read_vgaspec(argv[1]);
		
		vgaspec_to_sna();
	}
	else if ((intype==SP)&&((status.st_size == (sizeof(vga)+IMSIZE-6))))
	{
		read_old_vgaspec(argv[1]);
		
		vgaspec_to_sna();
	}
	else if (intype==RAW)
	{
		read_raw(argv[1]);

		raw_to_sna();
	}
	else if (intype==Z80)
	{
		read_z80(argv[1]);

		z80_to_sna();
	}
	else if (intype==PRG)
	{
		if(status.st_size != (sizeof(prg)+IMSIZE))
		{
			printf("Warning: the image part of %s is not exactly 48k!\n",argv[1]);
			printf("         Converting anyway, the converted file may not work\n");
		}
		read_prg(argv[1]);

		prg_to_sna();
	}
	else if((intype==ACH) && (status.st_size == (sizeof(ach)+16384L+IMSIZE)))
	{
		read_ach(argv[1]);

		ach_to_sna();
	}
	else if((intype==KGB) && (status.st_size == (132L+IMSIZE+sizeof(kgb))))
	{
		read_kgb(argv[1]);

		kgb_to_sna();
	}
	else
	{
		printf("Unrecognized input file type, can't convert\n");
		return 3;
	}

	/*
	 * convert internal SNA format to output file
	 */
	if(outtype==SNA)
	{
		write_sna(outfile);
	}
	else if (outtype==SP)
	{
		sna_to_vgaspec();
	
		write_vgaspec(outfile);
	}
	else if (outtype==Z80)
	{
		sna_to_z80();

		write_z80(outfile);
	}
	else if (outtype==PRG)
	{
		sna_to_prg(outfile);

		write_prg(outfile);
	}
	else if (outtype==ACH)
	{
		sna_to_ach();

		write_ach(outfile);
	}
	else if (outtype==KGB)
	{
		sna_to_kgb();

		write_kgb(outfile);
	}
	else
	{
		printf("Unrecognized output file type, can't convert\n");
		return 4;
	}
	return 0;
}

/* I/O FUNCTIONS */

/* GENERIC I/O - READ/WRITE RAM IMAGE */

void read_image(char * s)
{
	if(read(fd,image,(unsigned int)IMSIZE)==-1)
	{
		perror(s);
		exit(1);
	}
}

void write_image(char * s)
{
	if(write(fd,image,(unsigned int)IMSIZE)==-1)
	{
		perror(s);
		exit(2);
	}
}

/* SPECIFIC I/O - READ/WRITE .SNA IMAGE */

void read_sna(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&sna,sizeof(sna))==-1)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_sna(char * s)
{
	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}
		
	if(write(fd,&sna,sizeof(sna))==-1)
	{
		perror(s);
		exit(2);
	}
	write_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ NEW .SP HEADER */

void read_vgaspec_header(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&vga,sizeof(vga))==-1)
	{
		perror(s);
		exit(1);
	}
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE NEW .SP IMAGE */

void read_vgaspec(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&vga,sizeof(vga))==-1)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_vgaspec(char * s)
{
	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}

	if(write(fd,&vga,sizeof(vga))==-1)
	{
		perror(s);
		exit(2);
	}
	write_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE OLD .SP IMAGE */

void read_old_vgaspec(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,((char *)&vga)+6,sizeof(vga)-6)==-1)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_old_vgaspec(char * s)
{
	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}

	if(write(fd,((char *)&vga)+6,sizeof(vga)-6)==-1)
	{
		perror(s);
		exit(2);
	}
	write_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ RAW IMAGE */

void read_raw(char * s)
{
	int i;

	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&h,sizeof(h))==-1)
	{
		perror(s);
		exit(1);
	}
	for(i=0;i<9;i++)
	{
		if(h.in[i]!=expect[i])
		{
			fprintf(stderr,"Header of spectum image not ok, ");
			fprintf(stderr,"Spectrum image should be saved with:\n");
			fprintf(stderr,"SAVE *\"b\"CODE 16384,49152");
			exit(1);
		}
	}
	read_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE .Z80 IMAGE */

void read_z80(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&z80,sizeof(z80))==-1)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_z80(char * s)
{
	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}
		
	if(write(fd,&z80,sizeof(z80))==-1)
	{
		perror(s);
		exit(2);
	}
	if(write(fd,image,z80_size)==-1)
	{
		perror(s);
		exit(2);
	}
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE .PRG IMAGE */

void read_prg(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&prg,sizeof(prg))==-1)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_prg(char * s)
{
	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}

	if(write(fd,&prg,sizeof(prg))==-1)
	{
		perror(s);
		exit(2);
	}
	write_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE .ACH IMAGE */

void read_ach(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}
	if(read(fd,&ach,sizeof(ach))==-1)
	{
		perror(s);
		exit(1);
	}
	/* lseek over the 16K ram area */
	if(lseek(fd,16384L,SEEK_CUR)==-1L)
	{
		perror(s);
		exit(1);
	}
	read_image(s);
	close(fd);
}

void write_ach(char * s)
{
	char buffer[1024];
	char *p;
	int  i;
	struct stat status;
	char * rom;

	/* clean buffer first */
	p=(char *) buffer;
	for(i=0; i < 1024; i++)
		p[i]='\0';

	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}

	if(write(fd,&ach,sizeof(ach))==-1)
	{
		perror(s);
		exit(2);
	}

	strcat(my_directory,"\\spectrum.rom");

	rom=NULL;

	if(stat("spectrum.rom",&status)>=0)
	{
		rom="spectrum.rom";
	}
	else if (stat(my_directory,&status)>=0)
	{
		rom=my_directory;
	}

	if(rom==NULL)
	{
		printf("Warning: The file \"spectrum.rom\" needed for proper conversion to the .ach\n");
		printf("         format could not be located, Converting anyway. The ROM space\n");
		printf("         will be filled with 0 bytes, the converted file may not work\n");

		/* write the 16K ram area as zero's */
		for(i=0; i < 16; i++)
		{
			if(write(fd,buffer,1024)==-1)
			{
				perror(s);
				exit(2);
			}
		}
	}
	else
	{
		int  fd_specrom;

		if (status.st_size != 16384)
		{
		printf("Warning: The file \"spectrum.rom\" needed for proper conversion to the .ach\n");
		printf("         format has a wrong size (not 16K). Converting anyway. The ROM space\n");
		printf("         will be filled with 0 bytes, the converted file may not work\n");

			/* copy the 16K ROM area */
			for(i=0; i < 16; i++)
			{
				if(write(fd,buffer,1024)==-1)
				{
					perror(s);
					exit(2);
				}
			}
		}
		else
		{
			printf("Using Spectrum ROM: %s\n",rom);

			fd_specrom=open(rom,O_RDONLY|O_BINARY);

			if(fd_specrom<0)
			{
					perror(rom);
					exit(2);
			}

			for(i=0; i < 16; i++)
			{
				if(read(fd_specrom,buffer,1024)==-1)
				{
					perror(rom);
					exit(2);
				}
				if(write(fd,buffer,1024)==-1)
				{
					perror(s);
					exit(2);
				}
			}
			close(fd_specrom);
		}
	}
	write_image(s);
	close(fd);
}

/* SPECIFIC I/O - READ/WRITE .ZX IMAGE */

void read_kgb(char * s)
{
	fd=open(s,O_RDONLY|O_BINARY);
	if(fd < 0)
	{
		perror(s);
		exit(1);
	}

	/* lseek over the first 132 bytes */
	if(lseek(fd,132L,SEEK_CUR)==-1L)
	{
		perror(s);
		exit(1);
	}

	read_image(s);

	if(read(fd,&kgb,sizeof(kgb))==-1)
	{
		perror(s);
		exit(1);
	}

	close(fd);
}

void write_kgb(char * s)
{
	char buffer[132];
	char *p;
	int  i;
	struct stat status;
	char * rom;

	/* clean buffer first */
	p=(char *) buffer;
	for(i=0; i < 132; i++)
		p[i]='\0';

	unlink(s);

	fd=open(s,O_WRONLY|O_CREAT|O_BINARY,0666);
	if(fd<0)
	{
		perror(s);
		exit(2);
	}

	strcat(my_directory,"\\spectrum.rom");

	rom=NULL;

	if(stat("spectrum.rom",&status)>=0)
	{
		rom="spectrum.rom";
	}
	else if (stat(my_directory,&status)>=0)
	{
		rom=my_directory;
	}

	if(rom==NULL)
	{
		printf("Warning: The file \"spectrum.rom\" needed for proper conversion to the .zx\n");
		printf("         format could not be located, Converting anyway. The 132 bytes needed\n");
		printf("         from the ROM will be filled with 0 bytes, the converted file may not\n");
		printf("         work\n");

		/* write the 132 byte area as zero's */
		if(write(fd,buffer,132)==-1)
		{
			perror(s);
			exit(2);
		}
	}
	else
	{
		int  fd_specrom;

		if (status.st_size != 16384)
		{
		printf("Warning: The file \"spectrum.rom\" needed for proper conversion to the .zx\n");
		printf("         format has a wrong size (not 16K). Converting anyway. The 132 bytes\n");
		printf("         needed from the ROM will be filled with 0 bytes, the converted file\n");
		printf("         may not work\n");

			/* write the 132 byte area as zero's */
			if(write(fd,buffer,132)==-1)
			{
				perror(s);
				exit(2);
			}
		}
		else
		{
			printf("Using Spectrum ROM: %s\n",rom);

			fd_specrom=open(rom,O_RDONLY|O_BINARY);

			if(fd_specrom<0)
			{
					perror(rom);
					exit(2);
			}

			/* lseek over the first 16384-132 bytes */
			if(lseek(fd_specrom,16252L,SEEK_CUR)==-1L)
			{
				perror(rom);
				exit(2);
			}
			if(read(fd_specrom,buffer,132)==-1)
			{
				perror(rom);
				exit(2);
			}
			if(write(fd,buffer,132)==-1)
			{
				perror(s);
				exit(2);
			}
			close(fd_specrom);
		}
	}

	write_image(s);
	if(write(fd,&kgb,sizeof(kgb))==-1)
	{
		perror(s);
		exit(2);
	}

	close(fd);
}

/* CONVERSION FUNCTIONS - TO .SNA FORMAT */

void vgaspec_to_sna()
{
	unsigned int addr;
	unsigned int sp;

	sna.f=vga.f;
	sna.a=vga.a;
	sna.b=vga.b;
	sna.c=vga.c;
	sna.d=vga.d;
	sna.e=vga.e;
	sna.h=vga.h;
	sna.l=vga.l;

	sna.fax=vga.fax;
	sna.aax=vga.aax;
	sna.bax=vga.bax;
	sna.cax=vga.cax;
	sna.dax=vga.dax;
	sna.eax=vga.eax;
	sna.hax=vga.hax;
	sna.lax=vga.lax;

	sna.ixh=vga.ixh;
	sna.ixl=vga.ixl;
	sna.iyh=vga.iyh;
	sna.iyl=vga.iyl;

	sna.border=vga.border; 

	sna.i=vga.i;
	sna.r=vga.r;

	/* If register I has changed, chances are good that it runs in
	   IM2 mode */
	if(sna.i==0x3f)
		sna.im=0x01; 
	else
		sna.im=0x02; 
		
	if((vga.im & 0x01) == 0)
		sna.iff2=0x00;
	else
		sna.iff2=0xff;

	sp=256*vga.sph+vga.spl;
	sp=sp-2;
	addr=sp-0x4000;
	image[addr]=vga.pcl;
	image[addr+1]=vga.pch;
		
	sna.sph=sp/256;
	sna.spl=sp%256;
}

void raw_to_sna()
{
	unsigned int addr;
	unsigned int sp;
	unsigned int pc;

	pc=0x1bf4; /* entry of "next statement" */

	sna.f=0x99;
	sna.a=0x5f;
	sna.b=0x1f;
	sna.c=0xf0;
	sna.d=0x5d;
	sna.e=0x0c;
	sna.h=0x5d;
	sna.l=0x0e;

	sna.fax=0x44;
	sna.aax=0x00;
	sna.bax=0x18;
	sna.cax=0x20;
	sna.dax=0x00;
	sna.eax=0x07;
	sna.hax=0x5c;
	sna.lax=0xf1;

	sna.ixh=0x03;
	sna.ixl=0xd4;
	sna.iyh=0x5c;
	sna.iyl=0x3a;

	sna.i=0x3f;
	sna.r=0x00;
	sna.im=0x01;
	sna.iff2=0xFF;

	/* set sp by means of RAMTOP in the image */
	addr=0x5cb2-0x4000;
	sp=256*image[addr+1]+image[addr]-1;

	/* Reset ERR NR to no error */
	image[0x5c3a-0x4000]=0xff;
	
	/* Set border by means of BORDCR */
	sna.border=(image[0x5c48-0x4000] & 0x38)>>3;
	
	/* put return address to MAIN-4 (0x1303) on stack */
	sp=sp-2;
	addr=sp-0x4000;	
	image[addr]=0x03; 
	image[addr+1]=0x13;
	
	sp=sp-2;
	addr=sp-0x4000;
	image[addr]=pc%256;
	image[addr+1]=pc/256;

	sna.sph=sp/256;
	sna.spl=sp%256;
}

void z80_to_sna()
{
	unsigned int addr;
	unsigned int sp;

	sna.f=z80.f;
	sna.a=z80.a;
	sna.b=z80.b;
	sna.c=z80.c;
	sna.d=z80.d;
	sna.e=z80.e;
	sna.h=z80.h;
	sna.l=z80.l;

	sna.fax=z80.fax;
	sna.aax=z80.aax;
	sna.bax=z80.bax;
	sna.cax=z80.cax;
	sna.dax=z80.dax;
	sna.eax=z80.eax;
	sna.hax=z80.hax;
	sna.lax=z80.lax;

	sna.ixh=z80.ixh;
	sna.ixl=z80.ixl;
	sna.iyh=z80.iyh;
	sna.iyl=z80.iyl;

	sna.border=(z80.data/2) & 0x07; 

	sna.i=z80.i;

	if(z80.data==0xff) z80.data=0;

	if((z80.data & 0x01)==1)
		sna.r=(z80.r & 0x7f)+0x80;
	else
		sna.r=z80.r & 0x7f;

	sna.im=z80.im & 0x03;
	
	if(z80.iff2 != 0)
		sna.iff2=0xff;
	else
		sna.iff2=0x00;

	sp=256*z80.sph+z80.spl;
	sp=sp-2;
	addr=sp-0x4000;
		
	sna.sph=sp/256;
	sna.spl=sp%256;

	if((z80.data & 0x20)!=0)
		z80_uncompress();

	/* PC can only be stored in the image after decompression!! */
	image[addr]=z80.pcl;
	image[addr+1]=z80.pch;
}

void prg_to_sna()
{
	unsigned int addr;
	unsigned int sp;

	sna.b=prg.b;
	sna.c=prg.c;
	sna.d=prg.d;
	sna.e=prg.e;
	sna.h=prg.h;
	sna.l=prg.l;
	sna.fax=prg.fax;
	sna.aax=prg.aax;
	sna.bax=prg.bax;
	sna.cax=prg.cax;
	sna.dax=prg.dax;
	sna.eax=prg.eax;
	sna.hax=prg.hax;
	sna.lax=prg.lax;

	sna.ixh=prg.ixh;
	sna.ixl=prg.ixl;
	sna.iyh=prg.iyh;
	sna.iyl=prg.iyl;

	/* Set border by means of BORDCR */
	sna.border=(image[0x5c48-0x4000] & 0x38)>>3;

	sna.i=prg.i;

	if(prg.i==0x3f)
		sna.im=0x01;
	else
		sna.im=0x02;

	sp=256*prg.sph+prg.spl;
	sp=sp+4; /* there are two more words on the stack besides PC */
	addr=sp-0x4000;

	sna.r=image[addr-3];
	/* the af combo is on the stack */
	sna.f=image[addr-2];
	sna.a=image[addr-1];
		
	sna.sph=sp/256;
	sna.spl=sp%256;

	/* interrupts always on ??? */
	sna.iff2=prg.iff2;
}

void ach_to_sna()
{
	unsigned int addr;
	unsigned int sp;

	sna.f=ach.f;
	sna.a=ach.a;
	sna.b=ach.b;
	sna.c=ach.c;
	sna.d=ach.d;
	sna.e=ach.e;
	sna.h=ach.h;
	sna.l=ach.l;

	sna.fax=ach.fax;
	sna.aax=ach.aax;
	sna.bax=ach.bax;
	sna.cax=ach.cax;
	sna.dax=ach.dax;
	sna.eax=ach.eax;
	sna.hax=ach.hax;
	sna.lax=ach.lax;

	sna.ixh=ach.ixh;
	sna.ixl=ach.ixl;
	sna.iyh=ach.iyh;
	sna.iyl=ach.iyl;

	sna.border=ach.border;

	sna.i=ach.i;

	sna.r=ach.r;

	sna.im=ach.im & 0x02; 
		
	sna.iff2=ach.iff2;

	sp=256*ach.sph+ach.spl;
	sp=sp-2;
	addr=sp-0x4000;
		
	sna.sph=sp/256;
	sna.spl=sp%256;

	image[addr]=ach.pcl;
	image[addr+1]=ach.pch;
}

void kgb_to_sna()
{
	unsigned int addr;
	unsigned int sp;

	sna.f=kgb.f;
	sna.a=kgb.a;
	sna.b=kgb.b;
	sna.c=kgb.c;
	sna.d=kgb.d;
	sna.e=kgb.e;
	sna.h=kgb.h;
	sna.l=kgb.l;

	sna.fax=kgb.fax;
	sna.aax=kgb.aax;
	sna.bax=kgb.bax;
	sna.cax=kgb.cax;
	sna.dax=kgb.dax;
	sna.eax=kgb.eax;
	sna.hax=kgb.hax;
	sna.lax=kgb.lax;

	sna.ixh=kgb.ixh;
	sna.ixl=kgb.ixl;
	sna.iyh=kgb.iyh;
	sna.iyl=kgb.iyl;

	sna.i=kgb.i;
	sna.r=kgb.r;

	/* border-colour not found in KGB image */
	/* Set border by means of BORDCR */
	sna.border=(image[0x5c48-0x4000] & 0x38)>>3;

	/* determine interrupt mode using the value of register I */
	if (kgb.intmode_l==0xff)
		sna.im=0x00; 
	else if(kgb.intmode_l==1)
		sna.im=0x02; 
	else
		sna.im=0x01;
		
	if((kgb.interruptstatus & 0x01) != 0)
		sna.iff2=0xff;
	else
		sna.iff2=0x0;

	sp=256*kgb.sph+kgb.spl;
	sp=sp-2;
	addr=sp-0x4000;
		
	sna.sph=sp/256;
	sna.spl=sp%256;

	image[addr]=kgb.pcl;
	image[addr+1]=kgb.pch;
}

/* CONVERSION FUNCTIONS - FROM .SNA FORMAT */

void sna_to_vgaspec()
{
	unsigned int addr;
	unsigned int sp;
	unsigned int pc;

	sp=256*sna.sph+sna.spl;
	addr=sp-0x4000;
	pc=image[addr]+256*image[addr+1];
	sp=sp+2;

	vga.S='S';
	vga.P='P';
	vga.len_l=0x00;
	vga.len_h=0xC0;
	vga.start_l=0x00;
	vga.start_h=0x40;
	vga.f=sna.f;
	vga.a=sna.a;
	vga.b=sna.b;
	vga.c=sna.c;
	vga.d=sna.d;
	vga.e=sna.e;
	vga.h=sna.h;
	vga.l=sna.l;

	vga.fax=sna.fax;
	vga.aax=sna.aax;
	vga.bax=sna.bax;
	vga.cax=sna.cax;
	vga.dax=sna.dax;
	vga.eax=sna.eax;
	vga.hax=sna.hax;
	vga.lax=sna.lax;

	vga.ixh=sna.ixh;
	vga.ixl=sna.ixl;
	vga.iyh=sna.iyh;
	vga.iyl=sna.iyl;

	vga.i=sna.i;
	vga.r=sna.r;

	vga.im=sna.im & 0x02; /* 0 for IM1, 2 for IM2 */

	/* works? how does it know it was IM1 ? */
	if((sna.iff2 & 0x04) != 0)
		vga.im=vga.im | 0x01; 

	vga.sph=sp/256;
	vga.spl=sp%256;
	
	vga.pch=pc/256;
	vga.pcl=pc%256;

	vga.border=sna.border; 

	vga.res2=0;
	vga.res3=0;
	vga.res4=0;
	vga.res5=0;
}

void sna_to_z80()
{
	unsigned int addr;
	unsigned int sp;
	unsigned int pc;

	sp=256*sna.sph+sna.spl;
	addr=sp-0x4000;
	pc=image[addr]+256*image[addr+1];
	sp=sp+2;

	z80.f=sna.f;
	z80.a=sna.a;
	z80.b=sna.b;
	z80.c=sna.c;
	z80.d=sna.d;
	z80.e=sna.e;
	z80.h=sna.h;
	z80.l=sna.l;

	z80.fax=sna.fax;
	z80.aax=sna.aax;
	z80.bax=sna.bax;
	z80.cax=sna.cax;
	z80.dax=sna.dax;
	z80.eax=sna.eax;
	z80.hax=sna.hax;
	z80.lax=sna.lax;

	z80.ixh=sna.ixh;
	z80.ixl=sna.ixl;
	z80.iyh=sna.iyh;
	z80.iyl=sna.iyl;

	z80.i=sna.i;
	z80.r=sna.r | 0x080; /* bit 7 is stored somewhere else, always set */
	z80.im=sna.im & 0x03;
	z80.im=z80.im + 0x60; /* fixed normal video/kempston joystick */

	z80.sph=sp/256;
	z80.spl=sp%256;
	
	z80.pch=pc/256;
	z80.pcl=pc%256;

	/* all kinds of stuff put in "data" */
	z80.data=(sna.border & 0x07)*2; 
	if((sna.r & 0x80)!=0) z80.data=z80.data+1; /* here is bit 7 of r */
	z80.data=z80.data | z80_compress();

	if((sna.iff2 & 0x04) != 0)
	{
		z80.iff1=0xff;
		z80.iff2=0xff;
	}
	else
	{
		z80.iff1=0;
		z80.iff2=0;
	}
}

void sna_to_prg(char * n)
{
	unsigned int addr;
	unsigned int sp;
	int i;
	unsigned char * p;

	/* clean header structure first */
	p=(unsigned char *) &prg;
	for(i=0; i < 256; i++)
		p[i]='\0';

	prg.contains_0x61=0x61; /* size of image in sectors */
	prg.contains_0x35=0x35; /* don't know yet */
	prg.contains_0x03=0x03; /* don't know yet */

	sp=256*sna.sph+sna.spl;
	addr=sp-0x4000;

	/* these are on the stack */
	image[addr-1]=sna.a;
	image[addr-2]=sna.f;
	image[addr-3]=sna.r;
	image[addr-4]=sna.iff2;

	sp=sp-4;

	prg.name[0]='\0';
	strncpy(prg.name,n,10);
	prg.name[10]='\0';

	prg.b=sna.b;
	prg.c=sna.c;
	prg.d=sna.d;
	prg.e=sna.e;
	prg.h=sna.h;
	prg.l=sna.l;

	prg.fax=sna.fax;
	prg.aax=sna.aax;
	prg.bax=sna.bax;
	prg.cax=sna.cax;
	prg.dax=sna.dax;
	prg.eax=sna.eax;
	prg.hax=sna.hax;
	prg.lax=sna.lax;

	prg.ixh=sna.ixh;
	prg.ixl=sna.ixl;
	prg.iyh=sna.iyh;
	prg.iyl=sna.iyl;

	prg.i=sna.i;
	prg.iff2=sna.iff2;

	prg.sph=sp/256;
	prg.spl=sp%256;
	
	/* prg.border=sna.border; */
}

void sna_to_ach()
{
	unsigned int addr;
	unsigned int sp;
	unsigned int pc;
	int i;
	unsigned char * p;

	/* clean header structure first */
	p=(unsigned char *) &ach;
	for(i=0; i < 256; i++)
		p[i]='\0';

	sp=256*sna.sph+sna.spl;
	addr=sp-0x4000;
	pc=image[addr]+256*image[addr+1];
	sp=sp+2;

	ach.f=sna.f;
	ach.a=sna.a;
	ach.b=sna.b;
	ach.c=sna.c;
	ach.d=sna.d;
	ach.e=sna.e;
	ach.h=sna.h;
	ach.l=sna.l;

	ach.fax=sna.fax;
	ach.aax=sna.aax;
	ach.bax=sna.bax;
	ach.cax=sna.cax;
	ach.dax=sna.dax;
	ach.eax=sna.eax;
	ach.hax=sna.hax;
	ach.lax=sna.lax;

	ach.ixh=sna.ixh;
	ach.ixl=sna.ixl;
	ach.iyh=sna.iyh;
	ach.iyl=sna.iyl;

	ach.i=sna.i;
	ach.r=sna.r;

	ach.border=sna.border;

	if((sna.iff2 & 0x04) != 0)
		ach.iff2=0xff;
	else
		ach.iff2=0x00;

	ach.sph=sp/256;
	ach.spl=sp%256;
	
	ach.pch=pc/256;
	ach.pcl=pc%256;
}

void sna_to_kgb()
{
	unsigned int addr;
	unsigned int sp;
	unsigned int pc;
	int i;
	unsigned char * p;

	/* clean info structure first */
	p = (unsigned char *) &kgb;
	for(i=0; i < 202; i++)
		p[i]='\0';

	/* make some assumptions here */
	kgb.is3_1 = 3;		/* always 3, don't ask me why */
	kgb.colourmode = 1;	/* assume colour */
	kgb.soundmode = 1;	/* assume simple sound */
	kgb.haltmode = 1;	/* assume not in halt mode */

	sp=256*sna.sph+sna.spl;
	addr=sp-0x4000;
	pc=image[addr]+256*image[addr+1];
	sp=sp+2;

	kgb.f=sna.f;
	kgb.a=sna.a;
	kgb.b=sna.b;
	kgb.c=sna.c;
	kgb.d=sna.d;
	kgb.e=sna.e;
	kgb.h=sna.h;
	kgb.l=sna.l;

	kgb.fax=sna.fax;
	kgb.aax=sna.aax;
	kgb.bax=sna.bax;
	kgb.cax=sna.cax;
	kgb.dax=sna.dax;
	kgb.eax=sna.eax;
	kgb.hax=sna.hax;
	kgb.lax=sna.lax;

	kgb.ixh=sna.ixh;
	kgb.ixl=sna.ixl;
	kgb.iyh=sna.iyh;
	kgb.iyl=sna.iyl;

	kgb.i=sna.i;

	kgb.r=sna.r;

	/* kgb.border=sna.border; NOT IN KGB IMAGE! */

	/* Interupt mode is stored in a word in the KGB format. */
	/* Use byte accesses to be CPU independent              */

	switch (sna.im & 0x03) {
	case 0:	kgb.intmode_h = 0xff;
		kgb.intmode_l = 0xff;
		break;
	case 2: kgb.intmode_h = 0;
		kgb.intmode_l = 1;
		break;
	default: kgb.intmode_h = 0;
		kgb.intmode_l = 0;
		break;
	}

	if((sna.iff2 & 0x04) != 0)
		kgb.interruptstatus=0x01;
	else
		kgb.interruptstatus=0x00;

	kgb.sph=sp/256;
	kgb.spl=sp%256;
	
	kgb.pch=pc/256;
	kgb.pcl=pc%256;
}

/* COMPRESSION/DECOMPRESSION for .Z80 FORMAT */

void z80_uncompress()
{
#ifdef __TURBOC__
	unsigned char far * uc;
#else
	unsigned char * uc;
#endif
	unsigned int i,j,k;
	unsigned char l;

#ifdef __TURBOC__
	uc=farmalloc(IMSIZE+0x0100);
#else
	uc=malloc(IMSIZE+0x0100);
#endif

	if(uc==NULL)
	{
		fprintf(stderr,"Not enough memory to uncompress z80 image\n");
		exit(7);
	}

	j=0;
	i=0;
	while(i<IMSIZE)
	{
		if(image[j]!=0xed)
			uc[i]=image[j];
		else if(image[j+1]!=0xed)
			uc[i]=image[j];
		else
		{
			/* fetch count */
			k=(int) image[j+2];
			/* fetch character */
			l=image[j+3];
			while(k!=0)
			{
				uc[i]=l;
				i++;
				k--;
			}
			j=j+3;
			i--;
		}
		i++;
		j++;
	}

	if(i!=IMSIZE)
	{
		fprintf(stderr,"Z80 image corrupted, can't decompress\n");
		fprintf(stderr,"i=%u, should be %u\n",i,IMSIZE);
		exit(6);
	}

	/* copy back */
	j=0;
	i=0;
	while(i<IMSIZE)
		image[j++]=uc[i++];

#ifdef __TURBOC__
	farfree(uc);
#else
	free(uc);
#endif
}

int z80_compress()
{
#ifdef __TURBOC__
	unsigned char far * comp;
#else
	unsigned char * comp;
#endif
	unsigned int i,j;
	unsigned int num;
	unsigned char c,n;
	unsigned int ed;

	z80_size=(unsigned int) IMSIZE;

#ifdef __TURBOC__
	comp=farmalloc(IMSIZE+0x0100);
#else
	comp=malloc(IMSIZE+0x0100);
#endif
	if(comp==NULL)
	{
		printf("Warning: Not enough memory to compress the image, using uncopressed image\n");
		return NOTCOMPRESSED;
	}

	i=0;
	j=0;
	/* ensure 'ed' is not set */
	ed=NO;
	while(i<IMSIZE)
	{
		c=image[i];
		i++;
		if(i<IMSIZE)
		{
			n=image[i];
		}
		else
		{
			/* force 'n' to be unequal to 'c' */
			n=c;
			n++;
		}

		if(c!=n)
		{
			comp[j]=c;
			j++;
			if(c==0xed) 
				ed=YES;
			else
				ed=NO;
		}
		else
		{
			if(c==0xed)
			{
				/* two times 0xed - special care */
				comp[j]=0xed;
				j++;
				comp[j]=0xed;
				j++;
				comp[j]=0x02;
				j++;
				comp[j]=0xed;
				j++;
				i++; /* skip second ED */

				/* because 0xed is valid compressed we don't
				   have to watch it! */
				ed=NO;
			}
			else if(ed==YES)
			{
				/* can't compress now, skip this double pair */
				comp[j]=c;
				j++;
				ed=NO;	/* 'c' can't be 0xed */
			}
			else
			{
				num=1;
				while(i<IMSIZE)
				{
					if(c!=image[i])
						break;
					num++;
					i++;
					if(num==255)
						break;
				}
				if(num <= 4)
				{
					/* no use to compress */
					while(num!=0)
					{
						comp[j]=c;
						j++;
						num--;
					}
				}
				else
				{
					comp[j]=0xed;
					j++;
					comp[j]=0xed;
					j++;
					comp[j]=(unsigned char) num;
					j++;
					comp[j]=c;
					j++;
				}
			}
		}

		if(j >= (IMSIZE-4))
		{
			/* compressed image bigger or same than original */
#ifdef __TURBOC__
			farfree(comp);
#else
			free(comp);
#endif
			return NOTCOMPRESSED;
		}
	}
	/* append "end of compressed area" mark */
	comp[j]=0;
	j++;
	comp[j]=0xed;
	j++;
	comp[j]=0xed;
	j++;
	comp[j]=0;
	j++;

	z80_size = j;
	
	/* copy back */
	i=0;
	j=0;
	while(i<IMSIZE)
		image[i++]=comp[j++];
#ifdef __TURBOC__
	farfree(comp);
#else
	free(comp);
#endif

	return COMPRESSED;
}
