/*
 * socketcall interface for linux
 *
 * socket system calls are handled indirect, they all go through one real
 * system call. sys_socketcall() will call the right sys_... function with
 * tcp->u_args setup as if the function was called direct from syscall()
 */

#include <stdio.h>
#include <sys/types.h>
#include "defs.h"

int sys_socket(), sys_bind(), sys_connect(), sys_listen();
int sys_accept(), sys_getsockname(), sys_getpeername(), sys_socketpair();
int sys_send(), sys_recv(), sys_sendto(), sys_recvfrom();
int sys_shutdown(), sys_setsockopt(), sys_getsockopt();

extern int errno;

static struct sysent {
	int	nargs;
	int	(*sys_func)();
	char	*sys_name;
} sysent[] = {
	{1, printargs, "socketcall"},
	{3, sys_socket, "socket"},
	{3, sys_bind, "bind"},
	{3, sys_connect, "connect"},
	{2, sys_listen, "listen"},
	{3, sys_accept, "accept"},
	{3, sys_getsockname, "getsockname"},
	{3, sys_getpeername, "getpeername"},
	{4, sys_socketpair, "socketpair"},
	{4, sys_send, "send"},
	{4, sys_recv, "recv"},
	{6, sys_sendto, "sendto"},
	{6, sys_recvfrom, "recvfrom"},
	{2, sys_shutdown, "shutdown"},
	{5, sys_setsockopt, "setsockopt"},
	{5, sys_getsockopt, "getsockopt"},
};
static int nsyscall = sizeof sysent / sizeof sysent[0];


sys_socketcall(tcp)
struct tcb *tcp;
{
	int socketcall;
	char *addr;
	int i;

	if (entering(tcp))
		socketcall = tcp->u_args[7] = tcp->u_args[0];
	else
		socketcall = tcp->u_args[7];

	if (socketcall >= nsyscall) {
		if (entering(tcp))
			fprintf(outf, "socketcall(%d, %#x", socketcall, tcp->u_args[1]);
		return 0;
	}

	if (entering(tcp)) {
		fprintf(outf, "%s(", sysent[socketcall].sys_name);
		addr = (char *) tcp->u_args[1];
		for (i=0; i<sysent[socketcall].nargs; i++, addr += 4) {
			errno = 0;
			tcp->u_args[i] = ptrace(PTRACE_PEEKDATA, tcp->pid, addr, 0);
			if (errno) {
				perror("socketcall: ptrace(PTRACE_PEEKDATA)");
				return -1;
			}
		}
	}
	return (*sysent[socketcall].sys_func)(tcp);
}
