/* Hypercube C version of PIC */

#include <stdio.h>
#include "pic.h"

double sqrt();
long clock();

struct {	long push;
		long field;
		long comm;
		long charge;
		long total;
	} times;
long time0, time1;

float prt_x[PART_PROC];
float prt_y[PART_PROC];
float prt_vx[PART_PROC];
float prt_vy[PART_PROC];
                             
int debug=1;   /* debug=1 - printouts debug info */                  
int nsys=0;   /* syslog ordinals, incremented before each syslog */
int doc=0;    /*   degree of cube for whoami  */

int proc_list[NPROCX][NPROCY]; /* list of procs when mapped to grid */

int part_cnt[NCELLX][NCELLY];   /* array of particle counts */
int proc_map2dx;         /* proc ordinal in x direction */
int proc_map2dy;         /* proc ordinal in y direction */

float x_orig;   /* x origin of grid in this processor */
float y_orig;   /* y origin of grid in this processor */
float dxmin;
float dt;       /*   time step */
float energy;     /* kinetic energy of particles */
float q[NPOINTX+1][NPOINTY+1];     /* charge density */
float ans[(NPOINTX+1)*(NPOINTX+1)];
float f[NPOINTX+1][NPOINTY+1];     /* potential */
float ex[NCELLX][NPOINTY+1];    /* x component of E-field */
float ey[NPOINTX+1][NCELLY];    /* y component of E-field */

/*          grid definitions:
	(q,f)......ex......(q,f)
          .                  .
          .                  .
          .                  .
         ey                 ey    <--- a typical cell
          .                  .
          .                  .
          .                  .
        (q,f).....ex.......(q,f)
*/

float scratch[NPOINTS], temp[NPOINTS_PROC];

int n_proc,s_proc,e_proc,w_proc; 
int npart;     /* number of particles in this node */
int ncellx_proc;  /* number of cells per proc in x direction */
int ncelly_proc;  /* number of cells per proc in y direction */
int d;      /* degree of hypercube */
int num_procs;   /*  number of nodes in cube */
int procnum;    /* node number of this node */
int pid;        /* process id of this process */
int myparity;   /* parity of procnum: 0 for even number of bits */
int ci[8];    /* logical channels for communications  */
float totim;  /*  problem time  */
char string[132];

main()
{
float a[SA], c[SA], z[NR], r[NR], qbar;
int nit, time, max_level;
long l, n;

	/* initialize particle positions and other variables */

	setup();

	/* initialize multigrid routines */

	sfsu2t((int)ncellx_proc,(int)ncelly_proc,&max_level,a,c);
	szro1(f,(long)NV);
	nit=5;		/* number of multigrid iterations to converge */      
	qbar=(float)MAXPART/(float)(NCELLX*NCELLY);
	

	/* while loop over time */

	time=0;
/*      do a universal broadcast as a startup sync */
	universal_broadcast(d,procnum,scratch,1,ci,pid);

	times.total=clock();
	while (time<NITIT) {  

		time0=clock();
		charge_density(q);
		time1=clock();
		times.charge += (time1-time0);

		if (d != 0) accumulate_charge(q);
		if(debug==2) print_q(q,ncellx_proc+2,ncelly_proc+2);
		time0=clock();
		first_guess(f);
        	sfap2t(max_level,nit,(int)ncellx_proc,(int)ncelly_proc,q,a,c,f,r,z,qbar);  
		time1=clock();
		times.field += (time1-time0);

		if (d != 0) broadcast_potential(f);
		if(debug==2) mprint(f,NCELLX,NCELLY," potential");
         	if(debug==2) del2(f,ans); 
		
		time0=clock();
         	pmove();

		time1=clock();
		times.push += (time1-time0);

	        time++;
		if(debug>=0) {	
			sprintf(string," time iter: %d ",time);
			syslog(nsys,string);
			send_part_cnt();
		}
      }
	times.total = (clock()-times.total);

	send_part_cnt();
	sprintf(string,"total, push, field, charge, comm: %ld %ld %ld %ld %ld",times.total,times.push,times.field,times.charge,times.comm);
	syslog(0,string);

	sprintf(string," time iter: %d problem time: %f",time,totim);
	syslog(0,string);
	sprintf(string," final dt: %f",dt);
   	syslog(0,string);

}  

accumulate_charge(q)
float q[];
{
	int ngrid, k, i, j, ii, jj, is, js, proc, nx2, ny2;

	nx2=ncellx_proc+2;
	ny2=ncelly_proc+2;
	ngrid=ncellx_proc*ncelly_proc;

	if(debug==2) syslog(ngrid," accumulate_charge");

	for(j=0; j<NPROCY; j++) {
		for(i=0; i<NPROCX; i++) {
			proc=proc_list[i][j];
			is=ncellx_proc*i+1;
			js=ncelly_proc*j+1;
			k=0;
			for(jj=js; jj<(js+ncelly_proc); jj++)
				for(ii=is; ii<(is+ncellx_proc); ii++) {
					scratch[proc*ngrid+k]=q[ii*(NPOINTY+1)+jj];
					k++;
				}
	}	}

	time0=clock();
	universal_accumulate(d,procnum,scratch,temp,ngrid*sizeof(float),ci,pid);

	time1=clock();
	times.comm += (time1-time0);

	k=0;
	for(jj=1; jj<=ncelly_proc; jj++) {
		for(ii=1; ii<=ncellx_proc; ii++) {
			q[ii*ny2+jj]=scratch[procnum*ngrid+k];
			k++;
		}
	}

/*     recv and send ghost cell information */

	scsp2t(q,ncellx_proc,ncelly_proc);
}

broadcast_potential(f)
float f[];
{
	int ngrid, k, i, j, ii, jj, is, js, proc, nx2, ny2, n1, n2, n3;

	nx2=ncellx_proc+2;
	ny2=ncelly_proc+2;
	ngrid=ncellx_proc*ncelly_proc;

	if(debug==2) syslog(ngrid," broadcast_potential");

	k=0;
	for(ii=1; ii<=ncellx_proc; ii++)
		for(jj=1; jj<=ncelly_proc; jj++) {
			scratch[procnum*ngrid+k]=f[ii*ny2+jj];
			k++;
		}

	time0=clock();
	universal_broadcast(d,procnum,scratch,ngrid*sizeof(float),ci,pid);

	time1=clock();
	times.comm += (time1-time0);

	for(j=0; j<NPROCY; j++) {
		for(i=0; i<NPROCX; i++) {
			proc=proc_list[i][j];
			is=ncellx_proc*i+1;
			js=ncelly_proc*j+1;
			k=0;
			for(ii=is; ii<(is+ncellx_proc); ii++)
				for(jj=js; jj<(js+ncelly_proc); jj++) {
					f[ii*(NPOINTY+1)+jj]=scratch[proc*ngrid+k];
					k++;
	}	}		}

/*     take care of ghost cells */

	n1=NPOINTY+1;
	n2=n1*NCELLX;
	n3=n2+n1;
	for(j=1; j<=NCELLY; j++) {
		f[j]=f[n2+j];
		f[n3+j]=f[n1+j];
	}
	for(i=0; i<=NPOINTX+1; i++) {
		f[n1*i]=f[NCELLY+n1*i];
		f[NCELLY+1+n1*i]=f[1+n1*i];
	}
}

first_guess(f)
float f[];
{
	int i, j, is, js, k, ii, jj, proc;

	for(j=0; j<NPROCY; j++) {
		for(i=0; i<NPROCX; i++) {
			if((proc=proc_list[i][j])!=procnum) continue;

			is=ncellx_proc*i;
			js=ncelly_proc*j;
			for(ii=is; ii<(is+ncellx_proc+2); ii++)
				for(jj=js; jj<(js+ncelly_proc+2); jj++) {
					i=ii-is; j=jj-js;
					f[i*(ncelly_proc+2)+j] = f[ii*(NPOINTY+1)+jj];
				}
			if(debug==2) mprint(f,ncellx_proc,ncelly_proc,"pot");
			return;
	}	}
}


charge_density(q)
float q[NPOINTX+1][NPOINTY+1];
{   
int i, j, n, ix, iy;
float x, y;
	for (i=0; i<NPOINTX+1; i++) 
		for (j=0; j<NPOINTY+1; j++) 
			q[i][j] = 0.0;
   
	for (i=0; i<NCELLX; i++)
		for (j=0; j<NCELLY; j++)
			part_cnt[i][j] = 0;
   
	for (n=0; n<npart; n++) {
		x = prt_x[n]/DX;
		ix = x;
		y = prt_y[n]/DY;
		iy = y;
 
		x = x - (float)ix;
		y = y - (float)iy;
		part_cnt[ix][iy]++;
      
		energy += prt_vx[n]*prt_vx[n] 
		          + prt_vy[n]*prt_vy[n];
		      
		q[ix][iy] += CHG/MASS*(1.0-x)*(1.0-y);
		q[ix+1][iy] += CHG/MASS*x*(1.0-y);
		q[ix][iy+1] += CHG/MASS*(1.0-x)*y;
		q[ix+1][iy+1] += CHG/MASS*x*y;
	}
	
	for (i=0; i<NPOINTX; i++)
	{	q[i][NPOINTY-1] += q[i][0];
		q[i][0] = q[i][NPOINTY-1];
		q[i][NPOINTY] = q[i][1];
	}
	for (j=0; j<NPOINTY; j++)
	{	q[NPOINTX-1][j] += q[0][j];
		q[0][j] = q[NPOINTX-1][j];
		q[NPOINTX][j] = q[1][j];
	}

	if(debug==2) print_q(q,NPOINTX+1,NPOINTY+1);
}

pmove()
{
int n, i, j, ix, iy;
float amax, vmax, dt1, dt2;
float ax, ay, x, y;
float absf(), temp,dist;
      
   vmax=0.;
   amax=0.;
   
   for (i=0; i<NCELLX; i++) 
      for(j=0; j<NPOINTY; j++) 
         ex[i][j]= (f[i+1][j] - f[i][j]) /DX;
   for (i=0; i<NPOINTX; i++) 
      for(j=0; j<NCELLY; j++) 
         ey[i][j]= (f[i][j+1] - f[i][j]) /DY;
   
   for (n=0; n<npart; n++) {

	if(debug==2) {
		sprintf(string," old position:  %f %f",prt_x[n],prt_y[n]);
		nsys++;
		syslog(nsys,string);
	}

	x = prt_x[n]/DX;
	ix =(int)x;
	y = prt_y[n]/DY;
	iy = (int)y;
 
	x = x - (float)ix;
	y = y - (float)iy;
      
        ax = CHG/MASS*(ex[ix][iy]*(1.0-y) + ex[ix][iy+1]*y);
        ay = CHG/MASS*(ey[ix][iy]*(1.0-x) + ey[ix+1][iy]*x);
      
        prt_vx[n] +=  ax*dt;
        prt_vy[n] +=  ay*dt;

	if(debug==2) {
		sprintf(string," velocity:   %f %f",prt_vx[n],prt_vy[n]);
		nsys++;
		syslog(nsys,string);
	}
      
        x = prt_x[n] + dt*prt_vx[n];
        y = prt_y[n] + dt*prt_vy[n];

	if(debug>=1) {
		dist = (x-prt_x[n])*(x-prt_x[n])+(y-prt_y[n])*(y-prt_y[n]);
		dist = sqrt((double)dist);
		if(dist>DX) {
			sprintf(string," ERROR: travel dist %f is greater than limit %f",dist,DX);
			syslog(n,string);
	}	}

	prt_x[n] = (x>1.0) ? (x-1.0) : x;
	prt_y[n] = (y>1.0) ? (y-1.0) : y;
	prt_x[n] = (x<0.0) ? (x+1.0) : x;
	prt_y[n] = (y<0.0) ? (y+1.0) : y;
      
	if(debug==2) {
		sprintf(string,"  new position: %f %f",prt_x[n],prt_y[n]);
		nsys++;
		syslog(nsys,string);
	}

	absf(prt_vx[n],&temp);
	vmax= (temp>vmax) ? temp : vmax;
	absf(prt_vy[n],&temp);
	vmax= (temp>vmax) ? temp : vmax;
	absf(ax,&temp);
        amax=(temp>amax)? temp : amax;
	absf(ay,&temp);
        amax=(temp>amax)? temp : amax;
   }
   totim += dt;
	if(debug==2) {
		sprintf(string,"  vmax amax %f %f",vmax,amax);
		nsys++;
		syslog(nsys,string);
	}
    
   	if (d != 0) {
		global_max(&amax);
   		global_max(&vmax);
	}

	if(debug>=1) {
		sprintf(string," global vmax amax %f %f",vmax,amax);
		syslog(nsys,string);
	}
 
   if(vmax==0.) {
	syslog(nsys,"  vmax is 0.");
   }
   if(amax==0.) {
	syslog(nsys,"  amax is 0.");
   }
   dt1=dxmin/vmax;
   dt2 = (float)sqrt( (double)(2.*dxmin/amax));
   dt = (dt1<dt2) ? dt1 : dt2;
   if(debug>=1) {
   	sprintf(string," new dt: %f",dt);
   	syslog(0,string);
   }
}
shift(buffer,in_proc,out_proc,len)
char buffer[];
int in_proc, out_proc;
int len;
{
        int i, count, proc, rpid;
        
        if (out_proc != NULLPROC) {
		send(ci[0],procnum,buffer,len,out_proc,pid);
        }
        if (in_proc != NULLPROC) {
		recvw(ci[1],in_proc,buffer,len,&count,&proc,&rpid);
        }
}
send_part_cnt()
{
	int len, i, j;
	len=NCELLX*NCELLY*sizeof(int);

	for(j=0; j<NCELLY; j++) {
		for(i=0; i<NCELLX; i++) {
			sprintf(&string[i*4]," %3d",part_cnt[i][j]);
		}
		syslog(0,string);
	}

	send(ci[0],(int)100,part_cnt,len,(int)HOST,pid);

/*   recv any integer from the host for sync purposes */

	recvw(ci[1],100,&i,sizeof(int),&j,&j,&j);

}

print_q(q,nx,ny)
float q[];
int nx, ny;
{
	int i, j, n;
	nsys++;
	syslog(nsys,"charge density");
	n= (nx-1>11) ? 12 : (nx-1);
	for (j=0; j<=(ny-1); j++){
		for (i=0; i<=n; i++)
			sprintf(&string[7*i],"%7.3f",q[i*ny+j]);
		nsys++;
		syslog(nsys,string);
	}
}               

long nn2t(nx,ny)
int nx, ny;
{  long nn = (nx+2)*(ny+2);
   return(nn);
}
norm2t(r,nx,ny,s)
int nx, ny;
float r[], *s;
{  long i1, i2;
   long np = ny+2;
   float t;

   *s = 0.;
   for( i1=1; i1<=nx; i1++ ){
   for( i2=1; i2<=ny; i2++ ){
      t =  r[i2+np*i1];
      *s += t*t;
   }}
   t=(float)(nx*ny);
   *s = (*s)/t;
   *s = (float)sqrt((double)(*s));
}
sfap2t(max_level,nit,nx,ny,v,a,c,u,r,z,vbar)
int nx,ny;
int max_level, nit;
float v[], a[], c[], u[], r[], z[], vbar;
{  int it, i, j;  long nn; long nn2t();  float s;
   nn=nn2t(nx,ny);

/*      subtract mean value of v */


	if(debug==2) {
		sprintf(string," vbar: %f ",vbar);
		syslog(nsys++,string);
	}

	for (j=0; j<nn; j++) {
/*		u[j]=0.; */
		v[j] -= vbar;
	}

   for(it=1; it<=nit; it++)
   {  sres2t(r,v,u,a,nx,ny);
         if( debug == 1  )
         {  norm2t(r,nx,ny,&s);
            sprintf(string," %2d %10.5e \n",it,s );
            nsys++;
	    syslog(nsys,string);
         }
      fapin(z,r,a,c,nx,ny,max_level);
      sadd1(u,z,nn);
   }
   sres2t(r,v,u,a,nx,ny);
   norm2t(r,nx,ny,&s);
   if(debug>=1) {
   	sprintf(string," norm    %10.5e \n", s );
   	syslog(nsys,string);
   }
}

fapin(z,r,a,c,nx,ny,k)
int nx,ny;
int k;
float z[], r[], a[], c[];
{  long nn; long nn2t();
   nn=nn2t(nx,ny);
   if( k == 0 )
   {  szro1(z,nn);
      spin2t(z,r,c,nx,ny);
   }
   else
   {  sprj2t(nn+r,r,nx,ny);
      fapin(nn+z,nn+r,a,c,nx/2,ny/2,k-1);
      sint2t(z,nn+z,nx,ny);
      sres2t(r,r,z,a,nx,ny);
      spin2t(z,r,c,nx,ny);
}  }
sfsu2t(nx,ny,max_level,a,c)
int nx, ny;
int *max_level;
float a[], c[];
{  
	*max_level = 0;
	while(!(nx&1) && !(ny&1))
	{
		(*max_level)++;
		nx=nx>>1;
		ny=ny>>1;  
   	}
 
	sprintf(string," init multigrid: %d",(*max_level));
	nsys++;
	syslog(nsys,string);
	a[0] = 1.0;    a[1] = 1.0;    a[2] = -8.0;
	c[0] = -.0284; c[1] = -.0356; c[2] = -.157;
}
sprj2t(rj,rk,mkx,mky)
int mkx, mky;
float rk[], rj[];
{  int j1, j2, i1, i2, ii, id, mjx, mjy, np;
   np = mky+2;
   mjx = mkx/2;
   mjy = mky/2;
   for( j1=1; j1<=mjx; j1++ ){
   for( j2=1; j2<=mjy; j2++ ){
      i1 = 2*j1;
      i2 = 2*j2;
      ii = i2 + np*i1;
      id=j2+(2+mjy)*j1;
      rj[id] =  .25*rk[ii-1-np]+.50*rk[ii-1]+.25*rk[ii-1+np];
      rj[id] += .50*rk[ii  -np]+1.0*rk[ii  ]+.50*rk[ii  +np];
      rj[id] += .25*rk[ii+1-np]+.50*rk[ii+1]+.25*rk[ii+1+np];
   }}
   scsp2t(rj,mjx,mjy);
}
szro1(w,nn)
long nn;
float w[];
{  long i;
   for( i=0; i<nn; i++ )
     w[i] = 0.0;
}  
sres2t(r,v,u,a,nx,ny)
int nx, ny;
float r[], v[], u[], a[];
{  int i1, i2, ii, np;
   np = ny+2;
   for( i2=1; i2<=ny; i2++ ){
   for( i1=1; i1<=nx; i1++ ){
      ii = i2 + np*i1;
      r[ii] =   v[ii] - (u[ii-1-np]+   u[ii-1]+u[ii-1+np]);
      r[ii] -= (u[ii  -np]-8.*u[ii  ]+u[ii  +np]);
      r[ii] -= (u[ii+1-np]+   u[ii+1]+u[ii+1+np]);
   }}
   scsp2t(r,nx,ny);
   if(debug==2) mprint(r,nx,ny,"residual");
}
spin2t(z,r,c,nx,ny)
int nx, ny;
float z[], r[], c[3];
{  int i1, i2, ii, np;
   float sum_op;
   np = ny+2;
   for( i1=1; i1<=nx; i1++ ){
   for( i2=1; i2<=ny; i2++ ){
      ii = i2 + np*i1;
      sum_op =          c[0]*r[ii-1-np]+c[1]*r[ii-1]+c[0]*r[ii-1+np];
      sum_op = sum_op + c[1]*r[ii  -np]+c[2]*r[ii  ]+c[1]*r[ii  +np];
      sum_op = sum_op + c[0]*r[ii+1-np]+c[1]*r[ii+1]+c[0]*r[ii+1+np];
      z[ii] = z[ii] + sum_op;
   }}
   scsp2t(z,nx,ny);
}
sint2t(zk,zj,mkx,mky)
int mkx, mky;
float zj[], zk[];
{  int j1, j2, mjx, mjy, mkp, mjp, ik, ij;
   mjx  = mkx/2;
   mjy  = mky/2;
   mkp = mky+2;
   mjp = mjy+2;
   for(j1=1; j1<=mjx; j1++){
   for(j2=1; j2<=mjy; j2++){
      ij = j2   + mjp*j1;
      ik = 2*j2 + mkp*2*j1;
      zk[ik]        =       zj[ij];
      zk[ik-1]      =  0.5*(zj[ij]+zj[ij-1]);
      zk[ik-mkp]    =  0.5*(zj[ij]+zj[ij-mjp]);
      zk[ik-1-mkp]  = 0.25*(zj[ij]+zj[ij-1]+zj[ij-mjp]+zj[ij-1-mjp]);
   }}
   scsp2t(zk,mkx,mky);
}
sadd1(u,w,nn)
long nn;
float w[], u[];
{  long i;
   for( i=0; i<nn; i++ )
     u[i] += w[i];
}  

scsp2t(u,nx,ny)
int nx, ny;
float u[];
{  long  n1, n2, n3;
   int i, j;
   float buff[NPOINTX+1];
   n1 = ny+2;
   n2 = n1*nx;
   n3 = n2+n1;
   if( d == 0 )
   {
      for( i=1; i<=ny; i++)
      {  u[i]    = u[n2+i];
         u[n3+i] = u[n1+i];
      }
      for( j=0; j<=nx+1; j++)
      {  u[    n1*j] = u[ny+n1*j];
         u[ny+1+n1*j] = u[1+n1*j];
   }  }
   else
   {
      for( j=1; j<=ny; j++) buff[j] = u[n2+j];
      shift(&buff[1],w_proc,e_proc,(ny*sizeof(float)));
      for( j=1; j<=ny; j++)
      {
	  u[j] = buff[j]; 
	  buff[j] = u[n1+j]; 
      }
      
      shift(&buff[1],e_proc,w_proc,(ny*sizeof(float)));
      for( j=1; j<=ny; j++ ) u[n3+j] = buff[j];
      for( i=0; i<=nx+1; i++) buff[i] = u[ny+n1*i];
      shift(buff,n_proc,s_proc,((nx+2)*sizeof(float)));
      for( i=0; i<=nx+1; i++)
      {  u[n1*i] = buff[i]; buff[i] = u[1+n1*i];
      }
      shift(buff,s_proc,n_proc,((nx+2)*sizeof(float)));
      for( i=0; i<=nx+1; i++ ) u[ny+1+n1*i] = buff[i];
}  }

del2(f,ans)
float f[NPOINTX+1][NPOINTY+1];
float ans[];
{
	float sum;
        int i, j, kj, ki, lj, li;
	int m2=NCELLY+2;

	i=NCELLX;
	j=NCELLY;

	for (i=0; i<=NPOINTX; i++)
		for (j=0; j<=NPOINTY; j++)
			ans[j+m2*i]=0.;

        for (i=0; i<NPOINTX; i++) {
                for (j=0; j<NPOINTY; j++) {
                kj=j-1;
                lj=j+1;
                ki=i-1;
                li=i+1;
                if(j==0) kj=NPOINTY-2;
                if(i==0) ki=NPOINTX-2;
                sum =              f[ki][kj]+  f[i][kj]+f[li][kj];
                sum = sum         +f[ki][j]-8.*f[i][j]+ f[li][j];
                ans[j+m2*i] = sum +f[ki][lj]+  f[i][lj]+f[li][lj];
        }}
	i=NCELLX;
	j=NCELLY;
        mprint(ans,i,j," del2");
 }       

dummy_pot()
{
        int i,j;
        
        for (i=0; i<NPOINTX; i++)
         for (j=0; j<=NPOINTY; j++)
	{
                f[i][j]=i*.01;  
		if(i == (NPOINTX-1))
		{
			f[i][j]=f[0][j];
			f[i+1][j]=f[1][j];
		}
	}
}

dummy_q()
{
	int i,j;
	dummy_pot();
	del2(f,q);

	for (i=0; i<NPOINTX; i++)
	{	q[i][NPOINTY-1] = q[i][0];
		q[i][0] = q[i][NPOINTY-1];
		q[i][NPOINTY] = q[i][1];
	}
	for (j=0; j<NPOINTY; j++)
	{	q[NPOINTX-1][j] = q[0][j];
		q[0][j] = q[NPOINTX-1][j];
		q[NPOINTX][j] = q[1][j];
	}
	if(debug==2) print_q();

        for (i=0; i<NPOINTX; i++)
         for (j=0; j<=NPOINTY; j++)
                f[i][j]=.0;  
}
 
global_max(x)
float *x;
{
float y;
int step, dir, count, dest_proc, recv_proc, rpid;

	for (step=0; step<=d; step++) {
		if ( myparity == (step&1)) {
			for (dir=0; dir<d; dir++) {
				dest_proc=procnum^(1<<dir);
				send(ci[0],99,x,sizeof(float),dest_proc,pid);
		}	}
		else {
			for (dir=0; dir<d; dir++) {
				recvw(ci[1],99,&y,sizeof(float),&count,&recv_proc,&rpid);
				*x = (y > (*x)) ? y : (*x); 
			}
		}
	}
}

 float frand()
{
        static int i=3097157;
        static int j=4194304;
        i=(3*i)%j;
        return((float)i*2.384185791e-7);
}
setup()
{
	int nrem, count, proc ,rpid, i,j;
	int nprocs[2], perbc[2], recpnum[2], nextproc[2][2];
	int len, n;

	dxmin = DX<DY ? DX : DY;
	dxmin= dxmin/50.;
	dt=.0005;
	d=cubedim();
	doc=d;
	procnum=mynode();
	pid=mypid();
	myparity=parity(procnum);
	num_procs=1<<d;
	ncellx_proc=NCELLX/NPROCX;
	ncelly_proc=NCELLY/NPROCY;

	sprintf(string," procnum, myparity: %d %d",procnum,myparity);
	syslog(nsys,string);
	sprintf(string," grid partitioned: %d x  %d",ncellx_proc,ncelly_proc);
	syslog(nsys,string);

	nprocs[0]=NPROCX; nprocs[1]=NPROCY;
	perbc[0]=1; perbc[1]=1;
	for (proc=0; proc<num_procs; proc++) {
		whoami(proc,nprocs,perbc,recpnum,nextproc,2);
		proc_list[recpnum[0]][recpnum[1]]=proc;
		if(proc==procnum) {
			x_orig=(float)recpnum[0]/(float)NPROCX;
			y_orig=(float)recpnum[1]/(float)NPROCY;
			proc_map2dx=recpnum[0];
			proc_map2dy=recpnum[1];
		}
	}
	
	i = (proc_map2dx+1)%NPROCX;
	j = (proc_map2dy+1)%NPROCY;
	e_proc = proc_list[i][proc_map2dy];
	n_proc = proc_list[proc_map2dx][j];
	i=(proc_map2dx+NPROCX-1)%NPROCX;
	j=(proc_map2dy+NPROCY-1)%NPROCY;
	w_proc = proc_list[i][proc_map2dy];
	s_proc = proc_list[proc_map2dx][j];

	e_proc = e_proc==procnum ? NULLPROC : e_proc;
	w_proc = w_proc==procnum ? NULLPROC : w_proc;
	n_proc = n_proc==procnum ? NULLPROC : n_proc;
	s_proc = s_proc==procnum ? NULLPROC : s_proc;

	sprintf(string,"e,w,n,s procs %d %d %d %d",e_proc,w_proc,n_proc,s_proc);
	syslog(++nsys,string);
	for(j=0; j<NPROCY; j++){
		for(i=0; i<NPROCX; i++) 
			sprintf(&string[4*i]," %3d",proc_list[i][j]);
		syslog(++nsys,string);
	}	

	for(i=0; i<=d; i++)
		ci[i]=copen(pid);

	recvw(ci[0],0,&npart,sizeof(int),&count,&proc,&rpid);
	sprintf(string,"  particles/proc  = %d",npart);
	nsys++;
	syslog(nsys,string);
	
	n=npart;
	for(i=0; i<npart; i += 1000) {
		len = n>1000 ? 1000*sizeof(float): n*sizeof(float);
		n -= 1000;
		recvw(ci[0],1,&prt_x[i],len,&count,&proc,&rpid);
		recvw(ci[0],1,&prt_y[i],len,&count,&proc,&rpid);
		recvw(ci[0],1,&prt_vx[i],len,&count,&proc,&rpid);
		recvw(ci[0],1,&prt_vy[i],len,&count,&proc,&rpid);
	}

	if(debug>2) {
		for(i=0; i<npart; i++) { 
			sprintf(string,"  x,y: %f %f ",prt_x[i],prt_y[i]);
			syslog(0,string);
		}  
	}
}

float absf(x,y)
float x, *y;
{
	(*y) = x<0. ? -x : x;	
}
mprint(a,nx,ny,test)
float a[];
int nx, ny;
char test[];
{
	int i, j;
	nsys++;
	syslog(nsys,test);
	nx = (nx>11) ? 11 : nx;
	for (j=0; j<=(ny+1); j++){
		for (i=0; i<=(nx+1); i++)
			sprintf(&string[7*i],"%7.3f",a[i*(ny+2)+j]);
		nsys++;
		syslog(nsys,string);
	}
}
abort(string)
char string[];
{
	syslog(0,string);
}


