(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)
(* Created by stolfi on Mon Nov  7 21:25:55 1988               *)
(* Last modified on Wed Jun 17 11:44:43 PDT 1992 by stolfi     *)
(*      modified on Thu Jun 11 19:08:47 1992 by mhb            *)
(*      modified on Tue Feb 11 21:39:50 PST 1992 by muller     *)


(* A "CIE.T" is a color represented in the ``CIE color model''. 
  The CIE color model is a standard system for color notation that
  was developed in 1931 by Comission Internationale de l'Eclairage.
  The model was firmly grounded in psychophysiological experiments
  and color perception theory, and is widely used in science and
  industry, especially for lighting, photography, printing, textiles,
  paints, plastics, ceramics, and many more. *)

INTERFACE CIE;

IMPORT Color, RGB;

TYPE
  T = XYZ;

TYPE XYZ = Color.T; (* ARRAY [0..2] OF REAL; *)
(* A color in CIE 1931 XYZ coordinates. *)

(* Let A, B, C be any three colors such that none of them can be
   matched by a mixture of the other two.  It is an empirical
   fact that any color D can be matched by a mixture of
   appropriate amounts of A, B, C.  This may require a negative
   amount of one or two of the ``primary'' colors, that is, the
   addition of a positive amount of those primaries to the color
   D, in order to make the match possible.

   It is also a fact that the amounts required for the match are
   approximately the same for all observers, except some 5\% who
   are color-blind to some degree.  Therefore, a color D can be
   described by three real numbers, the amounts of three fixed
   primaries that a ``standard'' or ``average'' observer needs in
   order to match D

   The three primary colors chosen for the 1931 CIE system are
   mathematical abstractions that cannot be physically realized,
   because they lie outside the spectrum locus; that is, they
   would have to be purer than 100\% pure (monochromatic) colors.
   The CIE primaries are such that all realizable colors have
   non-negative XYZ coordinates.  Also, the X and Z primaries
   have zero brightness, so the brightness of a color is
   determined by its Y component alone. *)

CONST
  Primary = ARRAY [0..2] OF XYZ{
    XYZ{1.0, 0.0, 0.0}, 
    XYZ{0.0, 1.0, 0.0}, 
    XYZ{0.0, 0.0, 1.0}
  };
    
TYPE Yxy = Color.T; (* ARRAY [0..2] OF REAL; *)
(* A color in CIE 1931 Y x y coordinates. *)

(* In many applications it is convenient to ignore color changes
   that are due to variations in camera aperture, film exposure,
   intensity of illuminating light, and similar factors that
   change all coordinates by the same amount.  For such
   applications, it is customary to use the 1931 CIE Y
   (brightness) coordinate together with the ``chromaticity''
   coordinates x,y given by the formulas

|       x = X / (X+Y+Z)
|       y = Y / (X+Y+Z)

   Note that these transformations are projective but not linear.
   Therefore, in Yxy space a mixture of two colored lights A and
   B in equal amounts lies on the straight line connecting A to
   B, but not at their midpoint. *)


TYPE Yuv = Color.T; (* ARRAY [0..2] OF REAL; *)
(* A color in CIE 1976 Y u' v' coordinates. *)

(* The Yxy model gives a very poor picture of the perceptual
   distances between colors.  In 1976 the CIE proposed another
   ``normalized'' color model slightly better than Yxy in this
   regard.  This model consists of the 1931 CIE Y coordinate
   together with the "new" chromaticity coordinates

|       u' = 4X / (X + 15Y + 3Z)
|       v' = 9Y / (X + 15Y + 3Z)

   Note that this is still a projective (but not affine)
   transformation of XYZ space, so it preserves straight lines
   (but not ratios of distances). *)

PROCEDURE XYZFromYxy (READONLY yxy: Yxy): XYZ;
PROCEDURE YxyFromXYZ (READONLY xyz: XYZ): Yxy;

PROCEDURE XYZFromYuv (READONLY yuv: Yuv): XYZ;
PROCEDURE YuvFromXYZ (READONLY xyz: XYZ): Yuv;

PROCEDURE RGBFromXYZ (READONLY xyz: XYZ): RGB.T;
PROCEDURE XYZFromRGB (READONLY rgb: RGB.T): XYZ;
(* Basic conversions between RGB and CIE coordinates. *)


(* The remainder of the interface defines the "Default" and
   "Undefined" triples, and a procedure for safely
   LOOPHOLE'ing a triple into an array of bytes.  As
   explained in the "Color" interface, "Default" is useful as a
   default value for generic modules, and "Undefined" is useful
   for applications that need a special color value meaning
   ``undefined'', ``vacant'', ``don't care'', and so on.  The
   loophole function, "ToBytes", is useful for marshalling and
   pickling color triples. *)

CONST 
  Default = Color.Default;

  Undefined = Color.Undefined;

TYPE 
  Bytes = Color.Bytes;

CONST (*PROCEDURE*) 
  ToBytes = Color.ToBytes;


END CIE.













