/* utils.c: general utility functions like fprint, ealloc etc. */

#include <stdarg.h>
#include <signal.h>
#include "rc.h"
#include "utils.h"
#include "nalloc.h"
#include "jmp.h"
#include "status.h"
#include "input.h"

extern char nw[]; /* From lex.c; used by strprint to see if it needs to quote a word */
extern int sys_nerr;
extern char *sys_errlist[];

void rc_error(const char *s) {
	if (s != NULL) {
		if (interactive)
			fprint(2,"%s\n",s);
		else
			fprint(2,"line %d: %s\n", lineno, s);
	}
	setfalse();
	breakbuf = retbuf = NULL;
	longjmp(jb, 1);
}

void uerror(char *s) {
	if (errno > sys_nerr)
		return;

	if (s != NULL)
		fprint(2,"%s: %s\n",s,sys_errlist[errno]);
	else
		fprint(2,"%s\n",sys_errlist[errno]);
}

void *ealloc(int n) {
	char *p = malloc(n);

	if (p == NULL) {
		uerror("malloc");
		rc_exit(1);
	}

	return p;
}

void *erealloc(void *p, int n) {
	p = realloc(p, n);

	if (p == NULL) {
		uerror("realloc");
		rc_exit(1);
	}
	return p;
}

void efree(void *p) {
	if (p != NULL)
		free(p);
}

void clear(char *s, int z) {
	while (z != 0)
		s[--z] = 0;
}

void nullify_fifoq() {
	int sp;

	while (fifoq != NULL) {
		unlink(fifoq->w);
		wait(&sp);
		statprint(sp);
		fifoq = fifoq->n;
	}
}

void dprint(va_list ap, char *strbuf, char *f) {
	int i = 0;
	int j = 0;
	int v;
	char *s, **a;
	char num[16];

	while (*f) {
		if (*f == '%') {
			switch (*++f) {
			case 'a':
				a = va_arg(ap, char **);
				if (*a == NULL)
					break;
				strcpy(strbuf + i, *a);
				i += strlen(*a);
				while (*++a != NULL) {
					strbuf[i++] = ' ';
					strcpy(strbuf + i, *a);
					i += strlen(*a);
				}
				break;
			case 'c':
				strbuf[i++] = va_arg(ap, int);
				break;
			case 'd':
				v = va_arg(ap, int);
				j = 0;
				if (v == 0)
					num[j++] = '0';
				while (v) {
					num[j++] = v % 10 + '0';
					v /= 10;
				}
				while (--j >= 0)
					strbuf[i++] = num[j];
				break;
			case 'o':
				v = va_arg(ap, int);
				j = 0;
				if (v == 0)
					num[j++] = '0';
				while (v) {
					num[j++] = (v & 7) + '0';
					v >>= 3;
				}
				while (--j >= 0)
					strbuf[i++] = num[j];
				break;
			case 's':
				s = va_arg(ap, char *);
				while (*s)
					strbuf[i++] = *s++;
				break;
			case '%':
				strbuf[i++] = '%';
				break;
			default:
				strbuf[i++] = '%';
				strbuf[i++] = *f;
			}
		} else
			strbuf[i++] = *f;
		f++;
	}
	strbuf[i] = '\0';
}

void fprint(int fd, char *f,...) {
        va_list ap;
	char str[16384]; /* Just a large string */

	va_start(ap,f);
	dprint(ap, str, f);
	va_end(ap);
	writeall(fd,str,strlen(str));
}

void sprint(char *b, char *f,...) {
	va_list ap;

	va_start(ap, f);
	dprint(ap, b, f);
	va_end(ap);
}

int otoi(char *s) {
	int i = 0;
	while (*s) {
		if (*s > '7' || *s < '0')
			return -1;
		else {
			i <<= 3;
			i += *s - '0';
		}
		s++;
	}
	return i;
}

void sig(int s) {
	if (shell_pid != getpid()) /* child process must quit on ^C */
		rc_exit(1);

	if (s == SIGQUIT || posn == CRITICAL) /* sigquit has to wait for child to dump core */
		return;

	if (posn == TOPLEV)
		fprint(2,"\n");

	posn = TOPLEV;

	/* make sure interactive status is restored after a nested eval */
	popallinput();
	interactive = toplevinteractive;
	atthetop = 1;
	redirq = NULL;
	nullify_fifoq();
	signal(SIGINT, sig);
	breakbuf = retbuf = NULL;
	longjmp(jb, 1);
}

int atou(char *s) {
	int i = 0;

	while (*s) {
		if (*s > '9' || *s < '0')
			return -1;
		else {
			i *= 10;
			i += *s - '0';
		}
		s++;
	}
	return i;
}

/* the last word in portable ansi */

int starstrcmp(const void *s1, const void *s2) {
	return strcmp(*(char **)s1, *(char **)s2);
}

List *sort(List *s) {
	List *t;
	int i,j;
	char **arr;

	if (s == NULL)
		return NULL;
	if (s->n == NULL)
		return s;

	for (i = 0, t = s; t; t = t->n)
		i++;

	arr = nalloc(1 + i * sizeof (char **));

	for (j = 0, t = s; j < i; j++) {
		arr[j] = t->w;
		t = t->n;
	}

	qsort(arr, i, sizeof(char *), starstrcmp);

	for (j = 0, t = s; j < i; j++) {
		t->w = arr[j];
		t = t->n;
	}
	return s;
}

char *strprint(char *s, boolean quotable, boolean metaquote) {
	int i,j;
	char *t;

	if (*s == '\0')
		return "''";

	for (i = 0; s[i] != '\0'; i++)
		if (nw[s[i]] == 1 && (metaquote || (s[i] != '*' && s[i] != '?' && s[i] != '[')))
			quotable = TRUE;

	if (!quotable)
		return s;

	for(i = j = 0; s[i] != '\0'; i++, j++)
		if (s[i] == '\'')
			j++;

	t = nalloc(j + 3);

	t[0] = '\'';

	for (j = 1, i = 0; s[i] != '\0'; i++, j++) {
		t[j] = s[i];
		if (s[i] == '\'')
			t[++j] = '\'';
	}

	t[j++] = '\'';
	t[j] = '\0';

	return t;
}

void writeall(int fd, char *buf, int remain) {
	int i;

	for (i = 0; remain > 0; buf += i, remain -= i)
		i = write(fd, buf, remain);
}
