/* glom.c: builds an argument list out of words, variables, etc. */

#ifndef NONMPIPES
#include <sys/types.h>
#include <sys/stat.h>
#endif
#include "rc.h"
#include "utils.h"
#include "nalloc.h"
#include "glom.h"
#include "hash.h"
#include "walk.h"
#include "status.h"
#include "exec.h"
#include "lex.h"
#include "open.h"
#include "list.h"
#include "jmp.h"

static List *backq(Node *);
static List *bqinput(int);
static List *count(List *);
static List *mknmpipe(Node *);

Rq *redirq = NULL;
List *fifoq = NULL;

List *word(char *w, char *m) {
	List *s;

	if (w == NULL)
		return NULL;
	if (*w == '\'')
		w++; /* m is null, or had better be! */

	s = nnew(List);
	s->w = w;
	s->m = m;
	s->n = NULL;
	return s;
}

List *append(List *s1, List *s2) {
	List *r = nnew(List);
	List *top = r;

	if (s1 == NULL)
		return s2;
	if (s2 == NULL)
		return s1;

	while (s1 != NULL) {
		r->w = s1->w;
		r->m = s1->m;
		r->n = nnew(List);
		s1 = s1->n;
		r = r->n;
	}
	while (1) {
		r->w = s2->w;
		r->m = s2->m;
		s2 = s2->n;
		if (s2 != NULL) {
			r->n = nnew(List);
			r = r->n;
		} else {
			r->n = NULL;
			return top;
		}
	}
}

List *concat(List *s1, List *s2) {
	int n1, n2;
	int y,z;
	List *n, *s;

	if (s1 == NULL)
		return s2;
	if (s2 == NULL)
		return s1;

	n1 = listnel(s1);
	n2 = listnel(s2);

	if (n1 != n2 && n1 != 1 && n2 != 1)
		rc_error("bad concatenation");

	n = s = nnew(List);

	while (1) {
		z = strlen(s1->w) + strlen(s2->w) + 1;
		n->w = nalloc(z);
		strcpy(n->w,s1->w);
		strcat(n->w,s2->w);
		if (s1->m == NULL && s2->m == NULL) {
			n->m = NULL;
		} else {
			n->m = nalloc(z);
			y = strlen(s1->w);
			if (s1->m == NULL)
				clear(n->m, y);
			else
				memcpy(n->m, s1->m, y);
			if (s2->m == NULL)
				clear(n->m + y, strlen(s2->w));
			else
				memcpy(n->m + y, s2->m, strlen(s2->w));
			n->m[z] = 0;
		}
		if (n1 > 1)
			s1 = s1->n;
		if (n2 > 1)
			s2 = s2->n;
		if (s1 == NULL || s2 == NULL || (n1 == 1  && n2 == 1)) {
			n->n = NULL;
			return s;
		}
		n->n = nalloc(sizeof (List));
		n = n->n;
	}
}

List *varsub(List *v, List *subs) {
	int i,j;
	int n;
	List *r,*s;
	List *top,*cat;

	n = listnel(v);

	top = cat = NULL;

	for (s = subs; s != NULL; s = s->n) {
		i = atou(s->w);
		if (i < 1)
			rc_error("bad subscript");
		if (i <= n) {
			for (j = 1, r = v; j != i; j++, r = r->n)
				; /* loop until r == v(i) */
			if (top == NULL) {
				top = cat = nnew(List);
			} else {
				cat->n = nnew(List);
				cat = cat->n;
			}
			cat->w = r->w;
			cat->m = r->m;
		}
	}

	if (top == NULL)
		return NULL;

	cat->n = NULL;
	return top;
}

List *flatten(List *s) {
	List *r = nnew(List);
	int i;
	char *f;

	if (s == NULL)
		return NULL;
	if (s->n == NULL)
		return s;

	i = listlen(s);
	f = nalloc(i + 1);

	strcpy(f,s->w);

	for (s = s->n; s != NULL; s = s->n) {
		strcat(f," ");
		strcat(f,s->w);
	}

	r->w = f;
	r->m = NULL; /* flattened lists come from variables, so no meta */
	r->n = NULL;

	return r;
}

static List *count(List *l) {
	List *s;

	s = nnew(List);
	s->n = NULL;
	s->w = nalloc(16);
	sprint(s->w, "%d", listnel(l));
	s->m = NULL;
	return s;
}

void assign(List *s1, List *s2, boolean stack) {
	if (s1 == NULL)
		rc_error("null variable name");
	if (s1->n != NULL)
		rc_error("multi-word variable name");
	if (*s1->w == '\0')
		rc_error("zero-length variable name");
	if (s2 != NULL || stack) {
		varassign(s1->w, s2, stack);
		alias(s1->w, s2, stack);
	} else
		varrm(s1->w, stack);
}

/*
   The following two functions are by the courtesy of Paul Haahr,
   who could not stand the incompetence of my own backquote implementation.
*/

#define BUFSIZE	1000

static List *bqinput(int fd) {
	char *s, *bufend;
	List *lp, *list, *prev;
	int n, nremain, bufsize;
	static char isifs[256];

	/* this should be cached, and recomputed only when $ifs changes [--pgh] */ {
	/* no, a fork takes much longer than a varlookup --tbr */
		for (n = 0; n < sizeof isifs; n++)
			isifs[n] = 0;
		isifs['\0'] = 1;
		list = varlookup("ifs");
		for (lp = list; list != NULL; list = list->n)
			for (s = list->w; *s != '\0'; s++)
				isifs[*(unsigned char *)s] = 1;
	}

	nremain = bufsize = BUFSIZE;
	lp = list = nalloc(sizeof (List));
	s  = list->w = nalloc(bufsize + 1);
	list->m = NULL;
	prev = NULL;
	do {
		if (nremain == 0) {
			int n;
			char *buf;
			n = s - lp->w;
			while (bufsize < n + BUFSIZE)
				bufsize *= 2;
			buf = nalloc(bufsize + 1);
			memcpy(buf, lp->w, s - lp->w);
			lp->w = buf;
			lp->m = NULL;
			s = &buf[n];
			nremain = bufsize - n;
		}
		n = read(fd, s, nremain);
		nremain -= n;
		for (bufend = &s[n]; s < bufend; s++)
			if (isifs[*(unsigned char *)s]) {
				*s = '\0';
				prev = lp;
				lp = nalloc(sizeof (List));
				lp->w = s + 1;
				lp->m = NULL;
				prev->n = lp;
			}
	} while (n > 0);
	if (n == -1) {
		uerror("backquote read");
		rc_error(NULL);
	}

	if (s == lp->w)
		if (prev == NULL)
			list = NULL;
		else
			prev->n = NULL;
	else {
		lp->n = NULL;
		*s = '\0';
	}

	return list;
}

static List *backq(Node *n) {
	int p[2];
	int pid;
	int sp;
	List *result;

	if (n == NULL)
		return NULL;

	if (pipe(p) < 0) {
		uerror("pipe");
		rc_error(NULL);
	}
	pid = fork();

	if (pid < 0) {
		uerror("fork");
		rc_error(NULL);
	}
	if (pid == 0) {
		setsigdefaults();
		dup2(p[1],1);
		close(p[0]);
		close(p[1]);
		redirq = NULL;
		fifoq = NULL;
		breakbuf = retbuf = NULL;
		walk(n, CHILD);
		rc_exit(getstatus());
		return NULL;	/* prevent warnings */
	}

	close(p[1]);
	result = bqinput(p[0]);
	close(p[0]);
	while (pid != wait(&sp))
		;
	statprint(sp);
	return result;
}

void qredir(Node *n) {
	Rq *next;

	if (redirq == NULL) {
		next = redirq = nnew(Rq);
	} else {
		for (next = redirq; next->n != NULL; next = next->n)
			;
		next->n = nnew(Rq);
		next = next->n;
	}

	next->r = n;
	next->n = NULL;
}

static List *mknmpipe(Node *n) {
#ifdef NONMPIPES
	rc_error("named pipes are not supported");
	return NULL;
#else
	int pid, fd;
	char *fifoname = nalloc(32);
	List *fifolist = nnew(List);
	List *f = nnew(List);
	static int fifonumber = 0;

	sprint(fifoname,"/tmp/rc%d.%d",getpid(),fifonumber++);

	if (mknod(fifoname, S_IFIFO | 0644, 0) < 0) {
		uerror("mknod");
		return NULL;
	}

	pid = fork();

	if (pid < 0) {
		uerror("fork");
		rc_error(NULL);
	}

	if (pid == 0) {
		fd = rc_open(fifoname, CREATE);
		if (fd < 0) {
			uerror("open");
			exit(1);
		}
		if (dup2(fd, (n->u[0].i == FROM)) < 0) { /* stupid hack */
			uerror("dup2");
			exit(1);
		}
		close(fd);
		redirq = NULL;
		breakbuf = retbuf = NULL;
		fifoq = NULL;
		walk(n->u[2].p, CHILD);
	}

	f->w = fifoname;
	f->n = fifoq;
	fifoq = f;

	fifolist->w = fifoname;
	fifolist->m = NULL;
	fifolist->n = NULL;

	return fifolist;
#endif
}

List *glom(Node *n) {
	List *v;
	List *first;

	if (n == NULL)
		return NULL;

	switch (n->type) {
	case ARGS:
	case LAPPEND:
		first = glom(n->u[0].p); /* force left-to-right evaluation */
		return append(first, glom(n->u[1].p));
	case BACKQ:
		return backq(n->u[0].p);
	case CONCAT:
		first = glom(n->u[0].p);
		return concat(first, glom(n->u[1].p));
	case rDUP:
	case rREDIR:
		qredir(n);
		return NULL;
	case rWORD:
		return word(n->u[0].s,n->u[1].s);
	case NMPIPE:
		return mknmpipe(n);
	default:
		break;
	}

	/*
           the next three operations depend on the left-child of glom
	   to be a variable name. Therefore they are all treated here.
	   (previously each function looked up and checked the validity
	   of a variable name)
	*/

	v = glom(n->u[0].p);
	if (v == NULL)
		rc_error("null variable name");
	if (v->n != NULL)
		rc_error("multi-word variable name");
	if (*v->w == '\0')
		rc_error("zero-length variable name");
	v = varlookup(v->w);

	switch (n->type) {
	case rCOUNT:
		return count(v);
	case rFLAT:
		return flatten(v);
	case VAR:
		return v;
	case VARSUB:
		return varsub(v, glom(n->u[1].p));
	default:
		break;
	}

	fprint(2,"glom: unknown node %d this can't happen", n->type);
	return NULL; /*To satisfy gcc -Wall */
}
