/* builtins.c: the collection of rc's builtin commands */

#ifndef NOLIMITS
#include <sys/time.h>
#include <sys/resource.h>
#endif
#include "rc.h"
#include "utils.h"
#include "walk.h"
#include "input.h"
#include "builtins.h"
#include "hash.h"
#include "nalloc.h"
#include "status.h"
#include "footobar.h"
#include "lex.h"
#include "open.h"
#include "jmp.h"

extern int umask(int);

void b_break(char **), b_builtin(char **), b_cd(char **),
	b_echo(char **), b_eval(char **), b_exit(char **),
	b_limit(char **), b_return(char **), b_shift(char **),
	b_umask(char **), b_wait(char **), b_whatis(char **),
	b_dot(char **);

jmp_buf *retbuf = NULL;

builtin_t *builtins[] = {
	b_break, b_builtin, b_cd, b_echo, b_eval, b_exec, b_exit,
	b_limit, b_return, b_shift, b_umask, b_wait, b_whatis, b_dot
};

static char *builtins_str[] = {
	"break", "builtin", "cd", "echo", "eval", "exec", "exit",
	"limit", "return", "shift", "umask", "wait", "whatis", "."
};

builtin_t *isbuiltin(char *s) {
	int i;
	for (i = 0; i < sizeof(builtins_str) / sizeof(char *); i++)
		if (streq(builtins_str[i], s))
			return builtins[i];
	return NULL;
}

void function(char **av) {
	boolean i = interactive;
	jmp_buf j, *oldret, *oldbreak;

	oldret = retbuf;
	oldbreak = breakbuf;
	breakbuf = NULL;
	retbuf = (jmp_buf *) j;
	if (setjmp(j) == 0) {
		interactive = FALSE;
		arrayassign("*", av+1, TRUE);
		walk(fnlookup(*av), PARENT);
	}
	varrm("*", TRUE);
	interactive = i;
	retbuf = oldret;
}

void b_exec(char **av) { /* bogus function which never gets called */
	return;
}

void b_echo(char **av) {
	int i;
	char **a, *e;

	if (*++av == NULL) {
		write(1, "\n", 1);
		return;
	}
	for (i = 0, a = av; *a != NULL; a++)
		i += strlen(*a) + 2;
		/* One for the null terminator, one for the space */
	e = nalloc(i + 2);
	if (streq(*av,"-n"))
		sprint(e,"%a", ++av);
	else if (streq(*av,"--"))
		sprint(e,"%a\n", ++av);
	else
		sprint(e,"%a\n", av);
	writeall(1, e, strlen(e));
	settrue();
}

void b_cd(char **av) {
	List *s, nil;
	char *path = NULL;
	int pathlen = 0, t;

	if (*++av == NULL) {
		s = varlookup("home");
		if (s != NULL)
			*av = s->w;
		else
			*av = "/";
	}
	if (**av == '/' ||
	   (**av == '.' && (*av)[1] == '/') ||
	   (**av == '.' && (*av)[1] == '.' && (*av)[2] =='/')) {
		if (chdir(*av) < 0) {
			setfalse();
			uerror(*av);
		} else
			settrue();
	} else {
		s = varlookup("cdpath");
		if (s == NULL) {
			s = &nil;
			nil.w = "";
			nil.n = NULL;
		}
		do {
			if (s != &nil && *s->w != '\0') {
				t = strlen(*av) + strlen(s->w) + 2;
				if (t > pathlen) {
					pathlen = t;
					path = nalloc(t);
				}
				strcpy(path, s->w);
				strcat(path, "/");
				strcat(path, *av);
			} else {
				pathlen = 0;
				path = *av;
			}
			if (chdir(path) >= 0) {
				settrue();
				if (*s->w != '\0' && !streq(s->w,"."))
					fprint(2,"%s\n",path);
				return;
			}
			s = s->n;
		} while (s != NULL);
		fprint(2,"couldn't cd to %s\n", *av);
		setfalse();
	}
}

void b_umask(char **av) {
	int i;

	if (*++av == NULL) {
		settrue();
		i = umask(0);
		umask(i);
		fprint(2,"0%o\n",i);
	} else {
		i = otoi(*av);
		if (i < 0) {
			setfalse();
			fprint(2,"bad umask\n");
		} else {
			settrue();
			umask(i);
		}
	}
}

void b_exit(char **av) {
	if (av[1] == NULL)
		exit(getstatus());
	if (atou(av[1]) >= 0)
		exit(atou(av[1]));
	fprint(2,"%s is a bad number\n", av[1]);
	exit(1);
}

void b_return(char **av) {
	int stat;

	if (retbuf == NULL)
		rc_error("return outside of function");

	if (av[1] != NULL) {
		stat = atou(av[1]);
		if (stat < 0) {
			fprint(2,"%s is a bad number\n", av[1]);
			setfalse();
		} else {
			setstatus(stat << 8);
		}
	}

	longjmp(retbuf, 1);
}

void b_break(char **av) {
	if (breakbuf == NULL)
		rc_error("break outside of loop");
	longjmp(breakbuf, 1);
}

void b_shift(char **av) {
	int shift;
	List *s;

	if (av[1] == NULL)
		shift = 1;
	else
		shift = atou(av[1]);
	if (shift < 0) {
		fprint(2,"%s is a bad number\n", av[1]);
		setfalse();
		return;
	}
	s = varlookup("*");
	while(s != NULL && shift != 0) {
		s = s->n;
		--shift;
	}
	if (s != NULL)
		varassign("*", s, FALSE);
	else
		varrm("*", FALSE);
	settrue();
}

void b_builtin(char **av) {
	builtin_t *b;

	if (av[1] == NULL) {
		setfalse();
		fprint(2,"no arguments to 'builtin'\n");
		return;
	}
	b = isbuiltin(av[1]);
	if (b == NULL) {
		setfalse();
		fprint(2,"no such builtin\n");
	} else {
		settrue();
		b(++av);
	}
}

void b_wait(char **av) {
	int stat, pid;

	if (av[1] == NULL) {
		pid = wait(&stat);
		if (pid < 0)
			uerror("wait");
		while(wait(&stat) >= 0)
			;
		return;
	} else {
		pid = atou(av[1]);
		if (pid < 0) {
			setfalse();
			fprint(2,"%s is a bad number\n", av[1]);
			return;
		}
		while (pid != wait(&stat))
			if (pid < 0)
				uerror("wait");
	}
	if (pid < 0)
		setfalse();
	else
		setstatus(stat);
}

void b_whatis(char **av) {
	enum bool f,found;
	int i,j;
	List *s,*t;
	Node *n;
	char *e;

	if (av[1] == NULL) {
		whatare_all_vars();
		settrue();
		return;
	}
	found = TRUE;
	for (i = 1; av[i] != NULL; i++) {
		f = FALSE;
		if (s = varlookup(av[i])) {
			f = TRUE;
			if (s->n == NULL)
				fprint(1,"%s=", av[i]);
			else
				fprint(1,"%s=(", av[i]);
			for (t = s; t->n != NULL; t = t->n)
				fprint(1,"%s ",strprint(t->w, FALSE, TRUE));
			if (s->n == NULL)
				fprint(1,"%s\n",strprint(t->w, FALSE, TRUE));
			else
				fprint(1,"%s)\n", strprint(t->w, FALSE, TRUE));
		}
		if (n = fnlookup(av[i])) {
			f = TRUE;
			fprint(1,"fn %s {%s}\n",av[i],ptree(n));
		} else if (isbuiltin(av[i]) != NULL) {
			f = TRUE;
			for (j = 0; j < sizeof(builtins_str) / sizeof(char *); j++)
				if (streq(av[i], builtins_str[j]))
					break;
			fprint(1,"builtin %s\n",builtins_str[j]);
		} else if ((e = which(av[i])) != NULL) {
			f = TRUE;
			fprint(1,"%s\n",e);
		}
		if (!f) {
			found = FALSE;
			fprint(2,"%s not found\n", av[i]);
		}
	}
	if (found)
		settrue();
	else
		setfalse();
}

void b_eval(char **av) {
	int i, t;

	if (av[1] == NULL)
		return;
	i = interactive;
	interactive = FALSE;
	t = atthetop;
	atthetop = 0;
	pushinput(STRING, av + 1);
	doit();
	interactive = i;
	atthetop = t;
}

void b_dot(char **av) {
	int i, fd, t;
	jmp_buf *oldret, *oldbreak;

	if (av[1] == NULL || *av[1] == '\0')
		return;

	i = interactive;
	oldret = retbuf;
	oldbreak = breakbuf;

	fd = rc_open(av[1], FROM);
	if (fd < 0) {
		uerror(av[1]);
		setfalse();
		return;
	}
	arrayassign("*", av+2, TRUE);
	interactive = FALSE;
	t = atthetop;
	atthetop = 0;
	retbuf = breakbuf = NULL;
	pushinput(FD, &fd);
	doit();
	interactive = i;
	t = atthetop;
	retbuf = oldret;
	breakbuf = oldbreak;
	varrm("*", TRUE);
}

/* Berkeley limit support was cleaned up by Paul Haahr. */

#ifdef NOLIMITS
void b_limit(char **av) {
	fprint(2,"rc was compiled without berkeley limits\n");
}
#else

typedef struct Suffix Suffix;
struct Suffix {
	Suffix *next;
	long amount;
	char *name;
};

static Suffix
	kbsuf = { NULL, 1024, "k" },
	mbsuf = { &kbsuf, 1024*1024, "m" },
	gbsuf = { &mbsuf, 1024*1024*1024, "g" },
	stsuf = { NULL, 1, "s" },
	mtsuf = { &stsuf, 60, "m" },
	htsuf = { &mtsuf, 60*60, "h" };
#define	SIZESUF &gbsuf
#define	TIMESUF &htsuf

typedef struct {
	char *name;
	int flag;
	Suffix *suffix;
} Limit;
static Limit limits[] = {
	{ "cputime",		RLIMIT_CPU,	TIMESUF },
	{ "filesize",		RLIMIT_FSIZE,	SIZESUF },
	{ "datasize",		RLIMIT_DATA,	SIZESUF },
	{ "stacksize",		RLIMIT_STACK,	SIZESUF },
	{ "coredumpsize",	RLIMIT_CORE,	SIZESUF },
	{ "memoryuse",		RLIMIT_RSS,	SIZESUF },
	{ NULL, 0, NULL }
};

extern int getrlimit(int, struct rlimit *);
extern int setrlimit(int, struct rlimit *);

void printlimit(Limit *limit, boolean hard) {
	struct rlimit rlim;
	long lim;
	getrlimit(limit->flag, &rlim);
	if (hard)
		lim = rlim.rlim_max;
	else
		lim = rlim.rlim_cur;
	if (lim == RLIM_INFINITY)
		fprint(1, "%s \tunlimited\n", limit->name);
	else {
		Suffix *suf;
		for (suf = limit->suffix; suf != NULL; suf = suf->next)
			if (lim % suf->amount == 0) {
				lim /= suf->amount;
				break;
			}
		fprint(1, "%s \t%d%s\n", limit->name, lim, suf == NULL ? "" : suf->name);
	}
}

long parselimit(Limit *limit, char *s) {
	int len = strlen(s);
	long lim = 1;
	Suffix *suf = limit->suffix;
	if (streq(s, "unlimited"))
		return RLIM_INFINITY;
	if (suf == TIMESUF && strchr(s, ':') != NULL) {
		char *t = strchr(s, ':');
		*t++ = '\0';
		lim = 60 * atou(s) + atou(t);
	} else {
		for (; suf != NULL; suf = suf->next)
			if (streq(suf->name, s + len - strlen(suf->name))) {
				s[len - strlen(suf->name)] = '\0';
				lim *= suf->amount;
				break;
			}
		lim *= atou(s);
	}
	if (lim < 0)
		rc_error("bad limit");
	return lim;
}

void b_limit(char **av) {
	Limit *lp = limits;
	boolean hard = FALSE;

	if (*++av != NULL && streq(*av, "-h")) {
		av++;
		hard = TRUE;
	}

	if (*av == NULL) {
		for (; lp->name != NULL; lp++)
			printlimit(lp, hard);
		return;
	}

	for (;; lp++) {
		if (lp->name == NULL)
			rc_error("no such limit");
		if (streq(*av, lp->name))
			break;
	}

	if (*++av == NULL)
		printlimit(lp, hard);
	else {
		struct rlimit rlim;
		getrlimit(lp->flag, &rlim);
		if (hard)
			rlim.rlim_max = parselimit(lp, *av);
		else
			rlim.rlim_cur = parselimit(lp, *av);
		if (setrlimit(lp->flag, &rlim) == -1)
			uerror("setrlimit");
	}
}
#endif
