(* Top-level reporting: ties static and dynamic basis together, generates
   a report of bindings. *)

(*
$File: Common/TopLevelReport.sml$
$Date: 1993/03/05 14:39:13 $
$Revision: 1.3 $
$Locker: birkedal $
*)

(*$TopLevelReport:
	FUNID SIGID STRID IDENT BASIS INFIX_BASIS STATOBJECT_PROP
	ENVIRONMENTS_PROP MODULE_STATOBJECT MODULE_ENVIRONMENTS
	VAL_PRINT REPORT CRASH TOP_LEVEL_REPORT
 *)

functor TopLevelReport(structure FunId: FUNID
		       structure SigId: SIGID
		       structure StrId: STRID
		       structure Ident: IDENT
		       structure Basis: BASIS

		       structure InfixBasis: INFIX_BASIS
			 sharing type Basis.InfixBasis = InfixBasis.Basis

		       structure StatObjectProp: STATOBJECT_PROP

		       structure EnvironmentsProp: ENVIRONMENTS_PROP
			 sharing type EnvironmentsProp.strid = StrId.strid
			     and type EnvironmentsProp.id = Ident.id
			     and type EnvironmentsProp.TypeScheme
				      = StatObjectProp.TypeScheme

		       structure ModuleStatObject: MODULE_STATOBJECT
			 sharing type ModuleStatObject.Str
				      = EnvironmentsProp.Str

		       structure ModuleEnvironments: MODULE_ENVIRONMENTS
			 sharing type ModuleEnvironments.sigid = SigId.sigid
			     and type ModuleEnvironments.funid = FunId.funid
			     and type ModuleEnvironments.Basis
				      = Basis.StaticBasis
			     and type ModuleEnvironments.Sig
				      = ModuleStatObject.Sig
			     and type ModuleEnvironments.FunSig
				      = ModuleStatObject.FunSig
			     and type ModuleEnvironments.Env
			              = EnvironmentsProp.Env

		       structure ValPrint: VAL_PRINT
			 sharing type ValPrint.DynamicBasis
			   	      = Basis.DynamicBasis
			     and type ValPrint.id = Ident.id
			     and type ValPrint.strid = StrId.strid
			     and type ValPrint.TypeScheme
				      = StatObjectProp.TypeScheme

		       structure Report: REPORT
			 sharing type InfixBasis.Report
				      = EnvironmentsProp.Report
			   	      = ModuleEnvironments.Report
				      = Report.Report

		       structure Crash: CRASH
		      ): TOP_LEVEL_REPORT =
  struct
    type Basis = Basis.Basis
    type Report = Report.Report

    infix //
    val op // = Report.//

   (* We report a top-level environment (a Basis in fact) by traversing
      the static environment. For each item found, we print out a
      sensible dynamic value, looking in the dynamic environment for
      a representation if required. `report' can also be told to ignore
      the dynamic side of things. Note that EnvironmentsProp and
      ModuleEnvironments have a lot of reporting functions already, and
      can happily handle the cases where no dynamic information is needed.
      Oh: we deal with the infix basis here as well. *)

    fun reportVE(render, pathR, VE, bindings) =
      EnvironmentsProp.iterateVE(
	fn (id, tyScheme) =>
	  Report.line("val "
		      ^ Ident.pr_id id
		      ^ (if bindings then " = " ^ render(pathR, id, tyScheme)
			 	     else ""
			)
		      ^ " : "
		      ^ (let
			   val names = StatObjectProp.newTVNames()
			 in
			   StatObjectProp.pr_TypeSchemePRETTY names tyScheme
			 end
			)
		     ),
	VE
      )

    fun reportSig(sigid, Sig) =
      let
	val (_, S) = ModuleStatObject.unSig Sig
      in
	   Report.line("signature " ^ SigId.pr_SigId sigid ^ " =")
	// Report.line "  sig"
	// Report.indent(4, reportStrSTATIC S)
	// Report.line "  end"
      end

   (* I can't explain how I print out functors; run the damn thing and see
      for yourself. *)

    and reportFunSig(funid, funsig') =
      let
	val (_, S, N'S') = ModuleStatObject.unFunSig funsig'
	val (_, S') = ModuleStatObject.unSig N'S'

	val heading = "functor " ^ FunId.pr_FunId funid ^ "("
	val tab = String.size heading - 1
      in
	Report.decorate(heading, reportStrSTATIC S)
	// Report.indent(tab,    Report.line "): sig"
			      // Report.indent(5, reportStrSTATIC S')
			      // Report.line "   end"
			)
      end

    and reportSE(render, pathR, SE, bindings) =
      EnvironmentsProp.iterateSE(
	fn (strId, str) =>
	  Report.line("structure "
		      ^ StrId.pr_StrId strId
		      ^ (if bindings then " =" else " :")
		     )
	  // Report.line(if bindings then "  struct" else "  sig")
	  // Report.indent(4, reportStr(render, strId :: pathR, str, bindings))
	  // Report.line "  end",
	SE
      )

    and reportStr(render, pathR, str, bindings) =
      case EnvironmentsProp.unStr str
	of (_, env) => reportEnv(render, pathR, env, bindings)

    and reportStrSTATIC str =
      reportStr(fn _ => Crash.impossible "TopLevelReport.reportStaticStr",
		nil, str, false
	       )

    and reportEnv(render, pathR, env, bindings) =
      let
	val (SE, TE, VE, EE) = EnvironmentsProp.unEnv env
      in
	reportSE(render, pathR, SE, bindings)
	// EnvironmentsProp.reportTE{tyEnv=TE, bindings=bindings}
	// EnvironmentsProp.reportEE EE
	// reportVE(render, pathR, VE, bindings)
      end

    fun reportStaticBasis(render, sb: Basis.StaticBasis, bindings: bool)
          : Report =
      let
	val funenv = ModuleEnvironments.F_of_B sb
	val sigenv = ModuleEnvironments.G_of_B sb
	val env = ModuleEnvironments.E_of_B sb
      in		(* Sigs first; looks better (though in fact SML's
			   top-level syntax is knobbled so they can't be
			   mixed). *)
	ModuleEnvironments.reportSigEnv(reportSig, sigenv)
	// ModuleEnvironments.reportFunEnv(reportFunSig, funenv)
	// reportEnv(render, nil, env, bindings)
      end

    fun report{basis, bindings} =
      let
	val ib = Basis.Inf_of_B basis
	val sb = Basis.Stat_of_B basis
	val db = Basis.Dyn_of_B basis	(* Might be void (ELAB_ONLY) *)

	fun render(pathR, id, tyScheme) =
	  ValPrint.print(ValPrint.locate(db, rev pathR, id), tyScheme)
      in
	Report.decorate("> ", InfixBasis.reportBasis ib
			      // reportStaticBasis(render, sb, bindings)
		       )
      end
  end;
