(* The propagated objects of the static semantics - Definition pages 16-22 *)

(*
$File: Common/ENVIRONMENTS_PROP.sml $
$Date: 1993/03/05 14:38:14 $
$Revision: 1.24 $
$Locker: birkedal $
*)

(*$ENVIRONMENTS_PROP*)

signature ENVIRONMENTS_PROP = 
  sig
    (********
    Structure Names
    ********)

    eqtype StrName
    val freshStrName : unit -> StrName
    val bogus_StrName: StrName

    (********
    Type Names
    ********)

    type TyName

    (********
    Type name sets
    ********)

    type TyNameSet

    (********
    Structure Name sets
    ********)

    type StrNameSet 
    val emptyStrNameSet : StrNameSet
    val singleM : StrName -> StrNameSet
    val StrNameSetUnion : StrNameSet * StrNameSet -> StrNameSet
    val StrNameSetFold : (StrName * 'a -> 'a) -> 'a -> StrNameSet -> 'a

    (********
    Name sets
    ********)

    type NameSet
    val emptyN           : NameSet
    and isIn_StrName     : StrName * NameSet -> bool
    and isIn_TyName      : TyName  * NameSet -> bool
    and NameSetUnion     : NameSet * NameSet -> NameSet
    and NameSetIntersect : NameSet * NameSet -> NameSet
    and NameSetMinus     : NameSet * NameSet -> NameSet
    and mkNameSet_Str    : StrNameSet -> NameSet
    and mkNameSet        : StrNameSet * TyNameSet -> NameSet
    and unNameSet        : NameSet -> StrNameSet * TyNameSet 
    and T_of_N           : NameSet -> TyNameSet
    and eqNameSet        : NameSet * NameSet -> bool

    (********
    Environments
    ********)

   (* TVResult is in a structure because that's the only way we can get it
      into MODULE_ENVIRONMENTS as well without a fully-fledged `open'. *)

    eqtype TyVar
    structure TVResult:
      sig
	datatype T = IMP_OK
		   | FAULT of TyVar list
      end

    val tvUnion: TVResult.T * TVResult.T -> TVResult.T

    type Env
    val impTyVarsE : Env -> TVResult.T
    and E_plus_E : Env * Env -> Env
    and namesE   : Env -> NameSet
    and initialE : Env
    and emptyE   : Env

    val bogus_Env: Env

    (********
    Types and TypeSchemes
    ********)

    type Type and TypeScheme

    (********
    Constructors
    ********)

    eqtype con

    (********
    Constructor environments
    ********)

    type ConEnv
    val emptyCE    : ConEnv
    and domCE      : ConEnv -> con list
    and isemptyCE  : ConEnv -> bool
    and singleCE   : con * TypeScheme -> ConEnv
    and equalCE    : ConEnv * ConEnv -> bool (* up to bound vars *)
    and CEmap      : (TypeScheme -> TypeScheme) -> ConEnv -> ConEnv
    and CEfold     : ((TypeScheme * 'a) -> 'a) -> 'a -> ConEnv -> 'a
    and CE_plus_CE : ConEnv * ConEnv -> ConEnv

    (********
    The `Clos' function for constructor environments
    ********)

    val ClosCE : ConEnv -> ConEnv

    (********
    Unqualified identifiers
    ********)

    type id

    (********
    Type functions
    ********)

    eqtype TypeFcn

    (********
    Type structures
    ********)

    eqtype TyStr	(* eqtype needed for == against `bogus_TyStr'. *)
    val mkTyStr      : TypeFcn * ConEnv -> TyStr
    val unTyStr      : TyStr -> TypeFcn * ConEnv
    val TyStr_shares : TyStr * TyStr -> bool
    val Theta_of     : TyStr -> TypeFcn
    val isTyName     : TyStr -> bool
    val CE_of_TyStr  : TyStr -> ConEnv

    val bogus_TyStr: TyStr

    (********
    Structure identifier
    ********)

    type strid

    (********
    Structures
    ********)

    type Str
    val mkStr  : StrName * Env -> Str
    and unStr  : Str -> StrName * Env
    and namesS : Str -> NameSet  
    and lookupE_strid : Env * strid -> Str Option

    val bogus_Str: Str

    (********
    The type of items associated with identifiers in a VarEnv
    ********)

    datatype VarEnvRange =
       LONGVAR   of TypeScheme
     | LONGCON   of TypeScheme
     | LONGEXCON of Type

    (********
    Variable environments
    ********)

    type VarEnv

    val emptyVE  : VarEnv
    and VE_in_E  : VarEnv -> Env
    and ClosVE   : VarEnv -> VarEnv
    and lookupVE : VarEnv  * id -> VarEnvRange Option
    and VEfold   : ((VarEnvRange * 'a) -> 'a) -> 'a -> VarEnv -> 'a
    and VEFold   : (((id * VarEnvRange) * 'b) -> 'b) -> 'b -> VarEnv -> 'b
    and VE_plus_VE  : VarEnv * VarEnv -> VarEnv
    and VEdom       : VarEnv -> id EqSet.Set
    and singleVarVE : id * TypeScheme -> VarEnv

    (********
    The `Clos' function for constructor environments
    *********
    This function converts the resulting CE to a VE
    ********)

    val ClosCE_to_VE : ConEnv -> VarEnv

    (********
    Exception constructors
    ********)

    type excon

    (********
    Exception constructor environments
    ********)

    type ExConEnv

    val emptyEE  : ExConEnv
    and EEfold   : ((Type * 'a) -> 'a) -> 'a -> ExConEnv -> 'a
    and EEFold   : (((excon * Type) * 'a) -> 'a) -> 'a -> ExConEnv -> 'a
    and singleEE : excon * Type -> ExConEnv
    and lookupEE : ExConEnv * excon -> Type Option 
    and VE_of_EE : ExConEnv -> VarEnv
    and EEdom      : ExConEnv -> excon EqSet.Set
    and EE_plus_EE : ExConEnv * ExConEnv -> ExConEnv
    and VE_and_EE_in_E : (VarEnv * ExConEnv) -> Env

    (********
    Type constructors
    ********)

    eqtype tycon

    (********
    Type environments
    ********)

    type TyEnv

    val emptyTE    : TyEnv
    and TE_in_E    : TyEnv -> Env
    and TE_plus_TE : TyEnv * TyEnv -> TyEnv
    and singleTE   : tycon * TyStr -> TyEnv
    and lookupTE   : TyEnv * tycon -> TyStr Option
    and VE_and_TE_in_E : VarEnv * TyEnv -> Env

    val TEfold : ((TyStr * 'b) -> 'b) -> 'b -> TyEnv -> 'b
    and TEFold : (((tycon * TyStr) * 'c) -> 'c) -> 'c -> TyEnv -> 'c
    and TEdom  : TyEnv -> tycon list

    val lookupE_tycon : Env * tycon -> TyStr Option

    (********
    Structure environments
    ********)

    type StrEnv
    val emptySE    : StrEnv
    val singleSE   : strid  * Str    -> StrEnv
    val SE_plus_SE : StrEnv * StrEnv -> StrEnv
    val SE_in_E    : StrEnv -> Env
    val SEfold     : (Str * 'b  -> 'b) -> 'b -> StrEnv -> 'b
    val SEFold     : (((strid * Str) * 'c) -> 'c) -> 'c -> StrEnv -> 'c
    val SEdom      : StrEnv -> strid list
    val SEmap      : (Str -> Str) -> StrEnv -> StrEnv
    val mkEnv      : StrEnv * TyEnv * VarEnv * ExConEnv -> Env
    val unEnv      : Env -> StrEnv * TyEnv * VarEnv * ExConEnv
    val lookupSE   : StrEnv * strid -> Str Option

    (********
    Static contexts
    ********)

    type Context
    val mkC : TyNameSet * Env -> Context
    and C_cplus_TE : Context * TyEnv -> Context
    and Lookup_tycon : Context * tycon -> TyStr Option

    (********
    Type realisations
    ********)

    type tyrea
    val tyrea_on_CE    : tyrea -> ConEnv   -> ConEnv
    and tyrea_on_TyStr : tyrea -> TyStr    -> TyStr
    and tyrea_on_TE    : tyrea -> TyEnv    -> TyEnv
    and tyrea_on_VE    : tyrea -> VarEnv   -> VarEnv
    and tyrea_on_EE    : tyrea -> ExConEnv -> ExConEnv

    (********
    Function which find all flexible datatype names in a type environment.
    *********
    Any TyNames which already have their equality attribute set are ignored.
    ********)

    val flexible_tynames : TyEnv -> TyNameSet

    (********
    Maximising of equality
    ********)

    val maximise_TE_equality : TyNameSet * TyEnv -> TyNameSet

    (********
     Syntactic type variables, and 
     function which finds the set of type variables in a ty abstract 
     syntax tree 
     ********)

    type SyntaxTyVar
     and ty
    val syntaxtyvarsTy : ty -> SyntaxTyVar EqSet.Set
     

    (********
    PrettyPrinting functions
    ********)

    type StringTree
    val layoutNameSet : NameSet -> StringTree
    and layoutStr     : Str     -> StringTree
    and layoutEnv     : Env     -> StringTree

   (* The following are needed for the top-level printout routines. *)

    type Report
    val reportTE: {tyEnv: TyEnv, bindings: bool} -> Report
    val reportEE: ExConEnv -> Report
    val iterateSE: (strid * Str -> Report) * StrEnv -> Report
    val iterateVE: (id * TypeScheme -> Report) * VarEnv -> Report
  end;
