#ifndef FLOATVEC_H
#define FLOATVEC_H
#pragma once

/*
 *	Declarations for Float Precision Vectors
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)FloatVec.h	2.2	9/18/89
 */

/*	This code is designed to be as compatible as possible with the
 *	NIH Vector classes, while preserving efficiency.  These Vectors
 *	are NOT based on the NIH "Object" class, making them much
 *	smaller.  They also implement reference counting, making them
 *	faster. 
 */

#include "vdefs.h"
#include <math.h>

class istream;
class ostream;
class FloatVec;
class DoubleVec;

class FloatBlock {
  unsigned short  	refs;		// Number of references
  unsigned  		npts;		// Number of elements
  float    		array[1];	// The data
  friend		FloatVec;
public:
  FloatBlock(unsigned n);
  FloatBlock(unsigned n, float val);
  FloatBlock(unsigned n, float val, float by);
  ~FloatBlock();

  void			add_reference()	{refs++;}
  unsigned		references()	{return refs;}
  float*		data()		{return array;}
};

class FloatVec {
  FloatBlock*		block;
  float*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine; // For printing
  FloatVec(FloatVec&, int, unsigned, int); // For slices
protected:
  void			boundsErr(int);
  void			boundsCheck(int);
  void			lengthErr(int);
  void			lengthCheck(int i)	{if(npts!=i) lengthErr(i);}
  void			emptyErr(const char* fname);
  void			sliceErr(unsigned, int, unsigned, int);
public:
  FloatVec();
  FloatVec(unsigned n);
  FloatVec(unsigned n, float val);
  FloatVec(unsigned n, float val, float by);
  FloatVec(const FloatVec& a);
  FloatVec(const float* dat, unsigned n);  // Copy of dat will be made
  ~FloatVec();

  // Conversion:
  FloatVec(const DoubleVec& a);		// Conversion from DoubleVec
  operator		DoubleVec();	// Conversion to DoubleVec

  FloatVec		slice(int start, unsigned lgt, int strider=1);
  
  float*		data()		{return begin;}
  unsigned		length()	{return npts;}
  int			stride()	{return step;}

  FloatVec&		reference(FloatVec& v);	// Reference self to v
  FloatVec		deepCopy();	// copy of self with distinct instance variables 
  FloatVec		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Insures only 1 reference to data
  void			resize(unsigned);	// Will pad with zeroes if necessary

  void			scanFrom(istream& s); // Read to eof or delimit with []
  void			printOn(ostream& s);  // Pretty print
  void			setFormatting(int);   // Change # items per line
  void			readFrom(istream&);   // Internal ASCII formatting
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);// Internal binary formatting
  void			storeOn(fileDescTy&);
  
  // Indexing:
  float&		operator[](int i);	// With bounds checking
  float&		operator()(int i);	// With optional bounds checking
  
  // Assignment:
  FloatVec&		operator=(const FloatVec& v); // Must be same length as v
  FloatVec&		operator=(float);
  
  // Arithmetic operators:
  FloatVec&		operator++();
  FloatVec&		operator--();
  FloatVec&		operator+=(const FloatVec&);
  FloatVec&		operator+=(float);
  FloatVec&		operator-=(const FloatVec&);
  FloatVec&		operator-=(float);
  FloatVec&		operator*=(const FloatVec&);
  FloatVec&		operator*=(float);
  FloatVec&		operator/=(const FloatVec&);
  FloatVec&		operator/=(float);
  
  // Friendly arithmetic operators:
  friend FloatVec	operator-(const FloatVec&);
  friend FloatVec	operator+(const FloatVec&);
  friend FloatVec	operator*(const FloatVec&,const FloatVec&);
  friend FloatVec	operator/(const FloatVec&,const FloatVec&);
  friend FloatVec	operator+(const FloatVec&,const FloatVec&);
  friend FloatVec	operator-(const FloatVec&,const FloatVec&);
  friend FloatVec	operator*(const FloatVec&,float);
  friend FloatVec	operator*(float,const FloatVec&);
  friend FloatVec	operator/(const FloatVec&,float);
  friend FloatVec	operator/(float,const FloatVec&);
  friend FloatVec	operator+(const FloatVec&,float);
  friend FloatVec	operator+(float,const FloatVec&);
  friend FloatVec	operator-(const FloatVec&,float);
  friend FloatVec	operator-(float,const FloatVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  // Math functions:
  FloatVec	apply(mathFunTy);
  friend	FloatVec	abs(const FloatVec&);
  friend	FloatVec	acos(const FloatVec&);
  friend	FloatVec	asin(const FloatVec&);
  friend	FloatVec	atan(const FloatVec&);
  friend	FloatVec	atan2(const FloatVec&,const FloatVec&);
  friend	FloatVec	ceil(const FloatVec&);
  friend	FloatVec	cos(const FloatVec&);
  friend	FloatVec	cosh(const FloatVec&);
  friend	FloatVec	cumsum(const FloatVec&);
  friend	FloatVec	delta(const FloatVec&);
  friend	float		dot(const FloatVec&,const FloatVec&);
  friend	FloatVec	exp(const FloatVec&); 
  friend	FloatVec	floor(const FloatVec&);
  friend	FloatVec	log(const FloatVec&);
  friend	int		max(const FloatVec&);
  friend	int		min(const FloatVec&);
  friend	float		mean(const FloatVec&);
  friend	float		prod(const FloatVec&);
  friend	FloatVec	pow(const FloatVec&,const FloatVec&);
  friend	FloatVec	reverse(const FloatVec&);
  friend	FloatVec	rint(const FloatVec&);
  friend	FloatVec	sin(const FloatVec&);
  friend	FloatVec	sinh(const FloatVec&);
  friend	FloatVec	sqrt(const FloatVec&);
  friend	float		sum(const FloatVec&);
  friend	FloatVec	tan(const FloatVec&);
  friend	FloatVec	tanh(const FloatVec&);
  friend	float		variance(const FloatVec&);
#endif
  
};

// Other (related) declarations:
FloatVec	expandEven(const FloatVec&);
FloatVec	expandOdd(const FloatVec&);
ostream&	operator<<(ostream&, const FloatVec&);
istream&	operator>>(istream&, FloatVec&);

/******************* I N L I N E S **************************/

Inline void	FloatVec::setFormatting(int i){numberPerLine = i;}

Inline void	FloatVec::boundsCheck(int i){
  if(i<0 || i>npts) boundsErr(i);
}
Inline float&	FloatVec::operator[](int i){
  boundsCheck(i); return begin[i*step];
}
Inline float&	FloatVec::operator()(int i) {
#if BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

Inline FloatVec	operator+(const FloatVec& a)		{return a;}
Inline FloatVec	operator*(float a, const FloatVec& b)	{return b*a;}
Inline FloatVec	operator+(float a, const FloatVec& b)	{return b+a;}

#ifndef NO_VECTOR_MATHFUN
Inline FloatVec acos(const FloatVec& V)	{ return V.apply(::acos); }
Inline FloatVec asin(const FloatVec& V)	{ return V.apply(::asin); }
Inline FloatVec atan(const FloatVec& V)	{ return V.apply(::atan); }
Inline FloatVec ceil(const FloatVec& V)	{ return V.apply(::ceil); }
Inline FloatVec cos(const FloatVec& V)	{ return V.apply(::cos); }
Inline FloatVec cosh(const FloatVec& V)	{ return V.apply(::cosh); }
Inline FloatVec exp(const FloatVec& V)	{ return V.apply(::exp); }
Inline FloatVec floor(const FloatVec& V){ return V.apply(::floor); }
Inline FloatVec log(const FloatVec& V)	{ return V.apply(::log); }
Inline FloatVec rint(const FloatVec& V)	{ return V.apply(::rint); }
Inline FloatVec sin(const FloatVec& V)	{ return V.apply(::sin); }
Inline FloatVec sinh(const FloatVec& V)	{ return V.apply(::sinh); }
Inline FloatVec sqrt(const FloatVec& V)	{ return V.apply(::sqrt); }
Inline FloatVec tan(const FloatVec& V)	{ return V.apply(::tan); }
Inline FloatVec tanh(const FloatVec& V)	{ return V.apply(::tanh); }
Inline float mean(const FloatVec& V)	{ return sum(V)/V.length(); }
#endif
 
#endif
