/*******************************************************************************
+
+  LEDA  2.2.0                                                 03-05-1992
+
+
+  _graph_edit.c
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#include <LEDA/graph_edit.h>
#include <LEDA/d_array.h>


declare2(d_array,point,node)
declare(node_array,node)

static window*           Wp;
static GRAPH(point,int)* Gp;

static color node_color = blue;
static color edge_color = red;
static bool  directed;

static void draw_node_cursor(double x, double y)
{ Wp->draw_node(x,y); }


static void draw_node(node v)
{ Wp->draw_text_node((*Gp)[v],string("%d",index(v)),node_color); }


static void draw_edge(edge e)
{ if (directed) 
     Wp->draw_edge_arrow((*Gp)[source(e)],(*Gp)[target(e)],edge_color);
  else 
     Wp->draw_edge((*Gp)[source(e)],(*Gp)[target(e)],edge_color); 
 }

static void message(window& W, string s)
{ s += "                              ";
  W.del_message();
  W.message(s);
}


static void read_graph(window& W, GRAPH(point,int)& G, d_array(point,node)&  V,
                       string s, bool clear = false)
{
  if (s=="") 
  { message(W,"No file.");
    return;
   }

  message(W,string("Reading file %s.",~s));

  GRAPH(point,int) X;
  node v;
  edge e;

  int x = X.read(s);

  if (x == 1) 
  { message(W,string("Cannot open file %s.",~s));
    return;
   }

  if (x == 2)
  { message(W,"File is not written by graph_edit (random embedding).");
    int max = int(W.xmax())-3;
    int min = 2;
    forall_nodes(v,X)
      X[v] = point(random(min,max), random(min,max));
   }

  if (clear) { G.clear(); W.clear(); }

  node_array(node) corr(X);

  forall_nodes(v,X) 
  { node u = G.new_node(X[v]);
    corr[v] = u;
    V[G[u]] = u;
    draw_node(u);
   }

  forall_edges(e,X) 
  { G.new_edge(corr[source(e)],corr[target(e)]);
    draw_edge(e);
   }
}


static void save_graph(window& W, GRAPH(point,int)& G,string s)
{ if (s=="") 
  { message(W,"Cannot open file.");
    return;
   }
  message(W,string("writing to file %s",~s));
  G.write(s);
 }


static void window_init(window& W, GRAPH(point,int)& G, int grid_size)
{ 
  W.init(0,grid_size,0,1);

  node v;
  forall_nodes(v,G) draw_node(v);

  edge e;
  forall_edges(e,G) draw_edge(e);

}



void graph_edit(window& W, GRAPH(point,int)& G, bool dir, bool redraw)
{
  d_array(point,node)   V(nil);
  int   grid_size  =  50;
  double  x,y;
  point p,q;
  int   key;
  node  v;

  forall_nodes(v,G) V[G[v]] = v;

  Wp = &W;
  Gp = &G;

  directed = dir;


  panel help_panel("GRAPH EDIT OPERATIONS");

  help_panel.text_item("                                                   ");
  help_panel.text_item("              LEFT            MIDDLE       RIGHT   ");
  help_panel.text_item("                                                   ");
  help_panel.text_item("         insert/move node   insert edge    exit    ");
  help_panel.text_item("(shift)    delete node      delete edge    file    ");
  help_panel.text_item("(ctrl )       clear          settings      help    ");
  help_panel.text_item("                                                   ");
  help_panel.button("continue");


  panel init_panel("SETTINGS");

  init_panel.int_item("grid size",grid_size);
  init_panel.color_item("node color",node_color);
  init_panel.color_item("edge color",edge_color);
  init_panel.button("continue");

  panel file_panel("FILE PANEL");

  string filename;

  file_panel.string_item("file name",filename);
  file_panel.button("read");
  file_panel.button("save"); 
  file_panel.button("load");
  file_panel.button("cancel"); 

  if (redraw) window_init(W,G,grid_size);

  W.message("GRAPH EDIT"); 
  W.message("(press <ctrl> right button for help) "); 

  drawing_mode save = W.set_mode(xor_mode);

  bool done = false;

  while ( ! done )
  {
     key = W.read_mouse(x,y);

     W.del_message();

     p = point(x,y);

     switch(key) {

     case 1:  { 
                node v;

                if (V[p] == nil)        // new node
                { v  = G.new_node(p);
                  draw_node(v);
                  V[p] = v;
                 }
                else                    // move node
                { v = V[p];

                  draw_node(v);

                  W.read_mouse_action(draw_node_cursor,x,y);

                  point q(x,y);           // new position

                  if (V[q] != nil)        // position not free
                  { draw_node(v);
                    break;
                   }

                  V[p] = nil;

                  edge e;

                  forall_edges(e,G) 
                    if (source(e) == v || target(e) == v) draw_edge(e);

                  G[v] = q;
                  V[q] = v;
                  draw_node(v);

                  forall_edges(e,G) 
                    if (source(e) == v || target(e) == v) draw_edge(e);

                 }
                break;
               }


     case 2:  { // new edge
                int k;
                node v = V[p];
                node w;
 
                if (v != nil)            
                { W.draw_filled_node(p);
                  do 
                  { k = W.read_mouse_seg(p.xcoord(),p.ycoord(),x,y);
                    point q(x,y);
                    w = V[q];
                    if (w == nil) 
                    { w  = G.new_node(q);    // new node
                      draw_node(w);
                      V[q] = w;
                     }

                    draw_edge(G.new_edge(v,w));

                   } while ( w == nil && k == 2);
  
                  W.draw_filled_node(p);
                 }

                break;
               }



     case  3: // return
              W.set_mode(save);
              done = true;
              break;



     // Shift + mouse key


     case -1: { // delete node

                node v = V[p];

                edge e;
                if (v != nil) 
                { forall_edges(e,G) 
                    if (source(e) == v || target(e) == v) draw_edge(e);

                  draw_node(v);

                  G.del_node(v);
                  V[p] = nil;
                 }

                }

     case -2: { // delete edge

                node v = V[p];

                if (v != nil) 
                { W.read_mouse_seg(p.xcoord(),p.ycoord(),x,y);
                  q = point(x,y);
                  node w = V[q];
                  if (w != nil)
                  { //linear search
                    edge e = G.first_adj_edge(v);
                    while (e != nil && target(e) != w) e = G.adj_succ(e);
                    if (e != nil)
                    { draw_edge(e);
                      G.del_edge(e);
                     }
                   }
                 }
                break;
               }


     case  -3: { // file menu 

                switch(file_panel.open()) {

                   case 0 : // read
                            read_graph(W,G,V,filename,false);
                            break;
  
                   case 1 : // save
                            save_graph(W,G,filename);
                            break;

                   case 2 : // load 
                            read_graph(W,G,V,filename,true);
                            break;
  
                 }

                break;
              }



     // ctrl + mouse key


     case 4:  // clear
              G.clear();
              window_init(W,G,grid_size);
              break;

     case 5:  // settings
              init_panel.open();
              window_init(W,G,grid_size);
              break;

     case 6:  // help
              help_panel.open();
              break;
 

    } // switch

  } // for(;;)

  W.set_mode(save);

}
