/*
 * Copyright (c) 1991 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 *
 * Routines to package up drawing messages to be sent to the device driver.
 * These routines are the low level graphics driver routines, and should
 * not be called by clients.  Clients must call the GrXXXX routines instead
 * of these GdXXXX routines (or the GsXXXX routines too).
 */
#include "graph_serv.h"
#include <minix/graph_msg.h>
#include <fcntl.h>


/*
 * The device name for the graphics device.
 */
#define	DEVICENAME	"/dev/graphics"


/*
 * The following definition is useful for debugging.
 * Making it nonzero disables actual writing to the graphics device,
 * thus preserving the screen and allowing you to see printfs that
 * your program is doing.
 */
#define	DEBUG	0

#define	NULL	0


static char 	*lastmsg;		/* most recent message */
static char	*newmsg;		/* place to put new message */
static int	msgavail;		/* space left in current message */
static GR_BOOL	error;			/* error occurred from functions */
static int	fd;			/* file descriptor for device */
static char	msgbuf[GR_MAX_SIZEOF_MSG];	/* message buffer being filled */


/*
 * Prepare the graphics device for use, and initialize the message buffer.
 * The parameters are the requested number of rows, columns, and colors
 * to be used.  If they are nonzero, then an error occurs if the requested
 * configuration cannot be met by the hardware.  If any parameter value is
 * zero, then it is a "free" parameter.   The device driver can then select
 * the best value for the parameter consistent with the other parameters.
 * This best value is usually the one giving the highest possible resolution.
 * Returns zero if successful, -1 on failure.
 */
int
GdOpenScreen(rows, cols, colors)
	GR_SIZE	rows;			/* desired number of rows */
	GR_SIZE	cols;			/* desired number of columns */
	long	colors;			/* desired number of colors */
{
	GR_MSG_INIT	init;		/* initialization message */

#if !DEBUG
	fd = open(DEVICENAME, O_RDWR);
	if (fd < 0)
		return -1;
#endif

	lastmsg = msgbuf;
	newmsg = msgbuf;
	msgavail = sizeof(msgbuf);
	((GR_MSG_HEADER *) lastmsg)->func = GR_FUNC_NOP;

	/*
	 * Send the initial configuration message requesting the
	 * desired parameters of the display.
	 */
	init.head.func = GR_FUNC_INIT;
	init.head.flags = 0;
	init.head.count = 0;
	init.magic = GR_INIT_MAGIC;
	init.rows = rows;
	init.cols = cols;
	init.colors = colors;
#if !DEBUG
	if (write(fd, &init, sizeof(init)) != sizeof(init)) {
		close(fd);
		fd = -1;
		return -1;
	}
#endif
	error = GR_FALSE;
	return 0;
}


/*
 * Close the graphics device, first flushing any waiting messages.
 */
void
GdCloseScreen()
{
	if (fd <= 0)
		return;
	GdTextMode();
	GdFlush();
	if (close(fd) < 0)
		error = GR_TRUE;
	fd = -1;
}


/*
 * Check to see if there was a device error.
 * Clears the error if indicated.
 */
GR_BOOL
GdCheckError(clearflag)
	GR_BOOL		clearflag;	/* TRUE to clear error */
{
	if (!error)
		return GR_FALSE;
	if (clearflag)
		error = GR_FALSE;
	return GR_TRUE;
}


/*
 * Flush the message buffer of any messages it may contain.
 */
void
GdFlush()
{
	int	cc;

	if (error)
		return;

	cc = sizeof(msgbuf) - msgavail;
	if (cc <= 0)
		return;
#if !DEBUG
	if (write(fd, msgbuf, cc) != cc)
		error = GR_TRUE;
#endif
	lastmsg = msgbuf;
	newmsg = msgbuf;
	msgavail = sizeof(msgbuf);
	((GR_MSG_HEADER *) lastmsg)->func = GR_FUNC_NOP;
}


/*
 * Allocate space for a new message of the specified size,
 * flushing the buffer if necessary to make room.  Returns
 * a pointer to the header of the new message.
 */
static GR_MSG_HEADER *
GdNewMessage(size)
	GR_SIZE		size;		/* required size of message */
{
	GR_MSG_HEADER *	mp;		/* message header */

	if (msgavail < size)
		GdFlush();

	mp = (GR_MSG_HEADER *) newmsg;
	mp->func = GR_FUNC_NOP;
	mp->flags = 0;
	mp->count = 1;
	lastmsg = newmsg;
	newmsg += size;
	msgavail -= size;
	return mp;
}


/*
 * Attempt to grow an existing message by the specified size.
 * In order to grow, the last message must be the indicated function,
 * its flags must match the indicated flags, and the available size
 * must me at least as large as the indicated size.  If the message
 * cannot be grown, then a new message is allocated with the specified
 * minimum size.  The function code, flags, and count fields of the
 * message are initialized here.
 */
static GR_MSG_HEADER *
GdGrowMessage(func, flags, origsize, growsize)
	GR_FUNC		func;		/* required function */
	GR_FLAGS	flags;		/* required flags */
	GR_SIZE		origsize;	/* space for original message */
	GR_SIZE		growsize;	/* required new space */
{
	GR_MSG_HEADER *	mp;		/* message header */

	mp = (GR_MSG_HEADER *) lastmsg;
	if (growsize && (mp->func == func) && (mp->flags == flags) &&
		(growsize <= msgavail) && (mp->count < GR_MAX_COUNT))
	{
		newmsg += growsize;
		msgavail -= growsize;
		mp->count++;
		return mp;
	}

	mp = GdNewMessage(origsize);
	mp->func = func;
	mp->flags = flags;
	return mp;
}


/*
 * Terminate graphics mode, but keep the graphics device open.
 */
void
GdTextMode()
{
	GR_MSG_TERM	*mp;

	mp = (GR_MSG_TERM *) GdNewMessage(sizeof(GR_MSG_TERM));
	mp->head.func = GR_FUNC_TERM;
}


/*
 * Return information about the screen for clients to use.
 * This does a write to the device, then a read to get the results.
 */
void
GdGetScreenInfo(sip)
	GR_SCREEN_INFO	*sip;		/* pointer to screen info */
{
	GR_MSG_GETSCREENINFO	*mp;	/* message to send */
	int			cc;	/* number of bytes read */

	mp = (GR_MSG_GETSCREENINFO *)
		GdNewMessage(sizeof(GR_MSG_GETSCREENINFO));
	mp->head.func = GR_FUNC_GETSCREENINFO;
	GsFlush();
	cc = read(fd, (char *) sip, sizeof(GR_SCREEN_INFO));
	if (cc == sizeof(GR_SCREEN_INFO))
		return;

	/*
	 * Read failed, give an error and some dummy results.
	 */
	error = GR_TRUE;
	sip->rows = 1;
	sip->cols = 1;
	sip->xdpcm = 1;
	sip->ydpcm = 1;
	sip->maxcolor = 1;
	sip->black = 0;
	sip->white = 1;
	sip->buttons = 0;
	sip->modifiers = 0;
	sip->fonts = 1;
}


/*
 * Read information about a specified font.
 */
void
GdGetFontInfo(font, fip)
	GR_FONT		font;		/* font number to get info about */
	GR_FONT_INFO	*fip;		/* location to read info into */
{
	GR_MSG_GETFONTINFO	*mp;	/* message to send */
	int			cc;	/* number of bytes read */

	mp = (GR_MSG_GETFONTINFO *)
		GdNewMessage(sizeof(GR_MSG_GETFONTINFO));
	mp->head.func = GR_FUNC_GETFONTINFO;
	mp->font = font;

	GsFlush();

	cc = read(fd, (char *) fip, sizeof(GR_FONT_INFO));
	if (cc == sizeof(GR_FONT_INFO))
		return;
	fip->height = 1;
	fip->maxwidth = 1;
	fip->baseline = 0;
	fip->fixed = GR_TRUE;
}


/*
 * Set an array of clip rectangles for future drawing actions.
 * Each pixel will be drawn only if lies in one or more of the specified
 * clip rectangles.  As a special case, specifying no rectangles implies
 * clipping is for the complete screen.
 */
void
GdSetClipRects(count, cliprects)
	GR_COUNT	count;		/* number of rectangles */
	GR_RECT		*cliprects;	/* table of rectangles */
{
	GR_MSG_SETCLIPRECTS	*mp;
	long			size;

	/*
	 * Make sure there is room in the buffer for the new message.
	 */
	size = GR_SIZEOF_MSG_SETCLIPRECTS(count);
	if ((count < 0) || (count > GR_MAX_COUNT) || (count > GR_MAX_CLIPRECTS)
		|| (size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_SETCLIPRECTS *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_SETCLIPRECTS;
	mp->head.count = count;
	if (count > 0) {
		memcpy((char *) mp->cliprects, (char *) cliprects,
			(int) (count * sizeof(GR_RECT)));
	}
}


/*
 * Set the cursor size and bitmaps.
 */
void
GdSetCursor(width, height, foreground, background, fgbitmap, bgbitmap)
	GR_SIZE		width;		/* width of cursor */
	GR_SIZE		height;		/* height of cursor */
	GR_COLOR	foreground;	/* foreground color of cursor */
	GR_COLOR	background;	/* background color of cursor */
	GR_BITMAP	*fgbitmap;	/* foreground bitmap */
	GR_BITMAP	*bgbitmap;	/* background bitmap */
{
	GR_MSG_SETCURSOR	*mp;
	long			count;
	long			size;
	int			bytes;

	/*
	 * Make sure there is room in the buffer for the new message,
	 * and that the size of the cursor is not too large.
	 */
	count = BITMAP_WORDS((long) width) * ((long) height) * 2;
	size = GR_SIZEOF_MSG_SETCURSOR(count);
	if ((width <= 0) || (height <= 0) || (width > GR_MAX_CURSOR_SIZE) ||
		(height > GR_MAX_CURSOR_SIZE) || (count > GR_MAX_COUNT) ||
		(size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_SETCURSOR *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_SETCURSOR;
	mp->head.count = count;
	mp->width = width;
	mp->height = height;
	mp->foreground = foreground;
	mp->background = background;
	bytes = count * sizeof(GR_BITMAP) / 2;
	if (bytes > 0) {
		memcpy((char *) mp->bitmaps, (char *) fgbitmap, bytes);
		memcpy((char *) &mp->bitmaps[count / 2], (char *) bgbitmap,
			bytes);
	}
}


/*
 * Set the cursor position.
 */
void
GdMoveCursor(x, y)
	GR_COORD	x;		/* new x position of cursor */
	GR_COORD	y;		/* new y position of cursor */
{
	GR_MSG_MOVECURSOR	*mp;

	mp = (GR_MSG_MOVECURSOR *) GdNewMessage(sizeof(GR_MSG_MOVECURSOR));
	mp->head.func = GR_FUNC_MOVECURSOR;
	mp->x = x;
	mp->y = y;
}


/*
 * Set whether or not the background is used for drawing pixmaps and text.
 */
void
GdSetUseBackground(flag)
	GR_BOOL		flag;		/* TRUE if background is drawn */
{
	GR_MSG_SETUSEBACKGROUND	*mp;

	/*
	 * See if the previous message was a set usebackground function,
	 * and if so, overwrite that message.
	 */
	mp = (GR_MSG_SETUSEBACKGROUND *) lastmsg;
	if (mp->head.func == GR_FUNC_SETUSEBACKGROUND) {
		mp->flag = flag;
		return;
	}

	mp = (GR_MSG_SETUSEBACKGROUND *)
		GdNewMessage(sizeof(GR_MSG_SETUSEBACKGROUND));
	mp->head.func = GR_FUNC_SETUSEBACKGROUND;
	mp->flag = flag;
}


/*
 * Set the drawing mode for future calls.
 */
void
GdSetMode(mode)
	GR_MODE		mode;		/* drawing mode */
{
	GR_MSG_SETMODE	*mp;

	/*
	 * See if the previous message was a set mode function, and
	 * if so, overwrite that message.
	 */
	mp = (GR_MSG_SETMODE *) lastmsg;
	if (mp->head.func == GR_FUNC_SETMODE) {
		mp->mode = mode;
		return;
	}

	mp = (GR_MSG_SETMODE *) GdNewMessage(sizeof(GR_MSG_SETMODE));
	mp->head.func = GR_FUNC_SETMODE;
	mp->mode = mode;
}


/*
 * Set the foreground color for future calls.
 */
void
GdSetForeground(fg)
	GR_COLOR	fg;		/* foreground color */
{
	GR_MSG_SETFOREGROUND	*mp;

	/*
	 * See if the previous message was a set foreground function, and
	 * if so, overwrite that message.
	 */
	mp = (GR_MSG_SETFOREGROUND *) lastmsg;
	if (mp->head.func == GR_FUNC_SETFOREGROUND) {
		mp->foreground = fg;
		return;
	}

	mp = (GR_MSG_SETFOREGROUND *)
		GdNewMessage(sizeof(GR_MSG_SETFOREGROUND));
	mp->head.func = GR_FUNC_SETFOREGROUND;
	mp->foreground = fg;
}


/*
 * Set the background color for future calls.
 */
void
GdSetBackground(bg)
	GR_COLOR	bg;		/* background color */
{
	GR_MSG_SETBACKGROUND	*mp;

	/*
	 * See if the previous message was a set background function, and
	 * if so, overwrite that message.
	 */
	mp = (GR_MSG_SETBACKGROUND *) lastmsg;
	if (mp->head.func == GR_FUNC_SETBACKGROUND) {
		mp->background = bg;
		return;
	}

	mp = (GR_MSG_SETBACKGROUND *)
		GdNewMessage(sizeof(GR_MSG_SETBACKGROUND));
	mp->head.func = GR_FUNC_SETBACKGROUND;
	mp->background = bg;
}


/*
 * Draw a line using the current clipping region and foreground color.
 */
void
GdLine(x1, y1, x2, y2)
	GR_COORD	x1;
	GR_COORD	y1;
	GR_COORD	x2;
	GR_COORD	y2;
{
	GR_MSG_LINES	*mp;
	GR_LINE		*lp;

	mp = (GR_MSG_LINES *) GdGrowMessage(GR_FUNC_DRAWLINES, 0,
		sizeof(GR_MSG_LINES), sizeof(GR_LINE));

	lp = &mp->lines[mp->head.count - 1];
	lp->x1 = x1;
	lp->y1 = y1;
	lp->x2 = x2;
	lp->y2 = y2;
}


/*
 * Draw a rectangle using the current clipping region and foreground color.
 * This just draws in the outline of the rectangle.
 */
void
GdRect(x, y, width, height)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GR_MSG_RECTS	*mp;
	GR_RECT		*rp;

	mp = (GR_MSG_RECTS *) GdGrowMessage(GR_FUNC_DRAWRECTS, 0,
		sizeof(GR_MSG_RECTS), sizeof(GR_RECT));

	rp = &mp->rects[mp->head.count - 1];
	rp->x = x;
	rp->y = y;
	rp->width = width;
	rp->height = height;
}


/*
 * Fill a rectangle using the current clipping region and foreground color.
 * This draws the boarder plus all of its interior.
 */
void
GdFillRect(x, y, width, height)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
{
	GR_MSG_RECTS	*mp;
	GR_RECT		*rp;

	mp = (GR_MSG_RECTS *) GdGrowMessage(GR_FUNC_DRAWRECTS,
		GR_FLAG_FILLAREA, sizeof(GR_MSG_RECTS), sizeof(GR_RECT));

	rp = &mp->rects[mp->head.count - 1];
	rp->x = x;
	rp->y = y;
	rp->width = width;
	rp->height = height;
}


/*
 * Draw an ellipse using the current clipping region and foreground color.
 * This just draws in the outline of the ellipse.
 */
void
GdEllipse(x, y, rx, ry)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GR_MSG_ELLIPS	*mp;
	GR_ELLIPSE	*ep;

	mp = (GR_MSG_ELLIPS *) GdGrowMessage(GR_FUNC_DRAWELLIPS, 0,
		sizeof(GR_MSG_ELLIPS), sizeof(GR_ELLIPSE));

	ep = &mp->ellips[mp->head.count - 1];
	ep->x = x;
	ep->y = y;
	ep->rx = rx;
	ep->ry = ry;
}


/*
 * Fill an ellipse using the current clipping region and foreground color.
 */
void
GdFillEllipse(x, y, rx, ry)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		rx;
	GR_SIZE		ry;
{
	GR_MSG_ELLIPS	*mp;
	GR_ELLIPSE	*ep;

	mp = (GR_MSG_ELLIPS *) GdGrowMessage(GR_FUNC_DRAWELLIPS,
		GR_FLAG_FILLAREA, sizeof(GR_MSG_ELLIPS), sizeof(GR_ELLIPSE));

	ep = &mp->ellips[mp->head.count - 1];
	ep->x = x;
	ep->y = y;
	ep->rx = rx;
	ep->ry = ry;
}


/*
 * Copy a rectangular area from one location on the screen to another.
 * This bypasses clipping.
 */
void
GdCopyArea(srcx, srcy, width, height, destx, desty)
	GR_COORD	srcx;
	GR_COORD	srcy;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COORD	destx;
	GR_COORD	desty;
{
	GR_MSG_COPYAREA	*mp;

	mp = (GR_MSG_COPYAREA *) GdNewMessage(sizeof(GR_MSG_COPYAREA));
	mp->head.func = GR_FUNC_COPYAREA;
	mp->sx = srcx;
	mp->sy = srcy;
	mp->width = width;
	mp->height = height;
	mp->dx = destx;
	mp->dy = desty;
}


/*
 * Draw a rectangular area using the current clipping region and the
 * specified bit map.  This differs from rectangle drawing in that the
 * rectangle is drawn using the foreground color and possibly the background
 * color as determined by the bit map.  Each row of bits is aligned to the
 * next bitmap word boundary (so there is padding at the end of the row).
 * The background bit values are only written if the usebackground flag
 * is set in the GC.
 */
void
GdBitmap(x, y, width, height, bitmaptable)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_BITMAP	*bitmaptable;
{
	GR_MSG_BITMAP	*mp;
	long		count;
	long		size;

	/*
	 * Make sure there is room in the buffer for the new message,
	 * and that the size of the rectangle is not too large.
	 */
	count = BITMAP_WORDS((long) width) * ((long) height);
	size = GR_SIZEOF_MSG_BITMAP(count);
	if ((width <= 0) || (height <= 0) || (width > GR_MAX_COUNT) ||
		(height > GR_MAX_COUNT) || (count > GR_MAX_COUNT) ||
		(size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_BITMAP *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_DRAWBITMAP;
	mp->head.count = count;
	mp->x = x;
	mp->y = y;
	mp->width = width;
	mp->height = height;
	if (count > 0) {
		memcpy((char *) mp->bitmaps, (char *) bitmaptable,
			(int) (count * sizeof(GR_BITMAP)));
	}
}


/*
 * Draw a rectangular area using the current clipping region.
 * This differs from rectangle drawing in that the color values for
 * each pixel in the rectangle is specified.  The color values are
 * restricted to 8 bit values.  The color table is indexed row by row.
 */
void
GdArea8(x, y, width, height, colortable)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GR_MSG_AREA8	*mp;
	GR_COUNT	rows;		/* current number of rows sending */
	GR_COUNT	maxrows;	/* maximum rows that can be sent */
	GR_COUNT	count;		/* current count of cells sent */
	GR_SIZE		rowsize;	/* size needed for each row */
	GR_SIZE		fullsize;	/* total size required for message */
	long		minsize;	/* minimum size of message */

	/*
	 * Make sure that at least one row at a time will fit in the buffer.
	 */
	minsize = GR_SIZEOF_MSG_AREA8(width);
	if ((width <= 0) || (height <= 0) || (width > GR_MAX_COUNT) ||
		(height > GR_MAX_COUNT) || (minsize > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	/*
	 * Now iterate as necessary fitting as many rows as will fit at
	 * one time into the buffer.  The number of rows is limited both
	 * by the maximum message size, and by the maximum cell count.
	 */
	rowsize = width * sizeof(GR_COLOR8);
	maxrows = GR_MAX_COUNT / width;

	while (height > 0) {
		mp = (GR_MSG_AREA8 *) GdNewMessage((GR_SIZE) minsize);
		rows = (msgavail / rowsize) + 1;
		if (rows > height)
			rows = height;
		if (rows > maxrows)
			rows = maxrows;
		count = rows * width;
		fullsize = GR_SIZEOF_MSG_AREA8(count);
		msgavail -= (fullsize - minsize);
		newmsg += (fullsize - minsize);
		mp->head.func = GR_FUNC_DRAWAREA8;
		mp->head.count = count;
		mp->x = x;
		mp->y = y;
		mp->width = width;
		mp->height = rows;
		memcpy((char *) mp->colors, (char *) colortable,
			rows * rowsize);
		colortable += count;
		y += rows;
		height -= rows;
	}
}


/*
 * Read a rectangular area of the screen.  The color values are restricted
 * to 8 bit values.  The color table is indexed row by row.
 */
void
GdReadArea8(x, y, width, height, colortable)
	GR_COORD	x;
	GR_COORD	y;
	GR_SIZE		width;
	GR_SIZE		height;
	GR_COLOR8	*colortable;
{
	GR_MSG_READAREA8	*mp;
	GR_COUNT		rows;	/* current number of rows reading */
	GR_COUNT		maxrows; /* maximum rows to read at once */
	GR_SIZE			size;	/* size needed for read rows */
	int			cc;

	/*
	 * Make sure that at least one row at a time will fit in the buffer.
	 */
	if ((width <= 0) || (height <= 0) || (width > GR_MAX_COUNT) ||
		(height > GR_MAX_COUNT) ||
		((width * sizeof(GR_COLOR8)) > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	/*
	 * Now iterate as necessary fitting as many rows as will fit at
	 * one time into the buffer.  The number of rows is limited by the
	 * maximum message size.
	 */
	maxrows = (GR_MAX_SIZEOF_MSG / (width * sizeof(GR_COLOR8)));

	while (height > 0) {
		mp = (GR_MSG_READAREA8 *) GdNewMessage(sizeof(GR_MSG_READAREA8));
		rows = height;
		if (rows > maxrows)
			rows = maxrows;
		size = rows * width * sizeof(GR_COLOR8);
		mp->head.func = GR_FUNC_READAREA8;
		mp->x = x;
		mp->y = y;
		mp->width = width;
		mp->height = rows;

		GsFlush();

		cc = read(fd, (char *) colortable, (int) size);
		if (cc != size) {
			error = GR_TRUE;
			return;
		}

		colortable += rows * width;
		y += rows;
		height -= rows;
	}
}


/*
 * Draw a point using the current clipping region and foreground color.
 */
void
GdPoint(x, y)
	GR_COORD	x;
	GR_COORD	y;
{
	GR_MSG_POINTS	*mp;
	GR_POINT	*pp;

	mp = (GR_MSG_POINTS *) GdGrowMessage(GR_FUNC_DRAWPOINTS, 0,
		sizeof(GR_MSG_POINTS), sizeof(GR_POINT));

	pp = &mp->points[mp->head.count - 1];
	pp->x = x;
	pp->y = y;
}


/*
 * Draw a polygon from the specified list of points.  The polygon is
 * only complete if the first point is repeated at the end.
 */
void
GdPoly(count, pointtable)
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GR_MSG_POLY	*mp;
	long		size;

	/*
	 * Make sure there is room in the buffer for the new message.
	 */
	size = GR_SIZEOF_MSG_POLY(count);
	if ((count < 0) || (count > GR_MAX_COUNT) ||
		(size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_POLY *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_DRAWPOLY;
	mp->head.count = count;
	if (count > 0) {
		memcpy((char *) mp->points, (char *) pointtable,
			(int) (count * sizeof(GR_POINT)));
	}
}


/*
 * Draw a filled polygon from the specified list of points.  The last
 * point may be a duplicate of the first point, but this is not required.
 */
void
GdFillPoly(count, pointtable)
	GR_COUNT	count;
	GR_POINT	*pointtable;
{
	GR_MSG_POLY	*mp;
	long		size;

	/*
	 * Make sure there is room in the buffer for the new message.
	 */
	size = GR_SIZEOF_MSG_POLY(count);
	if ((count < 0) || (count > GR_MAX_COUNT) ||
		(size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_POLY *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_DRAWPOLY;
	mp->head.flags = GR_FLAG_FILLAREA;
	mp->head.count = count;
	if (count > 0) {
		memcpy((char *) mp->points, (char *) pointtable,
			(int) (count * sizeof(GR_POINT)));
	}
}


/*
 * Draw a text string starting at the specified coordinates.
 */
void
GdText(x, y, str, count)
	GR_COORD	x;
	GR_COORD	y;
	GR_CHAR		*str;
	GR_COUNT	count;
{
	GR_MSG_TEXT	*mp;
	long		size;

	/*
	 * Make sure there is room in the buffer for the new message.
	 */
	size = GR_SIZEOF_MSG_TEXT(count);
	if ((count < 0) || (count > GR_MAX_COUNT) ||
		(size > GR_MAX_SIZEOF_MSG))
	{
		error = GR_TRUE;
		return;
	}

	mp = (GR_MSG_TEXT *) GdNewMessage((GR_SIZE) size);
	mp->head.func = GR_FUNC_DRAWTEXT;
	mp->head.count = count;
	mp->x = x;
	mp->y = y;
	memcpy((char *) mp->chars, (char *) str, (int) count * sizeof(GR_CHAR));
}

/* END CODE */
